﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include "capsrv_ScreenShotSize.h"
#include "capsrv_ScreenShotOrientation.h"
#include "capsrv_AlbumFileDescription.h"

namespace nn{ namespace capsrv{

    namespace detail{

        enum ScreenShotAttributeFlag : uint32_t
        {
            ScreenShotAttributeFlag_IsCopyrightImageComposited = (1 << 0),
            ScreenShotAttributeFlag_HasUneditableArea          = (1 << 1),
        };

        struct ScreenShotAttributeEx0
        {
        public:
            void SetDefault() NN_NOEXCEPT
            {
                this->size = ScreenShotSize_Default;
                this->orientation = ScreenShotOrientation_Default;
                this->description = AlbumFileDescription_ScreenShot;
                this->frameCount = 1;
                this->frameRateNumerator = 0;
                this->frameRateDenominator = 0;
                this->dataDurationMilliseconds = 0;
                this->keyFrameInterval = 0;
                this->uneditableAreaCoordinateX = 0;
                this->uneditableAreaCoordinateY = 0;
                this->uneditableAreaWidth = 0;
                this->uneditableAreaHeight = 0;
                this->flags = 0;
                this->movieSize = ScreenShotSize_Default;

                std::memset(this->_reserved, 0, sizeof(this->_reserved));
            }

            bool IsCopyrightImageComposited() const NN_NOEXCEPT
            {
                return (this->flags & nn::capsrv::detail::ScreenShotAttributeFlag_IsCopyrightImageComposited) != 0;
            }

            bool HasUneditableArea() const NN_NOEXCEPT
            {
                return (this->flags & nn::capsrv::detail::ScreenShotAttributeFlag_HasUneditableArea) != 0;
            }

        public:
            ScreenShotSizeType size;
            ScreenShotOrientationType orientation;
            AlbumFileDescriptionType description;
            uint32_t frameCount;
            uint32_t frameRateNumerator;
            uint32_t frameRateDenominator;
            uint32_t dataDurationMilliseconds;
            uint32_t keyFrameInterval;
            uint16_t uneditableAreaCoordinateX;
            uint16_t uneditableAreaCoordinateY;
            uint16_t uneditableAreaWidth;
            uint16_t uneditableAreaHeight;
            uint32_t flags;
            ScreenShotSizeType movieSize;
            char _reserved[16];
        };

        NN_STATIC_ASSERT(sizeof(ScreenShotAttributeEx0) == 64);

    }   // namespace detail

    //! @brief 画面写真の属性を表します。
    struct ScreenShotAttribute
        : detail::ScreenShotAttributeEx0
    {
    public:
        //! @brief 既定値を設定します。
        void SetDefault() NN_NOEXCEPT
        {
            ScreenShotAttributeEx0::SetDefault();
        }
    };

    //! @brief 画面写真の属性を表します。
    //! @detail
    //! アプリプロセスの shim に埋め込まれるため、形を変更してはならない
    struct ScreenShotAttributeForApplication
    {
        typedef uint8_t ScreenShotSizeTypeUint8;
        typedef uint8_t ScreenShotOrientationTypeUint8;

    public:
        void SetDefault() NN_NOEXCEPT
        {
            this->description = AlbumFileDescription_ScreenShot;

            this->screenShotSize = ScreenShotSize_Default;
            this->movieSize = ScreenShotSize_Default;
            this->orientation = ScreenShotOrientation_Default;
            this->_padding0 = 0;

            this->flags = 0;
            this->frameCount = 1;
            this->frameRateNumerator = 0;
            this->frameRateDenominator = 0;
            this->dataDurationMilliseconds = 0;
            this->keyFrameInterval = 0;
            this->uneditableAreaCoordinateX = 0;
            this->uneditableAreaCoordinateY = 0;
            this->uneditableAreaWidth = 0;
            this->uneditableAreaHeight = 0;

            std::memset(this->_reserved, 0, sizeof(this->_reserved));
        }

        void CopyFromScreenShotAttribute(const ScreenShotAttribute& attr) NN_NOEXCEPT
        {
            this->description = attr.description;

            this->screenShotSize = static_cast<ScreenShotSizeTypeUint8>(attr.size);
            this->movieSize = static_cast<ScreenShotSizeTypeUint8>(attr.movieSize);
            this->orientation = static_cast<ScreenShotOrientationTypeUint8>(attr.orientation);
            this->_padding0 = 0;

            this->flags = attr.flags;
            this->frameCount = attr.frameCount;
            this->frameRateNumerator = attr.frameRateNumerator;
            this->frameRateDenominator = attr.frameRateDenominator;
            this->dataDurationMilliseconds = attr.dataDurationMilliseconds;
            this->keyFrameInterval = attr.keyFrameInterval;
            this->uneditableAreaCoordinateX = attr.uneditableAreaCoordinateX;
            this->uneditableAreaCoordinateY = attr.uneditableAreaCoordinateY;
            this->uneditableAreaWidth = attr.uneditableAreaWidth;
            this->uneditableAreaHeight = attr.uneditableAreaHeight;

            std::memset(this->_reserved, 0, sizeof(this->_reserved));
        }

    public:
        AlbumFileDescriptionType description;

        ScreenShotSizeTypeUint8 screenShotSize;
        ScreenShotSizeTypeUint8 movieSize;
        ScreenShotOrientationTypeUint8 orientation;
        uint8_t _padding0;

        uint32_t flags;
        uint32_t frameCount;
        uint32_t frameRateNumerator;
        uint32_t frameRateDenominator;
        uint32_t dataDurationMilliseconds;
        uint32_t keyFrameInterval;
        uint16_t uneditableAreaCoordinateX;
        uint16_t uneditableAreaCoordinateY;
        uint16_t uneditableAreaWidth;
        uint16_t uneditableAreaHeight;

        uint32_t _reserved[6];
    };

    NN_STATIC_ASSERT(sizeof(ScreenShotAttributeForApplication) == sizeof(uint32_t) * 16);

}}
