﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
#include <autoTestAssistTool/CaptureProcessorTypes.h>
#include <autoTestAssistTool/Common/autoTestAssistTool_LibTypes.h>

#ifdef CAPTUREPROCESSOR_EXPORTS
#define CAPTUREPROCESSOR_API extern "C" __declspec(dllexport)
#else
#define CAPTUREPROCESSOR_API extern "C" __declspec(dllimport)
#endif

/**
*   @file
*   @brief  キャプチャデバイスアクセス、画像処理に関する API
*/

/*---------------------------------------------------------------------
/ 初期化・終了処理
/--------------------------------------------------------------------*/
/**
* @brief        キャプチャデバイス向けに CaptureProcessor ライブラリの初期化処理を行います。

* @param[in]     deviceId               キャプチャデバイスID
* @param[in]     isPreview              プレビュー表示フラグ
* @param[in]     captureResolution      キャプチャ解像度

* @details      PC に接続されているキャプチャデバイスをキャプチャ対象のデバイスとして設定します。@n
*               キャプチャデバイスID は、利用環境で接続されているカメラ、キャプチャボードなどのキャプチャデバイスに対して 0 から順番に値が割り当てられています。
*               deviceId には、0 以上、接続中のビデオデバイス数未満の整数値を指定してください。@n
*               接続されているキャプチャデバイス数は、GetCaptureDeviceCount 関数より取得することができます。@n
*               isPreview に true を指定するとライブラリ内でウィンドウを生成し、キャプチャ中の画像をリアルタイムでプレビュー表示します。 @n
*               初期化が完了していない状態では、画像操作の API 呼び出しが正常に実行されません。
*/
CAPTUREPROCESSOR_API void InitializeForCaptureDevice(int deviceId, bool isPreview, CaptureResolution captureResolution);

/**
* @brief        開発機向けに CaptureProcessor ライブラリの初期化処理を行います。

* @param[in]     serialNumber           対象開発機のシリアルナンバー
* @param[in]     isPreview              プレビュー表示フラグ

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success            操作に成功したときの値です。
* @retval       CaptureResult_Unexpected         不測のエラーが発生したときの値です。
* @retval       CaptureResult_NotFound           対象開発機が見つからなかったときの値です。
* @retval       CaptureResult_TargetManagerError Target Manager からの情報取得に失敗したときの値です。

* @details      TargetManager で接続している開発機をキャプチャ対象デバイスとして設定します。@n
*               キャプチャするデバイスには SDEV のシリアルナンバーを指定してください。（EDEV は未対応です）@n
*               isPreview に true を指定するとライブラリ内でウィンドウを生成し、キャプチャ中の画像をリアルタイムでプレビュー表示します。 @n
*               初期化が完了していない状態では、画像操作の API 呼び出しが正常に実行されません。
*/
CAPTUREPROCESSOR_API CaptureResult InitializeForDevkitCapture(const SerialNumberString& serialNumber, bool isPreview);

/**
* @brief    CaptureProcessor ライブラリの終了処理を行います。

* @pre      Initialize 関数の呼び出しが完了している。
* @details  CaptureProcessor ライブラリの終了処理を行います。 @n
            この関数の実行後は、Initialize 関数が実行されるまで画像操作の API 呼び出しが正常に実行されなくなります。
*/
CAPTUREPROCESSOR_API void Finalize();

/*---------------------------------------------------------------------
/ キャプチャデバイス情報取得
/--------------------------------------------------------------------*/
/**
* @brief        接続されているキャプチャデバイス数を取得します。

* @param[out]   pOutDeviceCount                       接続されているキャプチャデバイス数の出力先

* @return       操作の成否を返します。
* @retval       CaptureResult_Success            操作に成功したときの値です。
* @retval       CaptureResult_Unexpected         不測のエラーが発生したときの値です。

* @details      利用環境で接続されているキャプチャデバイス数を取得することができます。 @n
*               0 からキャプチャデバイス数未満の値が、デバイスID として CaptureProcessor ライブラリへ指定できる値です。
*/
CAPTUREPROCESSOR_API CaptureResult GetCaptureDeviceCount(int* pOutDeviceCount);


/**
* @brief        キャプチャデバイス名を取得します。

* @param[out]   pOutNameBuffer                      キャプチャデバイス名の出力先
* @param[in]    nameBufferSize                      出力先のバッファサイズ
* @param[in]    deviceId                            キャプチャデバイスID

* @return       操作の成否を返します。
* @retval       CaptureResult_Success            操作に成功したときの値です。
* @retval       CaptureResult_Unexpected         不測のエラーが発生したときの値です。

* @details      指定されたデバイス ID に対応したキャプチャデバイスの名称を取得することができます。
*               キャプチャデバイス名は UTF-8 エンコードの文字列として出力されます。
*/
CAPTUREPROCESSOR_API CaptureResult GetCaptureDeviceName(char* pOutNameBuffer, int nameBufferSize, int deviceId);

/*---------------------------------------------------------------------
/ 画像取得・解放
/--------------------------------------------------------------------*/
/**
* @brief       	キャプチャ対象のデバイスから画像データを取得します。

* @param[out]   pOutImageId                         取得した画像データの ID 出力先

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success            操作に成功したときの値です。
* @retval       CaptureResult_Unexpected         不測のエラーが発生したときの値です。
* @retval       CaptureResult_NotInitialized     未初期化状態で呼び出されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      キャプチャ対象のデバイスから取得した最新の画像データをライブラリ内に記録し、対象画像データに対応する ID を取得することができます。
*/
CAPTUREPROCESSOR_API CaptureResult GetImageFromCaptureDevice( ImageIdType *pOutImageId );

/**
* @brief       	画像ファイルを読み込みます。

* @param[out]   pOutImageId                         読み込んだ画像データの ID 出力先
* @param[in]    filepath                            読み込み対象画像へのパス

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success            操作に成功したときの値です。
* @retval       CaptureResult_Unexpected         不測のエラーが発生したときの値です。
* @retval       CaptureResult_NotInitialized     未初期化状態で呼び出されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      指定された画像ファイルのデータをライブラリ内に記録し、対象画像データに対応する ID を取得することができます。@n
*               指定するファイルパスは UTF-8 エンコードの文字列として指定してください。
*/
CAPTUREPROCESSOR_API CaptureResult LoadImageFile(ImageIdType *pOutImageId, const char *filepath);

/**
* @brief       	画像データをファイル出力します。

* @param[in]    filepath                            出力先のパス
* @param[in]    id                                  出力画像データの ID

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success            操作に成功したときの値です。
* @retval       CaptureResult_Unexpected         不測のエラーが発生したときの値です。
* @retval       CaptureResult_InvalidImageId     不正な画像データ ID が指定された場合の値です。
* @retval       CaptureResult_NotInitialized     未初期化状態で呼び出されたときの値です。
* @retval       CaptureResult_FileCreationFailed ファイル作成に失敗した場合の値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      指定された ID に対応する画像データをファイル出力します。@n
*               指定するファイルパスは UTF-8 エンコードの文字列として指定してください。
*/
CAPTUREPROCESSOR_API CaptureResult SaveImageFile(const char* filepath, ImageIdType id);

/**
* @brief       	記録している画像データを解放します。

* @param[in]    id                                  解放する画像データの ID

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success            操作に成功したときの値です。
* @retval       CaptureResult_Unexpected         不測のエラーが発生したときの値です。
* @retval       CaptureResult_InvalidImageId     不正な画像データ ID が指定された場合の値です。
* @retval       CaptureResult_NotInitialized     未初期化状態で呼び出されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      指定された ID に対応する画像データを解放します。
*/

CAPTUREPROCESSOR_API CaptureResult ReleaseImage(ImageIdType id);

/**
* @brief       	記録しているすべての画像データを解放します。

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success            操作に成功したときの値です。
* @retval       CaptureResult_Unexpected         不測のエラーが発生したときの値です。
* @retval       CaptureResult_NotInitialized     未初期化状態で呼び出されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      記録されているすべての画像データを解放することができます。 @n
                Finalize 関数の実行時にも自動で呼び出されます。
*/
CAPTUREPROCESSOR_API CaptureResult ReleaseAllImages();

/*----------------------------------------------------------------------
/ 画像データ取得
/---------------------------------------------------------------------*/
/**
* @brief       	指定した ID に対応する画像データのサイズ情報を取得します。

* @param[out]   pOutWidth                           幅の出力先
* @param[out]   pOutHeight                          高さの出力先
* @param[in]    id                                  対象画像データの ID


* @return       操作の成否が返ります。
* @retval       CaptureResult_Success            操作に成功したときの値です。
* @retval       CaptureResult_Unexpected         不測のエラーが発生したときの値です。
* @retval       CaptureResult_InvalidImageId     不正な画像データ ID が指定された場合の値です。
* @retval       CaptureResult_NotInitialized     未初期化状態で呼び出されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      記録している画像データのサイズ情報を取得することができます。 @n
*/
CAPTUREPROCESSOR_API CaptureResult GetImageSize(int* pOutWidth, int* pOutHeight, ImageIdType id);


/**
* @brief       	指定した ID に対応する画像データの画素配列情報を取得します。

* @param[out]   pOutColorBuffer                     画素配列情報の出力先バッファ
* @param[in]    colorBufferSize                     出力先のバッファサイズ
* @param[in]    id                                  対象画像データ ID


* @return       操作の成否が返ります。
* @retval       CaptureResult_Success            操作に成功したときの値です。
* @retval       CaptureResult_Unexpected         不測のエラーが発生したときの値です。
* @retval       CaptureResult_InvalidImageId     不正な画像データ ID が指定された場合の値です。
* @retval       CaptureResult_NotInitialized     未初期化状態で呼び出されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      記録している画像データの画素配列情報を取得します。 @n
*               画素配列情報は、対象画像データの画素情報を、色（RGB)情報（1 byte） + (x, y)座標で表したときに以下のような並びで pOutColorBuffer へコピーされます。 @n
*               [ B(0,0), G(0,0), R(0,0), B(1,0), G(1,0), R(1,0)　・・・ B(x,y), G(x,y), R(x,y) ] @n
*               必要なバッファサイズは GetImageSize で得られる（幅 * 高さ * ColorChannelCount）で求められます。
*/
CAPTUREPROCESSOR_API CaptureResult GetImageData(void* pOutColorBuffer, size_t colorBufferSize, ImageIdType id);


/*----------------------------------------------------------------------
/ 標準テンプレートマッチング
/---------------------------------------------------------------------*/
/**
* @brief       	キャプチャ対象デバイスから取得された最新の画像に対して、テンプレートマッチングを行います。

* @param[out]   pOutX                               検出結果 X 座標
* @param[out]   pOutY                               検出結果 Y 座標
* @param[in]    id                                  テンプレート画像データの ID
* @param[in]    threshold                           マッチングの閾値（ 1.0 > x > 0.0 )

* @return       操作の成否が返ります。
* @retval       MatchingResult_Success         一致する画像が見つかった場合の値です。
* @retval       MatchingResult_Failed          一致する画像が見つからなかった場合の値です。
* @retval       MatchingResult_Unexpected      不測のエラーが発生したときの値です。
* @retval       MatchingResult_InvalidImageId  不正な画像データ ID が指定された場合の値です。
* @retval       MatchingResult_NotInitialized  未初期化状態で呼び出されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      キャプチャ対象デバイスから最新の画像を取得して、その画像内で指定した ID に記録されているテンプレート画像と一致する箇所を検出します。 @n
*               MatchingResult_Success が返った時、取得された画像内で、テンプレート画像との一致度合いが最も高かった領域の左上座標が pOutX, pOutY へ渡されます。 @n
*               threshold は、高いほど、よりテンプレート画像との一致度合いが高くなければ検出にヒットしなくなります。
*/
CAPTUREPROCESSOR_API MatchingResult DetectObject(int *pOutX, int *pOutY, ImageIdType id, double threshold );

/**
* @brief       	キャプチャ中の画像に対して、テンプレート画像が検出されるまでテンプレートマッチングを繰り返し行います。

* @param[out]   pOutX                               検出結果 X 座標
* @param[out]   pOutY                               検出結果 Y 座標
* @param[in]    id                                  テンプレート画像データの ID
* @param[in]    threshold                           マッチングの閾値（ 1.0 > x > 0.0 )
* @param[in]    timeout                             タイムアウト時間 [ms]

* @return       操作の成否が返ります。
* @retval       MatchingResult_Success         一致する画像が見つかった場合の値です。
* @retval       MatchingResult_Failed          一致する画像が見つからなかった場合の値です。
* @retval       MatchingResult_Unexpected      不測のエラーが発生したときの値です。
* @retval       MatchingResult_InvalidImageId  不正な画像データ ID が指定された場合の値です。
* @retval       MatchingResult_NotInitialized  未初期化状態で呼び出されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      キャプチャ対象デバイスから取得された最新の画像に対してのテンプレートマッチングをテンプレート画像が検出されるまで繰り返し行います。
*               timeout[ms] で検出されなかった場合はタイムアウトします。　@n
*               MatchingResult_Success が返った時、現在マッチングを行っている画像内で、テンプレート画像との一致度合いが最も高かった領域の左上座標が pOutX, pOutY へ渡されます。 @n
*               threshold は、高いほど、よりテンプレート画像との一致度合いが高くなければ検出にヒットしなくなります。
*/
CAPTUREPROCESSOR_API MatchingResult WaitDetectObject(int *pOutX, int *pOutY, ImageIdType id, double threshold, int timeout);

/**
* @brief       	画像処理フィルターをかけた状態で、キャプチャ対象デバイスから取得された最新の画像に対して、テンプレートマッチングを行います。

* @param[out]   pOutX                               検出結果 X 座標
* @param[out]   pOutY                               検出結果 Y 座標
* @param[in]    id                                  テンプレート画像データの ID
* @param[in]    threshold                           マッチングの閾値（ 1.0 > x > 0.0 )
* @param[in]    filterId                            画像処理フィルター ID

* @return       操作の成否が返ります。
* @retval       MatchingResult_Success         一致する画像が見つかった場合の値です。
* @retval       MatchingResult_Failed          一致する画像が見つからなかった場合の値です。
* @retval       MatchingResult_Unexpected      不測のエラーが発生したときの値です。
* @retval       MatchingResult_InvalidImageId  不正な画像データ ID が指定された場合の値です。
* @retval       MatchingResult_NotInitialized  未初期化状態で呼び出されたときの値です。
* @retval       MatchingResult_InvalidFilterId 不正なフィルター ID が指定された場合の値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      キャプチャ対象デバイスから最新の画像を取得して、その画像内で指定した ID に記録されているテンプレート画像と一致する箇所を検出します。 @n
*               MatchingResult_Success が返った時、取得された画像内で、テンプレート画像との一致度合いが最も高かった領域の左上座標が pOutX, pOutY へ渡されます。 @n
*               threshold は、高いほど、よりテンプレート画像との一致度合いが高くなければ検出にヒットしなくなります。@n
*               テンプレートマッチングは、テンプレート画像とキャプチャ画像に対して、filterId の画像処理フィルターを実行した状態で実行されます。
*/
CAPTUREPROCESSOR_API MatchingResult DetectObjectWithFilter(int* pOutX, int* pOutY, ImageIdType id, double threshold, FilterIdType filterId);

/**
* @brief       	キャプチャ中の画像に対して、テンプレート画像が検出されるまでテンプレートマッチングを繰り返し行います。

* @param[out]   pOutX                               検出結果 X 座標
* @param[out]   pOutY                               検出結果 Y 座標
* @param[in]    id                                  テンプレート画像データの ID
* @param[in]    threshold                           マッチングの閾値（ 1.0 > x > 0.0 )
* @param[in]    timeout                             タイムアウト時間 [ms]
* @param[in]    filterId                            画像処理フィルター ID

* @return       操作の成否が返ります。
* @retval       MatchingResult_Success         一致する画像が見つかった場合の値です。
* @retval       MatchingResult_Failed          一致する画像が見つからなかった場合の値です。
* @retval       MatchingResult_Unexpected      不測のエラーが発生したときの値です。
* @retval       MatchingResult_InvalidImageId  不正な画像データ ID が指定された場合の値です。
* @retval       MatchingResult_NotInitialized  未初期化状態で呼び出されたときの値です。
* @retval       MatchingResult_InvalidFilterId 不正なフィルター ID が指定された場合の値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      キャプチャ対象デバイスから取得された最新の画像に対してのテンプレートマッチングをテンプレート画像が検出されるまで繰り返し行います。
*               timeout[ms] で検出されなかった場合はタイムアウトします。　@n
*               MatchingResult_Success が返った時、現在マッチングを行っている画像内で、テンプレート画像との一致度合いが最も高かった領域の左上座標が pOutX, pOutY へ渡されます。 @n
*               threshold は、高いほど、よりテンプレート画像との一致度合いが高くなければ検出にヒットしなくなります。@n
*               テンプレートマッチングは、テンプレート画像とキャプチャ画像に対して、filterId の画像処理フィルターを実行した状態で実行されます。
*/
CAPTUREPROCESSOR_API MatchingResult WaitDetectObjectWithFilter(int* pOutX, int* pOutY, ImageIdType id, double threshold, int timeout, FilterIdType filterId);

/*----------------------------------------------------------------------
/ フィルタ
/---------------------------------------------------------------------*/
/**
* @brief       	新規の画像処理フィルターを作成します。

* @param[out]   pOutFilterId                  画像処理フィルター ID の出力先

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success         操作に成功したときの値です。
* @retval       CaptureResult_Unexpected      不測のエラーが発生したときの値です。
* @retval       CaptureResult_NotInitialized  未初期化状態で呼び出されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      新規の画像処理フィルターを作成します。@n
                作成された画像処理フィルターは、SetBinarizationFilter, SetColorRangeFilter 関数などを用いてフィルター処理をセットすることができます。@n
                複数のフィルター処理がセットされた場合、画像に対してフィルター処理を行う際に、セットされたフィルターの順番通りに全ての画像処理が実行されます。
*/
CAPTUREPROCESSOR_API CaptureResult CreateFilter( FilterIdType *pOutFilterId );

/**
* @brief       	画像処理フィルターを消去します。

* @param[in]   filterId                        消去する画像処理フィルター ID

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success         操作に成功したときの値です。
* @retval       CaptureResult_Unexpected      不測のエラーが発生したときの値です。
* @retval       CaptureResult_NotInitialized  未初期化状態で呼び出されたときの値です。
* @retval       CaptureResult_InvalidFilterId 不正な画像処理フィルター ID が指定されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      指定された画像処理フィルターを消去します。
*/
CAPTUREPROCESSOR_API CaptureResult DeleteFilter( FilterIdType filterId );

/**
* @brief       	対象画像に画像処理フィルターを適用します。

* @param[out]   pOutId                      画像処理フィルター適用後の画像データの ID 出力先
* @param[in]    id                          画像処理フィルターを適用する対象画像データの ID
* @param[in]    filterId                    適用する画像処理フィルター ID

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success         操作に成功したときの値です。
* @retval       CaptureResult_Unexpected      不測のエラーが発生したときの値です。
* @retval       CaptureResult_InvalidImageId  不正な画像データ ID が指定された場合の値です。
* @retval       CaptureResult_NotInitialized  未初期化状態で呼び出されたときの値です。
* @retval       CaptureResult_InvalidFilterId 不正な画像処理フィルター ID が指定されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      対象画像に画像処理フィルターを適用し、適用後の画像データの ID を取得することができます。@n
*/
CAPTUREPROCESSOR_API CaptureResult ApplyFilter( ImageIdType *pOutId, ImageIdType id, FilterIdType filterId );

/**
* @brief       	画像処理フィルターに二値化処理フィルターを追加します。

* @param[in]    filterId                        処理を追加する画像処理フィルター ID
* @param[in]    binarizeThreshold               二値化処理の閾値（0 <= binarizeThreshold <= 255)

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success           操作に成功したときの値です。
* @retval       CaptureResult_Unexpected        不測のエラーが発生したときの値です。
* @retval       CaptureResult_InvalidImageId    不正な画像データ ID が指定された場合の値です。
* @retval       CaptureResult_NotInitialized    未初期化状態で呼び出されたときの値です。
* @retval       CaptureResult_InvalidFilterId   不正な画像処理フィルター ID が指定されたときの値です。
* @retval       CaptureResult_InvalidParameter  不正なフィルターパラメーターが指定されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      画像処理フィルターに対して、二値化処理フィルターを追加します。@n
                二値化処理では、NTSC 加重平均法によるグレースケール変換（Gray = Red * 0.299 + Green * 0.587 + Blue * 0.114）の後に、
                画素値が binarizeThreshold を超えた部分が白、それ以外を黒といった変換が行われます。

*/
CAPTUREPROCESSOR_API CaptureResult SetBinarizationFilter( FilterIdType filterId, int binarizeThreshold );

/**
* @brief       	画像処理フィルターに色領域フィルターを追加します。

* @param[in]    filterId                        処理を追加する画像処理フィルター ID
* @param[in]    lowerColor                      色域の下限値(各値は 0 以上かつ 255 以下)
* @param[in]    upperColor                      色域の上限値(各値は 0 以上かつ 255 以下)

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success           操作に成功したときの値です。
* @retval       CaptureResult_Unexpected        不測のエラーが発生したときの値です。
* @retval       CaptureResult_InvalidImageId    不正な画像データ ID が指定された場合の値です。
* @retval       CaptureResult_NotInitialized    未初期化状態で呼び出されたときの値です。
* @retval       CaptureResult_InvalidFilterId   不正な画像処理フィルター ID が指定されたときの値です。
* @retval       CaptureResult_InvalidParameter  不正なフィルターパラメーターが指定されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      画像処理フィルターに色領域フィルターを追加します。@n
                色領域フィルター処理では、RGB の各要素の値が全て、lowerColor のそれぞれの要素の値以上、upperColor のそれぞれの要素の値以下の箇所以外を、全て黒く塗りつぶします。
*/
CAPTUREPROCESSOR_API CaptureResult SetColorRangeFilter( FilterIdType filterId, Rgb24Color lowerColor, Rgb24Color upperColor );

/**
* @brief       	画像処理フィルターにモルフォロジー変換フィルターを追加します。

* @param[in]    filterId                        処理を追加する画像処理フィルター ID
* @param[in]    type                            モルフォロジー変換の種類
* @param[in]    iterations                      変換処理の回数

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success           操作に成功したときの値です。
* @retval       CaptureResult_Unexpected        不測のエラーが発生したときの値です。
* @retval       CaptureResult_InvalidImageId    不正な画像データ ID が指定された場合の値です。
* @retval       CaptureResult_NotInitialized    未初期化状態で呼び出されたときの値です。
* @retval       CaptureResult_InvalidFilterId   不正な画像処理フィルター ID が指定されたときの値です。
* @retval       CaptureResult_InvalidParameter  不正なフィルターパラメーターが指定されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      画像処理フィルターに対して、モルフォロジー変換フィルターを追加します。
*/
CAPTUREPROCESSOR_API CaptureResult SetMorphologyFilter( FilterIdType filterId, MorphologyConversionType type, int iterations );

/*----------------------------------------------------------------------
/ 色検出
/---------------------------------------------------------------------*/
/**
* @brief       	画像データ上の指定座標の画素値を取得します。

* @param[out]   pOutColor                       画素値の出力先
* @param[in]    id                              対象画像データ ID
* @param[in]    x                               X 座標
* @param[in]    y                               Y 座標

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success           操作に成功したときの値です。
* @retval       CaptureResult_Unexpected        不測のエラーが発生したときの値です。
* @retval       CaptureResult_InvalidImageId    不正な画像データ ID が指定された場合の値です。
* @retval       CaptureResult_NotInitialized    未初期化状態で呼び出されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      対象画像データ上の指定された座標の画素値を取得します。
*/
CAPTUREPROCESSOR_API CaptureResult GetDotColor( Rgb24Color* pOutColor,  ImageIdType id, int x, int y );

/**
* @brief       	キャプチャ中の画像上の指定座標の画素値が、指定範囲内の値かをチェックします。

* @param[in]    x                               対象 X 座標
* @param[in]    y                               対象 Y 座標
* @param[in]    lowerColor                      画素値範囲の下限値(各値は 0 以上かつ 255 以下)
* @param[in]    upperColor                      画素値範囲の上限値(各値は 0 以上かつ 255 以下)

* @return       操作の成否が返ります。
* @retval       MatchingResult_Success         画素値が指定範囲内だった場合の値です。
* @retval       MatchingResult_Failed          画素値が指定範囲内では無かった場合の値です。
* @retval       MatchingResult_Unexpected      不測のエラーが発生したときの値です。
* @retval       MatchingResult_NotInitialized  未初期化状態で呼び出されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      キャプチャ中の画像上の指定座標の画素値が、指定範囲内の値かをチェックします。@n
                RGB の各要素の値が全て、lowerColor のそれぞれの要素の値以上、upperColor のそれぞれの要素の値以下だった場合に MatchingResult_Success を返します。

*/
CAPTUREPROCESSOR_API MatchingResult DetectDotColor( int x, int y, Rgb24Color lowerColor, Rgb24Color upperColor );

/**
* @brief       	キャプチャ中の画像上の指定座標の画素値が、指定範囲内の値となるまで待機します。

* @param[in]    x                               対象 X 座標
* @param[in]    y                               対象 Y 座標
* @param[in]    lowerColor                      画素値範囲の下限値(各値は 0 以上かつ 255 以下)
* @param[in]    upperColor                      画素値範囲の上限値(各値は 0 以上かつ 255 以下)
* @param[in]    timeout                         タイムアウト時間 [ms]

* @return       操作の成否が返ります。
* @retval       MatchingResult_Success         画素値が指定範囲内だった場合の値です。
* @retval       MatchingResult_Failed          タイムアウト時間内に画素値が指定範囲内の値とならなかった場合の値です。
* @retval       MatchingResult_Unexpected      不測のエラーが発生したときの値です。
* @retval       MatchingResult_NotInitialized  未初期化状態で呼び出されたときの値です。


* @pre          Initialize 関数の呼び出しが完了している。

* @details      キャプチャ中の画像上の指定座標の画素値が、指定範囲内の値となる、もしくはタイムアウトするまで待機します。@n
                RGB の各要素の値が全て、lowerColor のそれぞれの要素の値以上、upperColor のそれぞれの要素の値以下だった場合に MatchingResult_Success を返します。

*/
CAPTUREPROCESSOR_API MatchingResult WaitDetectDotColor( int x, int y,  Rgb24Color lowerColor, Rgb24Color upperColor, int timeout );

/*----------------------------------------------------------------------
/ ラベリング
/---------------------------------------------------------------------*/
/**
* @brief       	対象画像に対してラベリング処理を実行します。

* @param[out]   pOutResultAreaCount             検出ラベル領域数
* @param[out]   pOutLabelingResultHandle        ラベリング結果ハンドルの出力先
* @param[in]    imageId                         対象画像データ ID

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success            操作に成功したときの値です。
* @retval       CaptureResult_Unexpected         不測のエラーが発生したときの値です。
* @retval       CaptureResult_InvalidImageId     不正な画像データ ID が指定された場合の値です。
* @retval       CaptureResult_NotInitialized     未初期化状態で呼び出されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      対象画像に対してラベリング処理を実行します。@n
                ラベリングでは、対象画像を NTSC 加重平均法によるグレースケール変換（Gray = Red * 0.299 + Green * 0.587 + Blue * 0.114）の後に、
                画素値の近い箇所をグループ化し、グループの矩形領域、面積、重心などを算出しています。@n
                処理結果は、GetLabelingResults より取得することができます。@n
                不要となった処理結果は ReleaseLabelingResult 関数を利用して解放してください。@n
                精度を上げるために、ラベリング処理の実行前に、色領域フィルターで目的の部分以外を黒塗りした状態、
                もしくは二値化処理を行った状態の画像に対して実行することを推奨します。
*/
CAPTUREPROCESSOR_API CaptureResult ExecuteLabeling( int *pOutResultAreaCount, LabelingResultHandleType *pOutLabelingResultHandle, ImageIdType imageId );

/**
* @brief       	ラベリング処理の結果を取得します。

* @param[out]   pOutResultsCount                取得できた結果数
* @param[out]   pOutResultArray                 取得する結果の出力先
* @param[in]    labelingResultHandle            ラベリングの結果ハンドル
* @param[in]    arrayCount                      結果の出力先の配列長

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success                     操作に成功したときの値です。
* @retval       CaptureResult_Unexpected                  不測のエラーが発生したときの値です。
* @retval       CaptureResult_NotInitialized              未初期化状態で呼び出されたときの値です。
* @retval       CaptureResult_InvalidLabelingResultHandle 不正なラベリング結果ハンドルが指定されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      ラベリング処理の結果を pOutResultArray へ渡します。
*/
CAPTUREPROCESSOR_API CaptureResult GetLabelingResults( int *pOutResultsCount, LabelingResult pOutResultArray[], LabelingResultHandleType labelingResultHandle, int arrayCount );

/**
* @brief       	ラベリングの処理結果を解放します。

* @param[in]    labelingResultHandle              ラベリングの結果ハンドル

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success                     操作に成功したときの値です。
* @retval       CaptureResult_Unexpected                  不測のエラーが発生したときの値です。
* @retval       CaptureResult_NotInitialized              未初期化状態で呼び出されたときの値です。
* @retval       CaptureResult_InvalidLabelingResultHandle 不正なラベリング結果ハンドルが指定されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      ラベリングの処理結果を解放します。
*/
CAPTUREPROCESSOR_API CaptureResult ReleaseLabelingResult(LabelingResultHandleType labelingResultHandle);

/*----------------------------------------------------------------------
/ OCR
/---------------------------------------------------------------------*/
/**
* @brief       	対象画像に上に、指定された正規表現に一致する文字列が表示されているかをチェックします。

* @param[out]   pOutRect                        対象文字列が表示されている領域
* @param[in]    id                              対象画像データ ID
* @param[in]    regexString                     ヌル終端された正規表現の文字列
* @param[in]    separate                        OCR の認識単位
* @param[in]    language                        検出言語

* @return       操作の成否が返ります。
* @retval       MatchingResult_Success         一致するテキストが見つかった場合の値です。
* @retval       MatchingResult_Failed          一致するテキストが見つからなかった場合の値です。
* @retval       MatchingResult_Unexpected      不測のエラーが発生したときの値です。
* @retval       MatchingResult_InvalidImageId  不正な画像データ ID が指定された場合の値です。
* @retval       MatchingResult_NotInitialized  未初期化状態で呼び出されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      対象画像上に、指定された正規表現に一致する文字列が表示されているかをチェックします。@n
                指定する正規表現文字列は UTF-8 エンコードの文字列として指定してください。@n
                正規表現のマッチングには std::regex_match を使用しており、既定のロケール設定でマッチングが行われます。@n
                文字認識では、単語内の一部の認識に失敗することがあり、検出処理が成功しにくいことがあります。
                そのため、条件の緩い正規表現を指定することを推奨します。@n
                指定した検出言語以外の文字列の認識率は低くなります。
*/
CAPTUREPROCESSOR_API MatchingResult DetectTextForImage( ImageRect *pOutRect,
                                    ImageIdType id,
                                    const char regexString[],
                                    OcrSeparateLevel separate,
                                    OcrLanguage language );

/**
* @brief       	対象画像に対して OCR 処理を実行します。

* @param[out]   pOutResultCount                 OCR 処理実行後に検出された文字列数の出力先
* @param[out]   pOutOcrResultHandle             OCR の処理結果ハンドルの出力先
* @param[in]    id                              対象画像データの ID
* @param[in]    separate                        OCR の認識単位
* @param[in]    language                        検出言語

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success            操作に成功したときの値です。
* @retval       CaptureResult_Unexpected         不測のエラーが発生したときの値です。
* @retval       CaptureResult_NotInitialized     未初期化状態で呼び出されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      対象画像に対して OCR 処理を実行し、画像内の検出を行います。@n
                実行後は、separate の単位で検出された文字列を GetOcrResultStringSize、GetOcrResultString 関数を利用して取得することができます。@n
                不要となった処理結果は ReleaseOcrResult 関数を利用して解放してください。@n
                pOutResultCount へは、separate 単位で検出された文字列の数が渡されます。@n
                指定した検出言語以外の文字列の認識率は低くなります。
*/
CAPTUREPROCESSOR_API CaptureResult ExecuteOcr( int *pOutResultCount, OcrResultHandleType *pOutOcrResultHandle, ImageIdType id,  OcrSeparateLevel separate, OcrLanguage language );

/**
* @brief       	OCR 処理結果の指定番目の文字列のサイズを取得します。

* @param[out]   pOutResultSize                  OCR 処理結果の文字列サイズ
* @param[in]    ocrResultHandle                 OCR の処理結果ハンドル
* @param[in]    number                          文字列番号(0 以上、ExecuteOcr より取得した文字列数未満)

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success                 操作に成功したときの値です。
* @retval       CaptureResult_Unexpected              不測のエラーが発生したときの値です。
* @retval       CaptureResult_NotInitialized          未初期化状態で呼び出されたときの値です。
* @retval       CaptureResult_InvalidOcrResultHandle  不正な OCR 処理結果ハンドルが指定されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      OCR 処理結果の文字列サイズを pOutResultSize へ渡します。@n
                サイズにはヌル文字も含みます。
*/
CAPTUREPROCESSOR_API CaptureResult GetOcrResultStringSize( int *pOutResultSize, OcrResultHandleType ocrResultHandle, int number );

/**
* @brief       	OCR 処理結果の文字列、文字数、検出座標を取得します。

* @param[out]   pOutRect                        文字列の検出領域の出力先
* @param[out]   pOutResultSize                  取得できた文字列のサイズ
* @param[out]   pOutResultString                文字列の出力先
* @param[in]    bufferSize                      文字列の出力先の配列長
* @param[in]    ocrResultHandle                 OCR の処理結果ハンドル
* @param[in]    number                          文字列番号(0 以上、ExecuteOcr より取得した文字列数未満)

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success                 操作に成功したときの値です。
* @retval       CaptureResult_Unexpected              不測のエラーが発生したときの値です。
* @retval       CaptureResult_NotInitialized          未初期化状態で呼び出されたときの値です。
* @retval       CaptureResult_InvalidOcrResultHandle  不正な OCR 処理結果ハンドルが指定されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      OCR 処理結果の文字列、文字数、検出座標を取得します。@n
                取得した文字列はヌル終端された UTF-8 エンコードの文字列として格納されます。@n
*/
CAPTUREPROCESSOR_API CaptureResult GetOcrResultString( ImageRect *pOutRect, int *pOutResultSize, char pOutResultString[], int bufferSize, OcrResultHandleType ocrResultHandle, int number );

/**
* @brief       	OCR 処理結果を解放します。

* @param[in]    ocrResultHandle                    OCR の処理結果ハンドル

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success                  操作に成功したときの値です。
* @retval       CaptureResult_Unexpected               不測のエラーが発生したときの値です。
* @retval       CaptureResult_NotInitialized           未初期化状態で呼び出されたときの値です。
* @retval       CaptureResult_InvalidOcrResultHandle   不正な OCR 処理結果ハンドルが指定されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      OCR の処理結果を解放します。
*/
CAPTUREPROCESSOR_API CaptureResult ReleaseOcrResult( OcrResultHandleType ocrResultHandle );

/*----------------------------------------------------------------------
/ その他
/---------------------------------------------------------------------*/
/**
* @brief       	対象画像をウィンドウで表示します。

* @param[in]    windowName                      ウィンドウ名
* @param[in]    id                              表示する画像データの ID

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success           操作に成功したときの値です。
* @retval       CaptureResult_Unexpected        不測のエラーが発生したときの値です。
* @retval       CaptureResult_NotInitialized    未初期化状態で呼び出されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      対象画像をウィンドウで表示します。@n
                既に表示しているウィンドウと同一のウィンドウ名を指定した場合は、
                新規ウィンドウが生成されずに既存のウィンドウに表示されている画像が指定した ID の表示に更新されます。@n
                開発機のキャプチャプレビューを表示しているウィンドウと同一のウィンドウ名を指定することはできません。@n
                表示したウィンドウは、CloseImageWindow 関数より閉じることができます。また Finalize 関数実行後は全てのウィンドウが閉じられます。
*/
CAPTUREPROCESSOR_API CaptureResult OpenImageWindow( const char* windowName, ImageIdType id );

/**
* @brief       	対象ウィンドウを閉じます。

* @param[out]   windowName                      閉じるウィンドウ名

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success           操作に成功したときの値です。
* @retval       CaptureResult_Unexpected        不測のエラーが発生したときの値です。
* @retval       CaptureResult_NotInitialized    未初期化状態で呼び出されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      対象ウィンドウをクローズします。@n
                開発機のキャプチャプレビューを表示しているウィンドウと同一のウィンドウ名を指定することはできません。
*/
CAPTUREPROCESSOR_API CaptureResult CloseImageWindow( const char* windowName );

/**
* @brief       	対象画像の指定範囲を切り抜きます。

* @param[out]   pOutImageId                     処理後の画像データ ID の出力先
* @param[in]    id                              対象画像データの ID
* @param[in]    rect                            切り抜きを行う領域

* @return       操作の成否が返ります。
* @retval       CaptureResult_Success           操作に成功したときの値です。
* @retval       CaptureResult_Unexpected        不測のエラーが発生したときの値です。
* @retval       CaptureResult_NotInitialized    未初期化状態で呼び出されたときの値です。

* @pre          Initialize 関数の呼び出しが完了している。

* @details      対象画像の指定された領域を切り抜いた結果画像データ ID を pOutImageId へ渡します。
*/
CAPTUREPROCESSOR_API CaptureResult ExtractImage( ImageIdType *pOutImageId, ImageIdType id, ImageRect rect );
