﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "BuiltinDataDescrParser.h"

#include <cerrno>

#include "util.h"


//////////////////////////////////////////////////////////////////////////////
//  The FileLoader implementation to get cert data from DER file.
//////////////////////////////////////////////////////////////////////////////
FileLoader::FileLoader(const char *pFilePath) : m_dataSize(0)
{
    strcpy(m_pFilePath, pFilePath);
}


FileLoader::~FileLoader()
{
}


uint32_t FileLoader::GetDataBufSize()
{
    uint32_t                    ret = 0;
    FILE                        *pDataFile = nullptr;
    size_t                      size;
    int                         status;

    pDataFile = fopen(m_pFilePath, "rb");
    if (pDataFile == nullptr)
    {
        LOGE("FileLoader::GetDataBufSize: failed to open data file \'%s\': %d\n",
             m_pFilePath,
             errno);
        goto errExit;
    }

    status = fseek(pDataFile, 0L, SEEK_END);
    if (status != 0)
    {
        LOGE("FileLoader::GetDataBufSize: unable to seek to end of descr file: %d\n",
             errno);
        goto errExit;
    }

    size = ftell(pDataFile);
    if (size == -1L)
    {
        LOGE("FileLoader::GetDataBufSize: unable to get size of the file descr file: %d\n",
             errno);
        goto errExit;
    }

    //  Move back to the start of the file
    status = fseek(pDataFile, 0L, SEEK_SET);
    if (status < 0)
    {
        LOGE("FileLoader::GetDataBufSize: unable to reset descr file to start: %d\n", errno);
        goto errExit;
    }

    ret = static_cast<uint32_t>(size);
    LOGD("FileLoader::GetDataBufSize: file is %u bytes long\n", ret);

errExit:
    if (pDataFile != nullptr)
    {
        fclose(pDataFile);
        pDataFile = nullptr;
    }

    return ret;
}


int FileLoader::Load(uint8_t *pBuf, uint32_t maxBufSize, uint32_t *pOutDataSize)
{
    int                         ret = -1;
    FILE                        *pDataFile = nullptr;
    size_t                      bytesRead;

    if (m_dataSize == 0)
    {
        m_dataSize = GetDataBufSize();
    }

    if (m_dataSize == 0)
    {
        LOGE("FileLoader::Load failed to get file size for \'%s\'\n",
             );
        goto errExit;
    }

    if (maxBufSize < m_dataSize)
    {
        LOGE("FileLoader::Load provided size, %u, too small, need %u\n",
             maxBufSize,
             m_dataSize);
        goto errExit;
    }

    //  Open the file, read out its contents into the provided buffer
    pDataFile = fopen(m_pFilePath, "rb");
    if (pDataFile == nullptr)
    {
        LOGE("FileLoader::Load failed to open data file \'%s\': %d\n",
             m_pFilePath,
             errno);
        goto errExit;
    }

    bytesRead = fread(pBuf, sizeof(uint8_t), maxBufSize, pDataFile);
    if (bytesRead != static_cast<size_t>(maxBufSize))
    {
        if (feof(pDataFile) == 0)
        {
            LOGE("FileLoader::Load failed to read up to %u bytes from \'%s\': %d\n",
                 maxBufSize,
                 m_pFilePath,
                 ferror(pDataFile));
            goto errExit;
        }
    }

    *pOutDataSize = static_cast<uint32_t>(bytesRead);
    ret = 0;

errExit:
    if (pDataFile != nullptr)
    {
        fclose(pDataFile);
        pDataFile = nullptr;
    }

    return ret;
}


//////////////////////////////////////////////////////////////////////////////
//  The XML descriptor parser
//////////////////////////////////////////////////////////////////////////////
const xmlChar *BuiltinDataDescrParser::g_NnSdkNamespaceUrl =
    reinterpret_cast<const xmlChar *>("http://schemas.nintendo.com/ssl/desc");
const xmlChar *BuiltinDataDescrParser::g_ElementBuiltin =
    reinterpret_cast<const xmlChar *>("builtin");
const xmlChar *BuiltinDataDescrParser::g_ElementCertificate =
    reinterpret_cast<const xmlChar *>("certificate");
const xmlChar *BuiltinDataDescrParser::g_ElementCrl =
    reinterpret_cast<const xmlChar *>("crl");
const xmlChar *BuiltinDataDescrParser::g_AttrId =
    reinterpret_cast<const xmlChar *>("id");
const xmlChar *BuiltinDataDescrParser::g_AttrStatus =
    reinterpret_cast<const xmlChar *>("status");
const xmlChar *BuiltinDataDescrParser::g_AttrName =
    reinterpret_cast<const xmlChar *>("name");
const xmlChar *BuiltinDataDescrParser::g_AttrFilename =
    reinterpret_cast<const xmlChar *>("file");


BuiltinDataDescrParser::BuiltinDataDescrParser(const char *pInRoot,
                                               FILE       *pInFile) : m_pRoot(pInRoot),
                                                                      m_pInFile(pInFile),
                                                                      m_pXmlDoc(nullptr)
{
}


BuiltinDataDescrParser::~BuiltinDataDescrParser()
{
    if (m_pXmlDoc != nullptr)
    {
        LOGW("BuiltinDataDescrParser %p destroyed without Finalize\n",
             this);
    }
}


int BuiltinDataDescrParser::Initialize()
{
    int                         ret = -1;
    int                         status;
    long int                    size;
    size_t                      readSize;
    size_t                      desiredSize;
    char                        *pBuf = nullptr;

    //  Determine the size of the file so it can be read in and passed
    //  to the XML parser.
    status = fseek(m_pInFile, 0L, SEEK_END);
    if (status != 0)
    {
        LOGE("BuiltinDataDescrParser::Initialize: unable to seek to end of descr file: %d\n",
             errno);
        goto errExit;
    }

    size = ftell(m_pInFile);
    if (size == -1L)
    {
        LOGE("BuiltinDataDescrParser::Initialize: unable to get size of the file descr file: %d\n",
             errno);
        goto errExit;
    }

    //  Move back to the start of the file
    status = fseek(m_pInFile, 0L, SEEK_SET);
    if (status < 0)
    {
        LOGE("BuiltinDataDescrParser::Initialize: unable to reset descr file to start: %d\n",
             errno);
        goto errExit;
    }

    pBuf = new char[size + 1];
    if (pBuf == nullptr)
    {
        LOGE("BuiltinDataDescrParser::Initialize: failed to alloc scratch buf\n");
        goto errExit;
    }

    memset(pBuf, 0, sizeof(char) * (size + 1));

    //  Read out the entire file contents into RAM
    desiredSize = static_cast<size_t>(size);
    readSize = fread(pBuf, sizeof(char), desiredSize, m_pInFile);
    if (readSize != desiredSize)
    {
        LOGE("BuiltinDataDescrParser::Initialize: failed to read file contents: %d\n",
             ferror(m_pInFile));
        goto errExit;
    }

    m_pXmlDoc = xmlParseMemory(pBuf, size + 1);
    if (m_pXmlDoc == nullptr)
    {
        LOGE("BuiltinDataDescrParser::Initialize: unable to parse descr file and create XML doc\n");
        goto errExit;
    }

    //  Update status to success before cleaning up
    ret = 0;

errExit:
    if (pBuf != nullptr)
    {
        delete[] pBuf;
        pBuf = nullptr;
    }

    return ret;
}


void BuiltinDataDescrParser::Finalize()
{
    //  Destroy the XML doc, we're done with it
    if (m_pXmlDoc != nullptr)
    {
        xmlFreeDoc(m_pXmlDoc);
        m_pXmlDoc = nullptr;
    }
}


BuiltinDataInfo *BuiltinDataDescrParser::ParseCertificateElement(xmlNsPtr pNs, xmlNodePtr pNode)
{
    BuiltinDataInfo             *pRet = nullptr;
    FileLoader                  *pFileLoader = nullptr;
    char                        filenameBuf[AppConstants::g_MaxPathLen];
    const char                  *pTmpName;
    uint32_t                    id;
    uint32_t                    status;

    //  Extract the fields for the cert
    xmlChar *pName = xmlGetNsProp(pNode, g_AttrName, pNs->href);
    xmlChar *pId = xmlGetNsProp(pNode, g_AttrId, pNs->href);
    xmlChar *pStatus = xmlGetNsProp(pNode, g_AttrStatus, pNs->href);
    xmlChar *pFilename = xmlGetNsProp(pNode, g_AttrFilename, pNs->href);

    if ((pId == nullptr) ||
        (pStatus == nullptr) ||
        (pFilename == nullptr))
    {
        LOGE("ParseCertificateElement: malformed certificate element (id %s, status %s, file %s)\n",
             (pId != nullptr) ? "PRESENT" : "MISSING",
             (pStatus != nullptr) ? "PRESENT" : "MISSING",
             (pFilename != nullptr) ? "PRESENT" : "MISSING");
        goto errExit;
    }

    //  Extract the id and the status of the cert from the values
    pTmpName = reinterpret_cast<const char *>(pId);
    id = static_cast<uint32_t>(strtoul(pTmpName, nullptr, 0));
    pTmpName = reinterpret_cast<const char *>(pStatus);
    status = static_cast<uint32_t>(strtoul(pTmpName, nullptr, 0));

    if (id == BuiltinDataInfo::g_DataIdAll)
    {
        LOGE("ParseCertificateElement: reserved id 0x%8.8X found for \'%s\'\n",
             id,
             pName);
        goto errExit;
    }

    if (status > static_cast<uint32_t>(BuiltinDataInfo::BuiltinDataStatus_Revoked) ||
        status == static_cast<uint32_t>(BuiltinDataInfo::BuiltinDataStatus_Invalid))
    {
        LOGE("ParseCertificateElement: invalid status: %s\n",
             pTmpName);
        goto errExit;
    }

    nn_snprintf(filenameBuf,
                sizeof(filenameBuf),
                "%s%s%s%s%s",
                m_pRoot,
                FILE_PATH_DELIM,
                AppConstants::g_SslSrcBuiltin,
                FILE_PATH_DELIM,
                pFilename);
    pFileLoader = new FileLoader(filenameBuf);
    if (pFileLoader == nullptr)
    {
        LOGE("ParseCertificateElement: failed to create file loader for data\n");
        goto errExit;
    }

    pTmpName = reinterpret_cast<const char *>(pName);
    pRet = new BuiltinDataInfo(id,
                               static_cast<BuiltinDataInfo::BuiltinDataStatus>(status),
                               pTmpName,
                               pFileLoader);
    if (pRet == nullptr)
    {
        LOGE("ParseCertificateElement: failed to create new BuiltinDataInfo\n");
        goto errExit;
    }

    goto done;

errExit:
    if (pFileLoader != nullptr)
    {
        delete pFileLoader;
        pFileLoader = nullptr;
    }

done:
    return pRet;
}


BuiltinDataInfo *BuiltinDataDescrParser::ParseCrlElement(xmlNsPtr pNs, xmlNodePtr pNode)
{
    BuiltinDataInfo             *pRet = nullptr;
    FileLoader                  *pFileLoader = nullptr;
    char                        filenameBuf[AppConstants::g_MaxPathLen];
    const char                  *pTmpName;
    uint32_t                    id;
    uint32_t                    status;

    //  Extract the fields for the CRL element
    xmlChar *pName = xmlGetNsProp(pNode, g_AttrName, pNs->href);
    xmlChar *pId = xmlGetNsProp(pNode, g_AttrId, pNs->href);
    xmlChar *pStatus = xmlGetNsProp(pNode, g_AttrStatus, pNs->href);
    xmlChar *pFilename = xmlGetNsProp(pNode, g_AttrFilename, pNs->href);

    if ((pId == nullptr) ||
        (pStatus == nullptr) ||
        (pFilename == nullptr))
    {
        LOGE("ParseCrlElement: malformed CRL element (id %s, status %s, file %s)\n",
             (pId != nullptr) ? "PRESENT" : "MISSING",
             (pStatus != nullptr) ? "PRESENT" : "MISSING",
             (pFilename != nullptr) ? "PRESENT" : "MISSING");
        goto errExit;
    }

    //  Extract the id and the status of the CRL from the values
    pTmpName = reinterpret_cast<const char *>(pId);
    id = static_cast<uint32_t>(strtoul(pTmpName, nullptr, 0));
    pTmpName = reinterpret_cast<const char *>(pStatus);
    status = static_cast<uint32_t>(strtoul(pTmpName, nullptr, 0));

    if (id == BuiltinDataInfo::g_DataIdAll)
    {
        LOGE("ParseCrlElement: reserved id 0x%8.8X found for \'%s\'\n",
             id,
             pName);
        goto errExit;
    }

    //  Only Enabled and Removed are valid for CRL data types
    if (status > static_cast<uint32_t>(BuiltinDataInfo::BuiltinDataStatus_Enabled) ||
        status == static_cast<uint32_t>(BuiltinDataInfo::BuiltinDataStatus_Invalid))
    {
        LOGE("ParseCrlElement: invalid status: %s\n",
             pTmpName);
        goto errExit;
    }

    nn_snprintf(filenameBuf,
                sizeof(filenameBuf),
                "%s%s%s%s%s",
                m_pRoot,
                FILE_PATH_DELIM,
                AppConstants::g_SslSrcBuiltin,
                FILE_PATH_DELIM,
                pFilename);
    pFileLoader = new FileLoader(filenameBuf);
    if (pFileLoader == nullptr)
    {
        LOGE("ParseCrlElement: failed to create file loader for data\n");
        goto errExit;
    }

    pTmpName = reinterpret_cast<const char *>(pName);
    pRet = new BuiltinDataInfo(id,
                               static_cast<BuiltinDataInfo::BuiltinDataStatus>(status),
                               pTmpName,
                               pFileLoader);
    if (pRet == nullptr)
    {
        LOGE("ParseCrlElement: failed to create new BuiltinDataInfo\n");
        goto errExit;
    }

    goto done;

errExit:
    if (pFileLoader != nullptr)
    {
        delete pFileLoader;
        pFileLoader = nullptr;
    }

done:
    return pRet;
}


map<uint32_t, BuiltinDataInfo *> *BuiltinDataDescrParser::BuildBuiltinDataMap()
{
    map<uint32_t, BuiltinDataInfo *>    *pRet = nullptr;
    xmlNodePtr                          pCurNode;
    xmlNsPtr                            pNs;

    if (m_pXmlDoc == nullptr)
    {
        LOGE("BuildBuiltinDataMap: BuiltinDataDescrParser not initialized!\n");
        goto errExit;
    }

    //  Get the root element and make sure the namespace is set correctly
    //  and the root element is right.
    pCurNode = xmlDocGetRootElement(m_pXmlDoc);
    if (pCurNode == nullptr)
    {
        LOGE("BuildBuiltinDataMap: no root element\n");
        goto errExit;
    }

    pNs = xmlSearchNsByHref(m_pXmlDoc, pCurNode, g_NnSdkNamespaceUrl);
    if (pNs == nullptr)
    {
        LOGE("BuildBuiltinDataMap: nnsdk namespace not found\n");
        goto errExit;
    }

    if (xmlStrcmp(pCurNode->name, g_ElementBuiltin))
    {
        LOGE("BuildBuiltinDataMap: invalid root element \'%s\'\n",
             pCurNode->name);
        goto errExit;
    }

    pRet = new map<uint32_t, BuiltinDataInfo *>();
    if (pRet == nullptr)
    {
        LOGE("BuildBuiltinDataMap: could not alloc map\n");
        goto errExit;
    }

    //  Walk the list of sub-nodes.  There should only be a single level,
    //  so that's all which will be traversed.
    for (pCurNode = pCurNode->xmlChildrenNode;
         pCurNode != nullptr;
         pCurNode = pCurNode->next)
    {
        BuiltinDataInfo         *pInfo;

        if (xmlStrcmp(pCurNode->name, g_ElementCertificate) == 0)
        {
            pInfo = ParseCertificateElement(pNs, pCurNode);
            if (pInfo == nullptr)
            {
                goto errExit;
            }
        }
        else if (xmlStrcmp(pCurNode->name, g_ElementCrl) == 0)
        {
            pInfo = ParseCrlElement(pNs, pCurNode);
            if (pInfo == nullptr)
            {
                goto errExit;
            }
        }
        else
        {
            LOGD("BuildBuiltinDataMap: skip invalid element \'%s\'\n",
                 pCurNode->name);
            continue;
        }


        pair<uint32_t, BuiltinDataInfo *> insertData(pInfo->GetId(),
                                                     pInfo);
        pair<map<uint32_t, BuiltinDataInfo *>::iterator, bool> insertResult =
            pRet->insert(insertData);
        if (insertResult.second == false)
        {
            //  Error out, this is a duplicate
            LOGE("BuildBuiltinDataMap: FATAL - duplicate ID 0x%8.8X found with name \'%s\'\n",
                 pInfo->GetId(),
                 pInfo->GetName());
            delete pInfo;
            goto errExit;
        }
    }

    if (pRet->empty())
    {
        LOGE("BuildBuiltinDataMap: no trusted certs found\n");
        goto errExit;
    }

    goto done;

errExit:
    if (pRet != nullptr)
    {
        BuiltinDataDescrParser::DestroyBuiltinDataMap(pRet);
        pRet = nullptr;
    }

done:
    return pRet;
}


void BuiltinDataDescrParser::DestroyBuiltinDataMap(map<uint32_t, BuiltinDataInfo *> *pMap)
{
    for (map<uint32_t, BuiltinDataInfo *>::iterator it = pMap->begin();
         it != pMap->end();
         /*  Advance in loop  */)
    {
        BuiltinDataInfo         *pCurInfo = it->second;

        it = pMap->erase(it);
        delete pCurInfo;
    }

    delete pMap;
}
