﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace MakeFirmwareDebugSettings
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using Nintendo.ToolFoundation.CommandLine;

    internal static class Program
    {
        internal static void Main(string[] args)
        {
            try
            {
                var argument = new ProgramArgument();

                var parser = new DelegateBasedCommandLineParser();

                parser.AddHelp(Console.WriteLine);

                argument.RegiserWith(parser);

                if (!parser.ParseArgs(args, Console.WriteLine))
                {
                    Environment.Exit(1);
                }

                var outputPath = argument.OutputFilePath;

                if (File.Exists(outputPath))
                {
                    File.Delete(outputPath);
                }

                var bytes = new List<byte>();

                var name = GetSettingsName(argument.SettingsName);

                foreach (var filePath in argument.SettingsFilePaths)
                {
                    foreach (var entry in YamlParser.ParseFile(filePath))
                    {
                        var key = GetSettingsKey(entry.Key);

                        var mapKey = name + "!" + key;

                        var value = entry.Value;

                        var tuple = new Func<Tuple<Type, byte[]>>(() => {
                            if (value is bool)
                            {
                                return new Tuple<Type, byte[]>(
                                    Type.Bool,
                                    BitConverter.GetBytes((bool)value));
                            }
                            else if (value is int)
                            {
                                return new Tuple<Type, byte[]>(
                                    Type.Int,
                                    BitConverter.GetBytes((int)value));
                            }
                            else
                            {
                                return new Tuple<Type, byte[]>(
                                    Type.String,
                                    GetBytes((string)value));
                            }
                        })();

                        bytes = bytes.Concat(BitConverter.GetBytes(
                                                 (uint)(mapKey.Length + 1)))
                                     .Concat(GetBytes(mapKey))
                                     .Concat(new byte[] { (byte)tuple.Item1 })
                                     .Concat(BitConverter.GetBytes(
                                                 (uint)tuple.Item2.Length))
                                     .Concat(tuple.Item2)
                                     .ToList();
                    }
                }

                File.WriteAllBytes(outputPath, bytes.ToArray());
            }
            catch (Exception ex)
            {
                Console.Error.WriteLine("Error: {0}", ex.Message);

                Environment.Exit(1);
            }
        }

        private static byte[] GetBytes(string value)
        {
            return value.Select(x => (byte)x).Concat(new byte[] { 0 })
                        .ToArray();
        }

        private static bool IsSafeString(string str)
        {
            return (str.Length >= 1)
                && (!str.EndsWith("."))
                && (str.All(x => char.IsDigit(x)
                              || char.IsLower(x)
                              || new[] { '-', '.', '_' }.Contains(x)));
        }

        private static string GetSettingsName(string name)
        {
            if (!IsSafeString(name))
            {
                throw new FirmwareDebugSettingsException(
                    string.Format("Invalid settings name: '{0}'", name));
            }

            return name;
        }

        private static string GetSettingsKey(string key)
        {
            if (!IsSafeString(key))
            {
                throw new FirmwareDebugSettingsException(
                    string.Format("Invalid settings key: '{0}'", key));
            }

            return key;
        }

        private enum Type : byte
        {
            Unknown,
            String,
            Bool,
            Int,
        }
    }
}
