﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;

namespace MakeNsData
{
    public enum Endian
    {
        Little,
        Big,
    }

    // 共通ユーティリティ
    public static class Utility
    {
        public static byte[] Reorder(byte[] bytes, Endian endian)
        {
            if (BitConverter.IsLittleEndian ^ endian == Endian.Little)
            {
                return bytes.Reverse().ToArray();
            }
            else
            {
                return bytes;
            }
        }
    }

    // ログ出力
    public static class Log
    {
        public static string Prefix { get; set; } = "[MakeNsData] ";
        public static bool WithTrace { get; set; } = false;

        public static void WriteLine(string message)
        {
            Console.WriteLine(message);
            if (WithTrace)
            {
                System.Diagnostics.Trace.WriteLine(message);
            }
        }

        public static void WriteLine(string format, params object[] args)
        {
            var message = string.Format(format, args);
            Console.WriteLine(message);
            if (WithTrace)
            {
                System.Diagnostics.Trace.WriteLine(message);
            }
        }

        public static void WriteLineWithPrefix(string message)
        {
            Console.WriteLine(Prefix + message);
            if (WithTrace)
            {
                System.Diagnostics.Trace.WriteLine(Prefix + message);
            }
        }

        public static void WriteLineWithPrefix(string format, params object[] args)
        {
            var message = string.Format(Prefix + format, args);
            Console.WriteLine(message);
            if (WithTrace)
            {
                System.Diagnostics.Trace.WriteLine(message);
            }
        }
    }

    // ID（ApplicationId、ContentMetaId、TitleId など）
    public sealed class Id64 : IComparable, IComparable<Id64>, IEquatable<Id64>
    {
        private const ulong InvalidValue = 0;
        public static readonly Id64 Invalid = new Id64(InvalidValue);

        public static readonly Id64[] Unregistrable = {
                Invalid,
            };

        public ulong Value { get; }
        public string FormalStyle { get; }
        public string PrefixedStyle { get; }
        public bool IsValid { get; }

        public Id64(ulong value)
        {
            Value = value;

            FormalStyle = Value.ToString("x16");
            PrefixedStyle = $"0x{FormalStyle}";
            IsValid = InvalidValue != Value;
        }

        public Id64(string value, int radix = 16)
            : this(System.Convert.ToUInt64(value, radix))
        {
        }

        public override string ToString()
        {
            return FormalStyle;
        }

        public override int GetHashCode()
        {
            return Value.GetHashCode();
        }

        public int CompareTo(object obj)
        {
            return CompareTo(obj as Id64);
        }

        public int CompareTo(Id64 other)
        {
            if (object.ReferenceEquals(other, null))
            {
                return 1;
            }
            return Value.CompareTo(other.Value);
        }

        public override bool Equals(object obj)
        {
            return Equals(obj as Id64);
        }

        public bool Equals(Id64 other)
        {
            if (object.ReferenceEquals(other, null))
            {
                return false;
            }
            if (object.ReferenceEquals(other, this))
            {
                return true;
            }
            return Value.Equals(other.Value);
        }

        public static bool operator ==(Id64 lhs, Id64 rhs)
        {
            if (object.ReferenceEquals(lhs, null))
            {
                return object.ReferenceEquals(rhs, null);
            }
            return lhs.Equals(rhs);
        }

        public static bool operator !=(Id64 lhs, Id64 rhs)
        {
            return !(lhs == rhs);
        }

        public static bool TryParse(string s, out Id64 result)
        {
            try
            {
                result = new Id64(s);
            }
            catch (Exception)
            {
                result = Invalid;
                return false;
            }
            return true;
        }
    }
}
