﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using Microsoft.VisualBasic.FileIO;
using Nintendo.Foundation.IO;

namespace MakeNsData.Commands
{
    public class BlackListCommand : CommandBase
    {
        // 引数
        [CommandLineOption('i', "input", Description = "Input csv file path", IsRequired = true)]
        public string InputPathOption { get; set; }

        [CommandLineOption('o', "output", Description = "Output data file path.", IsRequired = true)]
        public string OutputPathOption { get; set; }

        // 定数
        private const int RecordCountMax = 32;

        // 内部変数
        private FileInfo InputFile { get; set; }
        private FileInfo OutputFile { get; set; }
        private SortedDictionary<Id64, uint> BlackList { get; set; }

        // コマンド実行
        public override void Run()
        {
            SetFiles(InputPathOption, OutputPathOption);
            ReadFile(InputFile.FullName);
            WriteFile(OutputFile.FullName);
        }

        // 対象ファイル確認
        private void SetFiles(string input, string output)
        {
            // 入力ファイル
            {
                if (!File.Exists(input))
                {
                    throw new Exception($"Couldn't find file \"{input}\".");
                }

                var info = new FileInfo(input);
                if (string.Compare(info.Extension, ".csv", true) != 0)
                {
                    throw new Exception($"Please specify csv format file as input.");
                }

                InputFile = info;
            }

            // 出力ファイル
            {
                var info = new FileInfo(output);
                var dir = info.Directory;
                if (!dir.Exists)
                {
                    dir.Create();
                }

                OutputFile = info;
            }
        }

        // 入力ファイル読み込み
        private void ReadFile(string path)
        {
            BlackList = new SortedDictionary<Id64, uint>();
            using (var parser = new TextFieldParser(path))
            {
                parser.Delimiters = new string[] { "," };
                while (!parser.EndOfData)
                {
                    var line = parser.LineNumber;
                    var fields = parser.ReadFields();
                    if (fields.Length == 0 || fields[0].StartsWith("#"))
                    {
                        // 空行、コメント行は無視
                        continue;
                    }
                    else if (fields.Length < 2)
                    {
                        throw new Exception($"Insufficient parameters in line {line}.");
                    }

                    Id64 id;
                    if (!Id64.TryParse(fields[0], out id) || id == Id64.Invalid)
                    {
                        throw new Exception($"Invalid application id \"{fields[0]}\" in line {line}.");
                    }

                    uint version;
                    if (!uint.TryParse(fields[1], out version) || version == uint.MaxValue)
                    {
                        throw new Exception($"Invalid version \"{fields[1]}\" in line {line}.");
                    }

                    if (!BlackList.ContainsKey(id))
                    {
                        BlackList.Add(id, version);
                    }
                    else if (BlackList[id] < version)
                    {
                        BlackList[id] = version;
                    }
                }
            }

            if (BlackList.Count > RecordCountMax)
            {
                PrintRecord();
                throw new Exception($"Too many records. Please limit the record to {RecordCountMax} or less.");
            }
            else if (Verbose)
            {
                PrintRecord();
            }
        }

        // 出力ファイル書き込み
        private void WriteFile(string path)
        {
            using (var writer = new BinaryWriter(File.Open(path, FileMode.Create)))
            {
                foreach (var pair in BlackList)
                {
                    var id = Utility.Reorder(BitConverter.GetBytes(pair.Key.Value), Endian.Little);
                    var version = Utility.Reorder(BitConverter.GetBytes(pair.Value), Endian.Little);
                    if (Verbose)
                    {
                        Log.WriteLine($"--------");
                        Log.WriteLine($"input:{pair.Key.PrefixedStyle},{pair.Value}");
                        Log.WriteLine($"output:{BitConverter.ToString(id)} {BitConverter.ToString(version)}");
                    }
                    writer.Write(id);
                    writer.Write(version);
                }
            }
        }

        // レコード一覧表示
        private void PrintRecord()
        {
            Log.WriteLine($"RecordCount={BlackList.Count}");
            foreach (var pair in BlackList)
            {
                Log.WriteLine($"{pair.Key.PrefixedStyle},{pair.Value}");
            }
        }
    }
}
