﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using Nintendo.Foundation.IO;
using System.Text.RegularExpressions;
using CommandUtility;

namespace ReplaceBootImagePackage
{
    public class ReplaceBootImagePackageArgument
    {
        [CommandLineOption('i', "input",
            Description = "set input nsp file.",
            IsRequired = true)]
        public string InputFile { get; set; }

        [CommandLineOption("overwrite",
            Description = "overwrite to input file.",
            IsRequired = false)]
        public bool OverWrite { get; set; }

        [CommandLineOption('o', "output",
            Description = "set output nsp path.",
            IsRequired = false)]
        public string OutputFile { get; set; }

        [CommandLineOption("bct",
            Description = "set bct file.",
            IsRequired = true)]
        public string BctFile { get; set; }

        [CommandLineOption("package1",
            Description = "set package1 file.",
            IsRequired = true)]
        public string Package1File { get; set; }

        [CommandLineOption("bct-mariko",
            Description = "set bct file for Mariko.",
            IsRequired = true)]
        public string BctMarikoFile { get; set; }

        [CommandLineOption("package1-mariko",
            Description = "set package1 file for Mariko.",
            IsRequired = true)]
        public string Package1MarikoFile { get; set; }

        [CommandLineOption("package2",
            Description = "set package2 file.",
            IsRequired = true)]
        public string Package2File { get; set; }
    }

    class Program
    {
        static void Main(string[] args)
        {
#if !DEBUG
            try
            {
#endif
            ReplaceBootImagePackageArgument parameters = new ReplaceBootImagePackageArgument();
            if (CommandLineParser.Default.ParseArgs<ReplaceBootImagePackageArgument>(args, out parameters))
            {
                using (var tempHolder = new TemporaryFileHolder("ReplaceFileInNsp"))
                {
                    var inputNsp = FileUtility.MakeExistedFileInfo("Input Nsp", parameters.InputFile);
                    var outputTemp = tempHolder.CreateTemporaryFilePath("replacedNsp", ".nsp");
                    var bctFile = FileUtility.MakeExistedFileInfo("Bct File", parameters.BctFile);
                    var package1File = FileUtility.MakeExistedFileInfo("Package1 File", parameters.Package1File);
                    var bctMarikoFile = FileUtility.MakeExistedFileInfo("Bct File for Mariko", parameters.BctMarikoFile);
                    var package1MarikoFile = FileUtility.MakeExistedFileInfo("Package1 File for Mariko", parameters.Package1MarikoFile);
                    var package2File = FileUtility.MakeExistedFileInfo("Package2 File", parameters.Package2File);

                    Program.ReplaceBootImagePackage(outputTemp, inputNsp, bctFile, package1File, bctMarikoFile, package1MarikoFile, package2File, tempHolder);

                    if (parameters.OverWrite)
                    {
                        File.Copy(outputTemp.FullName, parameters.InputFile, true);
                    }
                    else
                    {
                        File.Copy(outputTemp.FullName, parameters.OutputFile, true);
                    }
                }
            }
            else
            {
                return;
            }
#if !DEBUG
            }
            catch (Exception exception)
            {
                Console.Error.WriteLine("エラー: {0}", exception.Message);
                Console.Error.WriteLine("{0}", exception.StackTrace);
                Environment.Exit(1);
            }
#endif
        }

        private static void ReplaceBootImagePackage(FileInfo outputNsp, FileInfo inputNsp, FileInfo bctFile, FileInfo package1File, FileInfo bctMarikoFile, FileInfo package1MarikoFile, FileInfo package2File, TemporaryFileHolder tempHolder)
        {
            var tmp1 = tempHolder.CreateTemporaryFilePath("tmp1", ".nsp");
            var tmp2 = tempHolder.CreateTemporaryFilePath("tmp2", ".nsp");
            var tmp3 = tempHolder.CreateTemporaryFilePath("tmp2", ".nsp");
            var tmp4 = tempHolder.CreateTemporaryFilePath("tmp2", ".nsp");

            ReplaceFileInNsp(tmp1, inputNsp, "nx/bct", bctFile, tempHolder);
            ReplaceFileInNsp(tmp2, tmp1, "nx/package1", package1File, tempHolder);
            ReplaceFileInNsp(tmp3, tmp2, "a/bct", bctMarikoFile, tempHolder);
            ReplaceFileInNsp(tmp4, tmp3, "a/package1", package1MarikoFile, tempHolder);
            ReplaceFileInNsp(outputNsp, tmp4, "nx/package2", package2File, tempHolder); // package2はnx/marikoで共通
        }

        private static void ReplaceFileInNsp(FileInfo outputNsp, FileInfo inputNsp, string targetFile, FileInfo sourceFile, TemporaryFileHolder tempHolder)
        {
            var outputText = new StringBuilder();

            SdkTool.Execute(SdkPath.FindToolPath("AuthoringTool.exe"),
                new string[]
                {
                    "list",
                    "--keyconfig", Path.Combine(SdkPath.FindSdkRoot(), "Tools/CommandLineTools/AuthoringTool/AuthoringTool.repository.keyconfig.xml"),
                    inputNsp.FullName
                },
                output: outputText
            );

            var text = outputText.ToString();
            var match = Regex.Match(text, string.Format("([0-9a-fA-F]+.nca/.*{0})\\s*", targetFile));
            if (match.Success)
            {
                var outputDir = tempHolder.CreateTemporaryDirectory("ReplacedNsp");
                var targetFilePath = match.Groups[1].Value;
                SdkTool.Execute(SdkPath.FindToolPath("AuthoringTool.exe"),
                    new string[]
                    {
                        "replace",
                        "--keyconfig", Path.Combine(SdkPath.FindSdkRoot(), "Tools/CommandLineTools/AuthoringTool/AuthoringTool.repository.keyconfig.xml"),
                        "-o", outputDir.FullName,
                        inputNsp.FullName,
                        targetFilePath,
                        sourceFile.FullName
                    }
                );

                File.Copy(outputDir.EnumerateFiles().Single().FullName, outputNsp.FullName);
            }
            else
            {
                Console.Error.WriteLine("file list:");
                Console.Error.WriteLine(text);
                throw new Exception(string.Format("found no files: {0}", targetFile));
            }
        }
    }
}
