﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using CommandUtility;

namespace MakeInitialImage
{
    public class SaveDataExtractor
    {
        public SaveDataExtractor(Partition partition, TemporaryFileHolder tempHolder)
        {
            Partition = partition;
            TemporaryFileHolder = tempHolder;
            EscapedDirectory = TemporaryFileHolder.CreateTemporaryDirectory("EscapedSaveData");
        }

        public void Extract(DirectoryInfo targetPartitionDirectory)
        {
            if (!Partition.NeedsExtractingSaveData)
            {
                return;
            }

            foreach (var fileInfo in targetPartitionDirectory.EnumerateFiles("*", SearchOption.AllDirectories))
            {
                var relativePath = fileInfo.GetRelativePath(targetPartitionDirectory);
                var match = MatchPattern(relativePath);

                if (match != null)
                {
                    var targetPath = EscapedDirectory.CombineAsFile(relativePath);

                    if (!targetPath.Directory.Exists)
                    {
                        targetPath.Directory.Create();
                    }

                    var escapedFileInfo = fileInfo.CopyTo(targetPath);
                    ExtractedFiles.Add(Tuple.Create(relativePath, escapedFileInfo));
                }
            }
        }

        public void Export(Stream writer)
        {
            var partitionHeader = SavedataPartitionHeader.Create(ExtractedFiles.Count);
            Utility.WriteBinary(writer, partitionHeader);

            foreach (var targetSaveFile in ExtractedFiles)
            {
                var relativePath = targetSaveFile.Item1;
                var actualPath = targetSaveFile.Item2;

                var saveHeader = SaveDataInfoHeader.Create(relativePath, actualPath.Length);
                Utility.WriteBinary(writer, saveHeader);
                actualPath.CopyTo(writer);
            }
        }

        private string MatchPattern(string filePath)
        {
            foreach (var extractPattern in Partition.ExtractedSaves)
            {
                var regexPattern = extractPattern.Replace("*", ".*");
                var targetText = filePath.Replace("\\", "/");
                if( Regex.IsMatch(targetText, regexPattern) )
                {
                    return extractPattern;
                }
            }

            return null;
        }

        private Partition Partition;
        private TemporaryFileHolder TemporaryFileHolder;
        private DirectoryInfo EscapedDirectory;
        private List<Tuple<string, FileInfo>> ExtractedFiles = new List<Tuple<string, FileInfo>>();
    }
}
