﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.IO.Compression;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;
using YamlDotNet.Serialization;

namespace MakeInitialImage
{
    public class Program
    {
        static string tempPath = null;

        private static void PrintInstalledContent(InitialImageConfig config, string outputDir)
        {
            var partitions = from storage in config.StorageDefinitions
                             from partition in storage.Partitions
                             where storage.Name == "InternalStorage"
                             where partition.Name == "SAFE" || partition.Name == "SYSTEM"
                             select new { Name = partition.Name, Contents = partition.InstallContents };

            using (var sw = new StreamWriter(Path.Combine(outputDir, "InstalledContents.list")))
            {
                foreach (var partition in partitions)
                {
                    sw.WriteLine("{0}:", partition.Name);

                    if(partition.Contents != null)
                    {
                        foreach (var content in partition.Contents)
                        {
                            sw.WriteLine("  - {0}", content);
                        }
                    }
                }
            }
        }

        public static void ParseNfa(MakeInitialImageArguments parameters)
        {
            var variables = new Dictionary<string, string>();
            var inputFile = new FileInfo(parameters.InputPath);
            var outputFile = new FileInfo(parameters.OutputPath);
            var makeFatImagePath = Utility.EnsureExeFile(parameters.MakeFatImagePath, "MakeFatImage.exe");
            var installNspPath = Utility.EnsureExeFile(parameters.InstallNspPath, "InstallNspOnHost.exe");
            var nfaPath = inputFile.FullName;

            using (var tmpHolder = new TemporaryFileHolder("MakeInitialImage"))
            {
                var name = Path.GetFileName(nfaPath);
                var workingDirectoryRoot = tmpHolder.CreateTemporaryDirectory(name);
                ZipFile.ExtractToDirectory(nfaPath, workingDirectoryRoot.FullName);

                var files = new Dictionary<string, List<string>>();
                var dirs = new Dictionary<string, string>();
                var testDir = Path.Combine(workingDirectoryRoot.FullName);

                foreach (var dir in Directory.EnumerateDirectories(testDir))
                {
                    var partitionName = Path.GetFileName(dir);
                    files.Add(partitionName, new List<string>());
                    dirs.Add(partitionName, dir);

                    foreach (var file in Directory.EnumerateFiles(dir))
                    {
                        files[partitionName].Add(file);
                    }
                }

                const string configName = "config";
                //InitialImageConfig のパース
                if (!files.ContainsKey(configName))
                {
                    throw new Exception("No initialze config found.");
                }

                if (files[configName].Count != 1)
                {
                    throw new Exception("Invalid config files");
                }

                var initialImageConfig = InitialImageConfigParser.Parse(new FileInfo(files[configName].First()), variables);

                foreach (var storage in initialImageConfig.StorageDefinitions)
                {
                    foreach (var partition in storage.SpecialPartitions)
                    {
                        if (files.ContainsKey(partition.Name))
                        {
                            switch (partition.GeneratingType)
                            {
                                case PartitionGeneratingType.RawImage:
                                    partition.RawImageFile = files[partition.Name].First();
                                    break;
                                case PartitionGeneratingType.Installed:
                                    partition.InstallContents = files[partition.Name];
                                    break;
                                case PartitionGeneratingType.PairedInstalled:
                                    partition.InstallContents = files[partition.Name];
                                    break;
                            }
                        }
                    }

                    foreach (var partition in storage.Partitions)
                    {
                        if (files.ContainsKey(partition.Name))
                        {
                            if (partition.Name == "PRODINFOF")
                            {
                                partition.BaseDirectory = dirs[partition.Name];
                            }
                            else
                            {
                                switch (partition.GeneratingType)
                                {
                                    case PartitionGeneratingType.RawImage:
                                        partition.RawImageFile = files[partition.Name].First();
                                        break;
                                    case PartitionGeneratingType.Installed:
                                        partition.InstallContents = files[partition.Name];
                                        break;
                                    case PartitionGeneratingType.PairedInstalled:
                                        partition.InstallContents = files[partition.Name];
                                        break;
                                }
                            }
                        }
                    }
                }

                using (var outputStream = outputFile.OpenWrite())
                {
                    MakeInitialImage(outputStream, initialImageConfig, makeFatImagePath, installNspPath);
                }

                PrintInstalledContent(initialImageConfig, outputFile.Directory.FullName);

            }
        }

        public static void ParseYaml(MakeInitialImageArguments parameters)
        {
            var variables = parameters.MakeVariablesDictionary();
            var inputFile = new FileInfo(parameters.InputPath);
            var outputFile = new FileInfo(parameters.OutputPath);
            var initialImageConfig = InitialImageConfigParser.Parse(new FileInfo(parameters.InputPath), variables);
            var makeFatImagePath = Utility.EnsureExeFile(parameters.MakeFatImagePath, "MakeFatImage.exe");
            var installNspPath = Utility.EnsureExeFile(parameters.InstallNspPath, "InstallNspOnHost.exe");

            if (parameters.DumpProcessedConfig)
            {
                DumpProccessedConfig(outputFile, initialImageConfig);
            }

            using (var outputStream = outputFile.OpenWrite())
            {
                MakeInitialImage(outputStream, initialImageConfig, makeFatImagePath, installNspPath);
            }

            PrintInstalledContent(initialImageConfig, outputFile.Directory.FullName);
        }

        private static void DumpProccessedConfig(FileInfo outputImageFile, InitialImageConfig initialImageConfig)
        {
            var outputDir = outputImageFile.Directory;
            var outputFile = new FileInfo(Path.Combine(outputDir.FullName, Path.GetFileNameWithoutExtension(outputImageFile.Name) + ".config.yml"));

            using (var writer = outputFile.CreateText())
            {
                new YamlDotNet.Serialization.Serializer(SerializationOptions.DisableAliases).Serialize(writer, initialImageConfig);
            }
        }

        public static void Main(string[] args)
        {
            try
            {
                tempPath = Environment.GetEnvironmentVariable("MAKE_INITIAL_IMAGE_TEMP");
            }
            catch (Exception)
            {
                // If we throw a security exception, assume default behavior
                tempPath = null;
            }

            if (tempPath == null)
            {
                tempPath = Path.GetTempPath();
            }
#if !DEBUG
            try
            {
#endif
            MakeInitialImageArguments parameters = new MakeInitialImageArguments();
            if (CommandLineParser.Default.ParseArgs<MakeInitialImageArguments>(args, out parameters))
            {
                switch(Path.GetExtension(parameters.InputPath))
                {
                    case ".yml":
                        ParseYaml(parameters);
                    break;
                    case ".nfa":
                        ParseNfa(parameters);
                    break;
                }
            }
            else
            {
                return;
            }
#if !DEBUG
            }
            catch (Exception exception)
            {
                Console.Error.WriteLine("エラー: {0}", exception.Message);
                Console.Error.WriteLine("{0}", exception.StackTrace);
                Environment.Exit(1);
            }
#endif
        }

        public static void MakeInitialImage(FileStream outputImageStream, InitialImageConfig initialImageConfig, FileInfo makeFatImagePath, FileInfo installNspPath)
        {
            // TODO: パーティションごとに並列化できるので、パフォーマンス的に必要ならやる
            var constractor = new InitialImageConstructor(outputImageStream, initialImageConfig, new DirectoryInfo(tempPath), makeFatImagePath, installNspPath);
            constractor.ConstructImage();
        }
    }
}
