﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Windows;
using System.Threading.Tasks;
using System.Diagnostics;
using CommandUtility;

namespace MakeInitialImage
{
    public interface IFatImagePreprocessor
    {
        void Process(DirectoryInfo storageDirectory);
    }

    public class InstalledFatImage
    {
        public InstalledFatImage(FileInfo makeFatImagePath, FileInfo installNspPath, List<string> deletingFiles)
        {
            this.makeFatImagePath = makeFatImagePath;
            this.installNspPath = installNspPath;
            this.logger = new StringBuilder();
            this.deletingFiles = deletingFiles;
        }

        public void Create(FileInfo outputPath, DirectoryInfo baseDirectory, List<FileInfo> installContents, string storageType, Int64 partitionSizeMiB, bool isSparseImage, FileInfo keyFile = null, long ensuredFreeSpace = 0, List<Tuple<FileInfo, string>> escapedFiles = null, SaveDataExtractor saveDataExtractor = null)
        {
            installContents = installContents == null ? new List<FileInfo>() : installContents;

            using (var temporaryDirectoryHolder = new TemporaryFileHolder("InstalledFatImage", true))
            {
                try
                {
                    DirectoryInfo temporaryDirectory = temporaryDirectoryHolder.CreateTemporaryDirectory("FatImageRoot");

                    using (var writer = temporaryDirectoryHolder.CreateTemporaryFilePath("outputPath", ".txt").CreateText())
                    {
                        writer.WriteLine(outputPath);
                    }

                    if (baseDirectory != null)
                    {
                        Utility.CopyDirectory(baseDirectory, temporaryDirectory);
                    }

                    if (installContents != null)
                    {
                        foreach (var contentFile in installContents)
                        {
                            InstallContent(storageType, temporaryDirectory, contentFile);
                        }
                    }

                    foreach (var preprocessor in preprocessors)
                    {
                        preprocessor.Process(temporaryDirectory);
                    }

                    saveDataExtractor?.Extract(temporaryDirectory);

                    foreach (var fileEntry in deletingFiles == null ? new List<string>() : deletingFiles)
                    {
                        var path = Path.Combine(temporaryDirectory.FullName, fileEntry);
                        if (File.Exists(path))
                        {
                            File.Delete(path);
                        }

                        if (Directory.Exists(path))
                        {
                            new DirectoryInfo(path).DeleteWithRetry(true);
                        }
                    }

                    CreateFatImage(outputPath, temporaryDirectory, partitionSizeMiB, isSparseImage, keyFile, ensuredFreeSpace);
                }
                catch(Exception e)
                {
                    var totalNspSize = (from nsp in installContents select nsp.Length).Sum();

                    Console.Error.WriteLine($"StorageInfo:");
                    Console.Error.WriteLine($"    TotalStorageSize: {partitionSizeMiB * 1024 * 1024} Bytes");
                    Console.Error.WriteLine($"    TotalNspSize: {totalNspSize} Bytes");
                    Console.Error.WriteLine($"    EnsuredFreeSpace: {ensuredFreeSpace} Bytes");
                    Console.Error.WriteLine("");

                    Console.Error.WriteLine($"NspSize(num = {installContents.Count()}):");
                    foreach (var nsp in installContents)
                    {
                        Console.Error.WriteLine($"    {nsp.Name}: {nsp.Length} Bytes");
                    }
                    Console.Error.WriteLine("");

                    Console.Error.WriteLine($"{e.StackTrace}");

                    throw;
                }
            }

        }

        public void CreateFatImage(FileInfo fatImageFile, DirectoryInfo temporaryDirectory, long imageSizeMiB, bool isSparseImage, FileInfo keyFile = null, long ensuredFreeSpace = 0)
        {
            string arguments = string.Empty;

            arguments += "-o " + "\"" + fatImageFile.FullName + "\"";
            arguments += " -i " + "\"" + temporaryDirectory.FullName + "\"";
            arguments += " --size " + imageSizeMiB.ToString();
            arguments += " --verbose";
            arguments += " --ensure-free-space " + ensuredFreeSpace.ToString();

            if (isSparseImage)
            {
                arguments += " --sparse-image";
            }

            if (keyFile != null)
            {
                arguments += " --key-file \"" + keyFile.FullName + "\"";
            }

            using (var process = Process.Start(new ProcessStartInfo()
            {
                FileName = makeFatImagePath.FullName,
                Arguments = arguments,
                CreateNoWindow = true,
                UseShellExecute = false,
                RedirectStandardError = true,
                RedirectStandardOutput = true
            }))
            {
                process.OutputDataReceived += OutputDataReceived;
                process.ErrorDataReceived += ErrorDataReceived;
                process.BeginOutputReadLine();
                process.BeginErrorReadLine();
                process.WaitForExit();

                if (process.ExitCode != 0)
                {
                    Console.Error.Write(this.logger.ToString());
                    throw new InvalidDataException(string.Format("Failed to create FAT iamge. Arguments={0}", string.Join(",", arguments)));
                }

                if (!File.Exists(fatImageFile.FullName))
                {
                    Console.Error.Write(this.logger.ToString());
                    throw new InvalidDataException(string.Format("Failed to create FAT iamge. Arguments={0}", string.Join(",", arguments)));
                }
            }
        }

        public void AddPreprocessor(IFatImagePreprocessor preprocessor)
        {
            this.preprocessors.Add(preprocessor);
        }

        private void InstallContent(string installStorage, DirectoryInfo temporaryDirectory, FileInfo contentFile)
        {
            using (var process = Process.Start(new ProcessStartInfo()
            {
                FileName = installNspPath.FullName,
                Arguments = string.Format("\"{0}\" \"{1}\" \"{2}\"", installStorage, temporaryDirectory.FullName, contentFile.FullName),
                CreateNoWindow = true,
                RedirectStandardOutput = true,
                RedirectStandardError = true,
                UseShellExecute = false
            }))
            {
                process.OutputDataReceived += OutputDataReceived;
                process.ErrorDataReceived += ErrorDataReceived;
                process.BeginErrorReadLine();
                process.BeginOutputReadLine();
                process.WaitForExit();

                if (process.ExitCode != 0)
                {
                    Console.Error.Write(this.logger.ToString());
                    throw new InvalidDataException(string.Format("Failed to install the content. ContentName:{0}", contentFile.FullName));
                }
            }
        }

        private void ErrorDataReceived(object sender, DataReceivedEventArgs e)
        {
            if (e.Data != string.Empty && e.Data != null)
            {
                this.logger.AppendLine(e.Data);
            }
        }

        private void OutputDataReceived(object sender, DataReceivedEventArgs e)
        {
            if (e.Data != string.Empty && e.Data != null)
            {
                this.logger.AppendLine(e.Data);
            }
        }

        private StringBuilder logger;
        private FileInfo makeFatImagePath;
        private FileInfo installNspPath;
        private List<string> deletingFiles;
        private List<IFatImagePreprocessor> preprocessors = new List<IFatImagePreprocessor>();
    }

}
