﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using CommandUtility;

namespace MakeInitialImage
{
    public class InitialImageConfig
    {
        public InitialImageDefinition InitialImageDefinition { get; set; }
        public List<StorageDefinition> StorageDefinitions { get; set; }
    }

    public class InitialImageDefinition
    {
        public string Name { get; set; }
        public string Version { get; set; }
        public bool AllowOverwritingProductionInfo { get; set; }
    }

    public class StorageDefinition
    {
        public string Name { get; set; }
        public ByteUnitExpression SectorSize { get; set; }
        public ByteUnitExpression Size { get; set; }
        public PartitionFormat PartitionFormat { get; set; }
        public List<Partition> SpecialPartitions { get; set; }
        public List<Partition> Partitions { get; set; }
        public List<Dictionary<string, string>> Commands { get; set; }
        public string DiskGuid { get; set; }

        public Partition FindPartition(string name)
        {
            var partition = Partitions.Find(p => p.Name == name);
            if (partition != null)
            {
                return partition;
            }
            else
            {
                return SpecialPartitions.Find(p => p.Name == name);
            }
        }
    }

    public enum PartitionFormat
    {
        GPT
    }

    public enum InstallStorageType
    {
        System,
        User
    }

    public class Partition
    {
        public string Name { get; set; }
        public string Guid { get; set; }
        public string TypeGuid { get; set; }
        public string FsPartitionId { get; set; }
        public PartitionGeneratingType GeneratingType { get; set; }
        public ByteUnitExpression Size { get; set; }
        public string RawImageFile { get; set; }
        public FileSystem FileSystem { get; set; }
        public InstallStorageType InstallStorageType { get; set; }
        public List<string> InstallContents { get; set; }
        public List<string> DeleteFiles { get; set; }
        public List<string> ExtractedSaves { get; set; }
        public string BaseDirectory { get; set; }
        public List<string> EncryptedArchives { get; set; }
        public PartitionAllocationType AllocationType { get; set; }
        public string Address { get; set; }
        public string Offset { get; set; }
        public bool UpdateSecureInfo { get; set; }
        public string PairedPartition { get; set; }
        public long AddressAsNumber
        {
            get
            {
                return Convert.ToInt64(Address, 16);
            }
        }
        public long OffsetAsNumber
        {
            get
            {
                if (Offset != null)
                {
                    return Convert.ToInt64(Offset, 16);
                }
                else
                {
                    return 0;
                }
            }
        }
        public bool NeedsExtractingSaveData
        {
            get
            {
                return ExtractedSaves != null && ExtractedSaves.Count > 0;
            }
        }
    }

    public enum FileSystem
    {
        FAT32
    }

    public enum PartitionGeneratingType
    {
        RawImage,
        SparseImage,
        Installed,
        PairedInstalled,
        Clean,
        Keep
    }

    public enum PartitionType
    {
        Required,
        Optional
    }

    public enum PartitionAllocationType
    {
        Sequential,
        Fixed,
        Keep
    }
}
