﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;

namespace MakeInitialImage
{
    [Serializable()]
    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
    public struct ProtectiveMbr
    {
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = SizeOfBootCode)]
        public byte[] BootCode;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = SizeOfUniqueMbrDiskSignature)]
        public byte[] UniqueMbrDiskSignature;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = SizeOfUnknownArea)]
        public byte[] UnknownArea;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = NumPartitionRecord)]
        public ProtectiveMbrPartitionRecord[] PartitionRecords;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = SizeOfSignature)]
        public byte[] Signature;

        public static ProtectiveMbr Create()
        {
            return new ProtectiveMbr()
            {
                BootCode = new byte[SizeOfBootCode],
                UniqueMbrDiskSignature = new byte[SizeOfUniqueMbrDiskSignature],
                UnknownArea = new byte[SizeOfUnknownArea],
                PartitionRecords = new ProtectiveMbrPartitionRecord[NumPartitionRecord]
                {
                    ProtectiveMbrPartitionRecord.CreateFirstPartitionRecord(),
                    ProtectiveMbrPartitionRecord.CreateInvalidetedPartitionRecord(),
                    ProtectiveMbrPartitionRecord.CreateInvalidetedPartitionRecord(),
                    ProtectiveMbrPartitionRecord.CreateInvalidetedPartitionRecord()
                },
                Signature = new byte[SizeOfSignature] { 0x55, 0xAA },
            };
        }

        public const int SizeOfBootCode = 440;
        public const int SizeOfUniqueMbrDiskSignature = 4;
        public const int SizeOfUnknownArea = 2;
        public const int NumPartitionRecord = 4;
        public const int SizeOfSignature = 2;
    }

    [Serializable()]
    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
    public struct ProtectiveMbrPartitionRecord
    {
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = SizeOfBootIndicator)]
        public byte[] BootIndicator;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = SizeOfStartingChs)]
        public byte[] StartingChs;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = SizeOfOsType)]
        public byte[] OsType;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = SizeOfEndingChs)]
        public byte[] EndingChs;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = SizeOfStartingLba)]
        public byte[] StartingLba;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = SizeOfSizeInLba)]
        public byte[] SizeInLba;

        public static ProtectiveMbrPartitionRecord CreateInvalidetedPartitionRecord()
        {
            return new ProtectiveMbrPartitionRecord()
            {
                BootIndicator = new byte[SizeOfBootIndicator],
                StartingChs = new byte[SizeOfStartingChs],
                OsType = new byte[SizeOfOsType],
                EndingChs = new byte[SizeOfEndingChs],
                StartingLba = new byte[SizeOfStartingLba],
                SizeInLba = new byte[SizeOfSizeInLba],
            };
        }

        public static ProtectiveMbrPartitionRecord CreateFirstPartitionRecord()
        {
            return new ProtectiveMbrPartitionRecord()
            {
                BootIndicator = new byte[SizeOfBootIndicator],
                StartingChs = new byte[SizeOfStartingChs] { 0x00, 0x02, 0x00 },
                OsType = new byte[SizeOfOsType] { 0xEE },
                EndingChs = new byte[SizeOfEndingChs] { 0xFF, 0xFF, 0xFF },
                StartingLba = new byte[SizeOfStartingLba] { 0x01, 0x00, 0x00, 0x00 },
                SizeInLba = new byte[SizeOfSizeInLba] { 0xFF, 0xFF, 0xFF, 0xFF }
            };
        }

        public const int SizeOfBootIndicator = 1;
        public const int SizeOfStartingChs = 3;
        public const int SizeOfOsType = 1;
        public const int SizeOfEndingChs = 3;
        public const int SizeOfStartingLba = 4;
        public const int SizeOfSizeInLba = 4;
    }

    [Serializable()]
    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
    public struct GptHeader
    {
        public ulong Signature;

        public uint Revision;

        public uint HeaderSize;

        public uint HeaderCrc32;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = SizeOfReserved)]
        public byte[] Reserved;

        public long MyLba;

        public long AlternateLba;

        public long FirstUsableLba;

        public long LastUsableLba;

        public Guid DiskGuid;

        public long PartitionEntryLba;

        public int NumberOfPartitionEntries;

        public int SizeOfPartitionEntry;

        public uint PartitionEntryArrayCrc32;

        public static GptHeader Create(Guid diskGuid, int numPartitions)
        {
            return new GptHeader()
            {
                Signature = 0x5452415020494645UL,
                Revision = 0x00010000,
                HeaderSize = 92,
                HeaderCrc32 = 0,
                Reserved = new byte[SizeOfReserved],
                MyLba = 1,
                AlternateLba = 0,
                FirstUsableLba = 0,
                LastUsableLba = 0,
                DiskGuid = diskGuid,
                PartitionEntryLba = 2,
                NumberOfPartitionEntries = numPartitions,
                SizeOfPartitionEntry = 128,
                PartitionEntryArrayCrc32 = 0
            };
        }

        public const int SizeOfReserved = 4;
    }

    [Serializable()]
    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Unicode)]
    public struct GptPartitionEntry
    {
        public Guid PartitionTypeGuid;

        public Guid UniquePartitionGuid;

        public long StartingLba;

        public long EndingLba;

        public ulong Attributes;

        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = SizeOfPartitionName)]
        public string PartitionName;

        public static GptPartitionEntry Create(Guid partitionType, Guid partitionGuid, long startingLba, long endingLba, ulong attributes, string partitionName)
        {
            return new GptPartitionEntry()
            {
                PartitionTypeGuid = partitionType,
                UniquePartitionGuid = partitionGuid,
                StartingLba = startingLba,
                EndingLba = endingLba,
                Attributes = attributes,
                PartitionName = partitionName,
            };
        }

        public const int SizeOfPartitionName = 36;
    }

    public class GptDefinition
    {
        public const int ReservedPartitions = 128;
        public const int DefaultPartitionSize = 128;
    }
}
