﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/utilTool/utilTool_CommandFramework.h>
#include <nn/fs.h>
#include <nn/os.h>
#include "MakeFatImage.h"
#include "ListFiles.h"

namespace{

    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        return free(p);
    }

} // namespace

class MakeFatImageCommand
{
public:
    MakeFatImageCommand() : m_GenerateSparseImage(false), m_EnsuredFreeSpace(0) {}

    const char* Name()
    {
        return "MakeFatImage";
    }

    const char* Summary()
    {
        return "Make FAT image from directory.";
    }

    const char* Description()
    {
        return "";
    }

    void Arguments(nn::utilTool::SingleCommandInterface &commandInterface)
    {
        commandInterface.AddKeywordArgument(&m_OutputPath, 'o', "output", "Output this file as FAT image.", true);
        commandInterface.AddKeywordArgument(&m_InputDirectory, 'i', "input", "Create FAT image from this directory.", true);
        commandInterface.AddKeywordArgument(&m_ImageSize, "size", "Set FAT image size(MiB).", true);
        commandInterface.AddFlagArgument(&m_GenerateSparseImage, "sparse-image", "Generate sparse image.");
        commandInterface.AddKeywordArgument(&m_EncryptionKeyFile, 'k', "key-file", "Set key file.", false);
        commandInterface.AddFlagArgument(&m_ListUpFiles, "list", "list up files in partition.");
        commandInterface.AddKeywordArgument(&m_EnsuredFreeSpace, "ensure-free-space", "Set minium free space to be allowed (bytes).", false);
    }

    int Main()
    {
        nn::fs::SetAllocator(Allocate, Deallocate);
        nn::fs::MountHostRoot();

        bool enableEncryption = !m_EncryptionKeyFile.empty();

        if (m_ListUpFiles)
        {
            return ListUpFiles(m_InputDirectory.c_str(), enableEncryption, m_EncryptionKeyFile.c_str());
        }

        nn::Result result = MakeFatImage(m_OutputPath.c_str(), m_ImageSize, m_InputDirectory.c_str(), m_GenerateSparseImage, enableEncryption, m_EncryptionKeyFile.c_str(), m_EnsuredFreeSpace);

        if (result.IsFailure())
        {
            NN_UTILTOOL_LOG_ERROR("failure: module=%d, description=%d, value=%08x", result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());

            if(result <= nn::fs::ResultUsableSpaceNotEnough())
            {
                NN_UTILTOOL_LOG_ERROR("Usable space not enough");
            }
            else if(result <= nn::fs::ResultPathNotFound())
            {
                NN_UTILTOOL_LOG_ERROR("Path not found");
            }
            else if(result <= nn::fs::ResultPathAlreadyExists())
            {
                NN_UTILTOOL_LOG_ERROR("Path already exists");
            }

            NN_UTILTOOL_LOG_ERROR("Arguments:");
            int numArgs = nn::os::GetHostArgc();
            for(int i = 0; i < numArgs; i++)
            {
                NN_UTILTOOL_LOG_ERROR("  %d: %s", i, nn::os::GetHostArgv()[i]);
            }
            return 1;
        }
        else
        {
            NN_UTILTOOL_LOG_VERBOSE("done.");
            return 0;
        }
    }

    int ListUpFiles(const char* inputImage, bool enableEncryption, const char *encryptionKeyFile)
    {
        NN_UTILTOOL_LOG_VERBOSE("List up files.");

        auto result = ListFiles(inputImage, enableEncryption, encryptionKeyFile);

        if (result.IsFailure())
        {
            NN_UTILTOOL_LOG_ERROR("Failed to list files. result=0x%08x, path = %s", result.GetInnerValueForDebug(), inputImage);
            return 1;
        }

        return 0;
    }

private:
    std::string m_OutputPath;
    std::string m_InputDirectory;
    std::string m_EncryptionKeyFile;
    int m_ImageSize;
    bool m_GenerateSparseImage;
    bool m_ListUpFiles;
    int64_t m_EnsuredFreeSpace;
};


extern "C" void nnMain()
{
    std::exit(nn::utilTool::BasicCommandFramework<MakeFatImageCommand>::Run(
        nn::os::GetHostArgc(),
        const_cast<const char**>(nn::os::GetHostArgv())));
}

