﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.Diagnostics;
using System.Threading;

namespace Nintendo.HtcTools.Htclow
{
    /// <summary>
    /// チャネルごとの受信バッファです。
    /// パケットのシーケンス番号はこのクラス内で管理されます。
    /// </summary>
    internal class ReceiveBuffer : IDisposable
    {
        // 受信データを保持するパケットのリスト
        // 読み出されて不要になったパケットは破棄される
        private IList<Packet> m_Packets = new List<Packet>();

        // リストの先頭のパケットにおいて Read 済のバイト数
        private int m_HeadOffset = 0;

        // 受信データサイズがこの値を超えた時に、ReceiveReady が set される
        private int m_ReceiveReadyThreshold = 0;

        public int ReceiveReadyThreshold
        {
            get
            {
                return m_ReceiveReadyThreshold;
            }
            set
            {
                m_ReceiveReadyThreshold = value;
                UpdateReceiveReady();
            }
        }

        // 受信データの到達を示すイベント
        public ManualResetEventSlim m_ReceiveReady = new ManualResetEventSlim(true);
        public ManualResetEventSlim ReceiveReady
        {
            get
            {
                return m_ReceiveReady;
            }
        }

        public long NextSequenceId { get; private set; } = 1;

        public void AddPacket(Packet packet)
        {
            if (NextSequenceId != packet.SequenceId)
            {
                // パケットを破棄
                throw new ChannelSequenceIdException()
                {
                    Expect = NextSequenceId,
                    Actual = packet.SequenceId,
                };
            }

            var readbleBodySize = GetReadableBodySize();

            if (packet.BodySize > 0)
            {
                lock (m_Packets)
                {
                    m_Packets.Add(packet);
                }
            }

            UpdateReceiveReady();
            NextSequenceId++;
        }

        public int GetReadableBodySize()
        {
            lock (m_Packets)
            {
                return m_Packets.Select(packet => packet.BodySize).Sum() - m_HeadOffset;
            }
        }

        public int ReadBody(byte[] buffer, int offset, int length)
        {
            if (buffer == null)
            {
                throw new ArgumentNullException();
            }

            if (offset < 0 || buffer.Length < offset)
            {
                throw new ArgumentException();
            }

            if (length < 0 || buffer.Length < offset + length)
            {
                throw new ArgumentException();
            }

            if (GetReadableBodySize() < length)
            {
                // TORIAEZU: ReadBody() を呼ぶ前にあらかじめ GetReadableBodySize() を呼んでデータが十分あるチェックするのが前提
                throw new ArgumentException("Requested length is too long.");
            }

            if (length == 0)
            {
                return 0;
            }

            lock (m_Packets)
            {
                var totalCopiedSize = 0;

                while (totalCopiedSize < length)
                {
                    var packet = m_Packets.First();
                    var copySize = Math.Min(length - totalCopiedSize, packet.BodySize - m_HeadOffset);

                    System.Buffer.BlockCopy(packet.GetBodyBytes(), m_HeadOffset, buffer, offset + totalCopiedSize, copySize);
                    totalCopiedSize += copySize;

                    m_HeadOffset += copySize;
                    Debug.Assert(m_HeadOffset <= packet.BodySize, "Buffer state is corrupted.");

                    if (m_HeadOffset == packet.BodySize)
                    {
                        // リストの先頭のパケットを最後まで読み切った場合
                        m_Packets.Remove(packet);
                        m_HeadOffset = 0;
                    }
                }
            }

            UpdateReceiveReady();
            return length;
        }

        private void UpdateReceiveReady()
        {
            if (GetReadableBodySize() >= m_ReceiveReadyThreshold)
            {
                m_ReceiveReady.Set();
            }
            else
            {
                m_ReceiveReady.Reset();
            }
        }

        #region Dispose
        public void Dispose()
        {
            m_ReceiveReady.Dispose();
        }
        #endregion
    }
}
