﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.Threading;

namespace Nintendo.HtcTools.Htclow
{
    internal enum ChannelState
    {
        Opened,
        SynReceived,
        SynWait,
        Established,
        CloseWait,
        LastAck,
        FinWait1,
        FinWait2,
        Closing,
        Closed,
    }

    /// <summary>
    /// チャネルごとに確保・解放するリソースをまとめたクラスです。
    /// </summary>
    internal sealed class ChannelResource : IDisposable
    {
        public ushort Version
        {
            get
            {
                return PacketFactory.Version;
            }
            set
            {
                PacketFactory.Version = value;
            }
        }

        private ChannelState m_State = ChannelState.Opened;
        public ChannelState State
        {
            get
            {
                return m_State;
            }
            set
            {
                if (IsStateTransitionAllowed(m_State, value))
                {
                    m_State = value;

                    if (m_State == ChannelState.Established)
                    {
                        Established.Set();
                    }
                    if (m_State == ChannelState.Closed)
                    {
                        Closed.Set();
                    }
                }
                else
                {
                    throw new ChannelStateTransitionException() { Current = m_State, Next = value };
                }
            }
        }

        public bool RetransmitEnable
        {
            set
            {
                SendBuffer.RetransmitEnable = value;
            }
        }

        private ManualResetEventSlim m_Established = new ManualResetEventSlim(false);
        public ManualResetEventSlim Established {
            get
            {
                return m_Established;
            }
        }

        private ManualResetEventSlim m_Closed = new ManualResetEventSlim(false);
        public ManualResetEventSlim Closed
        {
            get
            {
                return m_Closed;
            }
        }

        public ManualResetEventSlim SendCompleted
        {
            get
            {
                return SendBuffer.EmptyEvent;
            }
        }

        private PacketFactory m_PacketFactory = new PacketFactory();
        public PacketFactory PacketFactory
        {
            get
            {
                return m_PacketFactory;
            }
        }

        private ReceiveBuffer m_ReceiveBuffer = new ReceiveBuffer();
        public ReceiveBuffer ReceiveBuffer
        {
            get
            {
                return m_ReceiveBuffer;
            }
        }

        private SendBuffer m_SendBuffer = new SendBuffer();
        public SendBuffer SendBuffer
        {
            get
            {
                return m_SendBuffer;
            }
        }

        public void CheckPacketVersion(Packet packet)
        {
            if (packet.Version != Version)
            {
                throw new HtclowException($"Packet version ({packet.Version}) is not matched with channel's ({Version}).");
            }
        }

        private bool IsStateTransitionAllowed(ChannelState current, ChannelState next)
        {
            switch (current)
            {
                case ChannelState.Opened:
                    return next == ChannelState.SynWait || next == ChannelState.SynReceived;
                case ChannelState.SynReceived:
                    return next == ChannelState.Established;
                case ChannelState.SynWait:
                    return next == ChannelState.Established;
                case ChannelState.Established:
                    return next == ChannelState.FinWait1 || next == ChannelState.CloseWait;
                case ChannelState.CloseWait:
                    return next == ChannelState.LastAck;
                case ChannelState.LastAck:
                    return next == ChannelState.Closed;
                case ChannelState.FinWait1:
                    return next == ChannelState.FinWait2 || next == ChannelState.Closing;
                case ChannelState.FinWait2:
                    return next == ChannelState.Closed;
                case ChannelState.Closing:
                    return next == ChannelState.Closed;
                default:
                    throw new ChannelStateTransitionException() { Current = current, Next = next };
            }
        }

        public void Dispose()
        {
            m_ReceiveBuffer.Dispose();
            m_SendBuffer.Dispose();
            m_Established.Dispose();
            m_Closed.Dispose();
        }
    }
}
