﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.IO;

namespace Nintendo.HtcTools.HtcfsLibrary
{
    internal class Header
    {
        public const int Size = 64;

        private const int OffsetProtocol = 0;
        private const int OffsetVersion = 2;
        private const int OffsetPacketCategory = 4;
        private const int OffsetPacketType = 6;
        private const int OffsetBodySize = 8;
        private const int OffsetParam0 = 16;
        private const int OffsetParam1 = 24;
        private const int OffsetParam2 = 32;
        private const int OffsetParam3 = 40;
        private const int OffsetParam4 = 48;
        private const int OffsetReserved = 56;

        private SpanBuffer buffer;

        public Header()
        {
            buffer = new SpanBuffer(Size);
        }

        public Header(byte[] buffer)
        {
            if (buffer.Length < Size)
            {
                throw new HtcfsException($"Given buffer size ({buffer.Length}) is too small.");
            }

            this.buffer = new SpanBuffer(buffer);
        }

        public Header(SpanBuffer buffer)
        {
            if (buffer.Length < Size)
            {
                throw new HtcfsException($"Given buffer size ({buffer.Length}) is too small.");
            }

            this.buffer = buffer;
        }

        public short Protocol
        {
            get
            {
                return buffer.GetInt16(OffsetProtocol);
            }
            set
            {
                buffer.SetInt16(OffsetProtocol, value);
            }
        }

        public short Version
        {
            get
            {
                return buffer.GetInt16(OffsetVersion);
            }
            set
            {
                buffer.SetInt16(OffsetVersion, value);
            }
        }

        public PacketCategory PacketCategory
        {
            get
            {
                return (PacketCategory)buffer.GetInt16(OffsetPacketCategory);
            }
            set
            {
                buffer.SetInt16(OffsetPacketCategory, (short)value);
            }
        }

        public PacketType PacketType
        {
            get
            {
                return (PacketType)buffer.GetInt16(OffsetPacketType);
            }
            set
            {
                buffer.SetInt16(OffsetPacketType, (short)value);
            }
        }

        public long BodySize
        {
            get
            {
                return buffer.GetInt64(OffsetBodySize);
            }
            set
            {
                buffer.SetInt64(OffsetBodySize, (long)value);
            }
        }

        public long Param0
        {
            get
            {
                return buffer.GetInt64(OffsetParam0);
            }
            set
            {
                buffer.SetInt64(OffsetParam0, value);
            }
        }

        public long Param1
        {
            get
            {
                return buffer.GetInt64(OffsetParam1);
            }
            set
            {
                buffer.SetInt64(OffsetParam1, value);
            }
        }

        public long Param2
        {
            get
            {
                return buffer.GetInt64(OffsetParam2);
            }
            set
            {
                buffer.SetInt64(OffsetParam2, value);
            }
        }

        public long Param3
        {
            get
            {
                return buffer.GetInt64(OffsetParam3);
            }
            set
            {
                buffer.SetInt64(OffsetParam3, value);
            }
        }

        public long Param4
        {
            get
            {
                return buffer.GetInt64(OffsetParam4);
            }
            set
            {
                buffer.SetInt64(OffsetParam4, value);
            }
        }

        public byte[] GetBytes()
        {
            return buffer.GetBytes();
        }

        public void SendTo(Stream stream)
        {
            stream.Write(buffer.GetBytes(), 0, buffer.Length);
        }

        public override string ToString()
        {
            return $"{PacketCategory}, {PacketType}, params={Param0},{Param1},{Param2},{Param3} bodySize={BodySize}, version={Version}";
        }
    }
}
