﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Portability.h"
#include "CrossBar.h"
#include "ResetEvent.h"
#include "CallbackHandler.h"
#include "ConfigDb.h"
#include "BridgeEeprom.h"
#include "TicsUtil.h"

#include "ConsolePrintUtil.h"
#include "BridgeUtil.h"

using namespace tics;
using namespace TicsTest;

tics::portability::stl::string DEFAULT_BRIDGE_CONFIG_IP = "192.168.0.10";
tics::portability::stl::string DEFAULT_BRIDGE_CONFIG_PORT = "8000";

// Convert from "XX-XX-XX-XX-XX-XX" to [ XX, XX, XX, XX, XX, XX ].
bool TryParseMacAddr( const tics::portability::stl::string& str, uint8_t* arr, int arr_size );
// Convert from "XXXXXXXXXXXX" to "XX-XX-XX-XX-XX-XX".
tics::portability::stl::string FormatMacAddr( const tics::portability::stl::string& macAddr );

int Enumerate( int timeout = 3000 )
{
    if( BridgeUtil::Enumerate( timeout ) )
    {
        return 0;
    }
    else
    {
        return -1;
    }
}

int Find( const tics::portability::stl::string& name, int timeout = 3000 )
{
    tics::portability::stl::string ip;
    int port;
    if( BridgeUtil::Find( name, timeout, &ip, &port) )
    {
        PRINT_OK_MSG( "%s:%d", ip.c_str(), port);
        return 0;
    }
    else
    {
        PRINT_ERROR_MSG( "%s not found.", name.c_str() );
        return -1;
    }
}

int SetBridgeConfig( const tics::portability::stl::string& ipPort, const tics::portability::stl::string& attr, const tics::portability::stl::string& value )
{
    BridgeUtil bridgeUtil;

    if( !bridgeUtil.Takeover( ipPort ) )
    {
        return -1;
    }

    if( !bridgeUtil.SetBridgeConfig( ipPort, attr, value ) )
    {
        return -1;
    }
    return 0;
}

int GetBridgeConfig( const tics::portability::stl::string& ipPort )
{
    BridgeUtil bridgeUtil;

    if( !bridgeUtil.GetBridgeConfig( ipPort, NULL ) )
    {
        bridgeUtil.Release();
        return -1;
    }
    bridgeUtil.Release();
    return 0;
}

int UpdateBridgeFirmware( const tics::portability::stl::string& ipPort, const tics::portability::stl::string& fwpath )
{
    BridgeUtil bridgeUtil;

    if( !bridgeUtil.UpdateBridgeFirmware(ipPort, fwpath ))
    {
        return -1;
    }
    return 0;
}

int GetBridgeFirmwareVersion( const tics::portability::stl::string& ipPort )
{
    BridgeUtil bridgeUtil;

    if( !bridgeUtil.GetBridgeFirmwareVersion( ipPort, NULL ) )
    {
        bridgeUtil.Release();
        return -1;
    }
    bridgeUtil.Release();
    return 0;
}

int ControlTargetPower( const tics::portability::stl::string& ipPort, BridgeTargetPowerState state, bool force )
{
    BridgeUtil bridgeUtil;

    if( !bridgeUtil.ControlTargetPower( ipPort, state, force ) )
    {
        bridgeUtil.Release();
        return -1;
    }
    bridgeUtil.Release();
    return 0;
}

int ControlCradle( const tics::portability::stl::string& ipPort, bool attach )
{
    BridgeUtil bridgeUtil;

    if( !bridgeUtil.SetTargetStatus( ipPort, EFlag_TargetCradleAttached, attach ) )
    {
        bridgeUtil.Release();
        return -1;
    }
    bridgeUtil.Release();
    return 0;
}

int GetTargetStatus( const tics::portability::stl::string& ipPort )
{
    BridgeUtil bridgeUtil;

    if( !bridgeUtil.GetTargetStatus( ipPort ) )
    {
        bridgeUtil.Release();
        return -1;
    }
    bridgeUtil.Release();
    return 0;
}

enum RunMode
{
    RunMode_None,
    RunMode_Enumerate,
    RunMode_Find,
    RunMode_SetBridgeConfig,
    RunMode_GetBridgeConfig,
    RunMode_UpdateBridgeFirmware,
    RunMode_GetBridgeFirmwareVersion,
    RunMode_ControlTargetPower,
    RunMode_ControlCradle,
    RunMode_GetTargetStatus,
    RunMode_CreateEepromBin
};

void parse_args_common( int argc, char* argv[], RunMode* mode, char* ipAddr, int ipAddrSize, char* port, int portSize )
{
    for( int i = 1; i < argc; ++i )
    {
        if( tics::util::STRCASECMP( argv[i], "--Enumerate" ) == 0 )
        {
            *mode = RunMode_Enumerate;
        }
        else if( tics::util::STRCASECMP( argv[i], "--Find" ) == 0 )
        {
            *mode = RunMode_Find;
        }
        else if( tics::util::STRCASECMP( argv[i], "--SetBridgeConfig" ) == 0 )
        {
            *mode = RunMode_SetBridgeConfig;
        }
        else if( tics::util::STRCASECMP( argv[i], "--GetBridgeConfig" ) == 0 )
        {
            *mode = RunMode_GetBridgeConfig;
        }
        else if( tics::util::STRCASECMP( argv[i], "--UpdateBridgeFirmware" ) == 0 )
        {
            *mode = RunMode_UpdateBridgeFirmware;
        }
        else if( tics::util::STRCASECMP( argv[i], "--GetBridgeFirmwareVersion" ) == 0 )
        {
            *mode = RunMode_GetBridgeFirmwareVersion;
        }
        else if( tics::util::STRCASECMP( argv[i], "--ControlTargetPower" ) == 0 )
        {
            *mode = RunMode_ControlTargetPower;
        }
        else if( tics::util::STRCASECMP( argv[i], "--ControlCradle" ) == 0 )
        {
            *mode = RunMode_ControlCradle;
        }
        else if( tics::util::STRCASECMP( argv[i], "--GetTargetStatus" ) == 0 )
        {
            *mode = RunMode_GetTargetStatus;
        }
        else if( tics::util::STRCASECMP( argv[i], "--CreateEepromBin" ) == 0 )
        {
            *mode = RunMode_CreateEepromBin;
        }
        else if( tics::util::STRCASECMP( argv[i], "--ip" ) == 0 || tics::util::STRCASECMP( argv[i], "--ipaddr" ) == 0 )
        {
            strncpy( ipAddr, argv[i + 1], ipAddrSize );
            i++;
        }
        else if( tics::util::STRCASECMP( argv[i], "--port" ) == 0 )
        {
            strncpy( port, argv[i + 1], portSize );
            i++;
        }
    }
}

void parse_args_config( int argc, char* argv[], char* attr, int attrSize, char* value, int valueSize )
{
    for( int i = 1; i < argc; ++i )
    {
        if( tics::util::STRCASECMP( argv[i], "--attr" ) == 0 )
        {
            strncpy( attr, argv[i + 1], attrSize );
            i++;
        }
        else if( tics::util::STRCASECMP( argv[i], "--value") == 0 )
        {
            strncpy( value, argv[i + 1], valueSize );
            i++;
        }
    }
}

void parse_args_mac(  int argc, char* argv[], char* macAddr, int macAddrSize )
{
    for( int i = 1; i < argc; ++i )
    {
        if( tics::util::STRCASECMP( argv[i], "--macaddr" ) == 0 )
        {
            strncpy( macAddr, argv[i + 1], macAddrSize );
            i++;
        }
    }
}

void parse_args_fwpath( int argc, char* argv[], char* fwpath, int fwpathSize )
{
    for( int i = 1; i < argc; ++i )
    {
        if( tics::util::STRCASECMP( argv[i], "--fwpath" ) == 0 )
        {
            strncpy( fwpath, argv[i + 1], fwpathSize );
            i++;
        }
    }
}

void parse_args_power_command( int argc, char* argv[], BridgeTargetPowerState* powerstate, bool* force )
{
    *powerstate = TargetPowerState_Query;
    *force = false;
    for(int i = 1; i < argc; ++i)
    {
        if( tics::util::STRCASECMP( argv[i], "poweroff" ) == 0 )
        {
            *powerstate = TargetPowerState_Off;
        }
        else if( tics::util::STRCASECMP( argv[i], "poweron" ) == 0 )
        {
            *powerstate = TargetPowerState_On;
        }
        else if( tics::util::STRCASECMP( argv[i], "reset" ) == 0 )
        {
            *powerstate = TargetPowerState_Reset;
        }
        else if( tics::util::STRCASECMP( argv[i], "shutdown" ) == 0 )
        {
            *powerstate = TargetPowerState_Shutdown;
        }
        else if( tics::util::STRCASECMP( argv[i], "sleep" ) == 0 )
        {
            *powerstate = TargetPowerState_Sleep;
        }
        else if( tics::util::STRCASECMP( argv[i], "softpower" ) == 0 )
        {
            *powerstate = TargetPowerState_SoftShutdown;
        }
        else if( tics::util::STRCASECMP( argv[i], "forceoff" ) == 0 )
        {
            *powerstate = TargetPowerState_SoftShutdown;
            *force = true;
        }
    }
}

void parse_args_hw_version( int argc, char* argv[], uint8_t* hwVersion )
{
    for(int i = 1; i < argc; ++i)
    {
        if( tics::util::STRCASECMP( argv[i], "--HwVersion" ) == 0 )
        {
            *hwVersion = (uint8_t)atoi(argv[i + 1]);
            return;
        }
    }
}

void PrintUsage()
{
    printf("--Enumerate\n\tEnumerate HostBridges in the network.\n");
    printf("--Find <bridge_name>\n\tFind HostBridge with the specified name.\n");
    printf("--SetBridgeConfig --ip <bridge_ip> --attr <attr> --value <value>\n\tSet a BridgeConfig parameter.\n");
    printf("--GetBridgeConfig --ip <bridge_ip>\n\tGet BridgeConfig parameters.\n");
    printf("--UpdateBridgeFirmware --ip <bridge_ip> --fwpath <fw>\n\tUpdate HostBridge FW.\n");
    printf("--GetBridgeFirmwareVersion --ip <bridge_ip>\n\tGet HostBridge FW version.\n");
    printf("--ControlTargetPower --ip <bridge_ip> <poweroff|poweron|shutdown|sleep|softpower|forceoff>\n");
    printf("\tpoweroff  : Turn off main power supply to the Target.\n");
    printf("\tpoweron   : Turn on main power supply to the Target.\n");
    printf("\reset      : Send a reset pulse to the Target.\n");
    printf("\tshutdown  : Send a shutdown pulsepower to the Target.\n");
    printf("\tsleep     : Send a sleep/wakeup pulse to the Target    = Pressing the Power Button 30 ms.\n");
    printf("\tsoftpower : Send a power on/off pulse to the Target    = Pressing the Power Button 4s.\n");
    printf("\tforceoff  : Send a force power off pulse to the Target = Pressing the Power Button 13s.\n");
    // printf("--ControlCradle --ip <bridge_ip> <0|1>\n\t0 to detach, 1 to attach.\n");
    // printf("--GetTargetStatus --ip <bridge_ip>\n\tGet the current state of the Target.\n");
}

void CheckIpAddr(char* ip)
{
    if( ip == NULL || strlen(ip) < 1 )
    {
        PRINT_ERROR_MSG("IP Address Required! Use '-ipaddr <bridge_ip>' option.\n");
        exit(-1);
    }
}

int main(int argc, char* argv[])
{
    RunMode runMode = RunMode_None;
    char ipAddr[32];
    char port[32];
    memset(ipAddr, 0, 32);
    memset(port, 0, 32);

    parse_args_common( argc, argv, &runMode, ipAddr, 32, port, 32 );
    tics::portability::stl::string ipPort = ipAddr;
    if( strlen(port) != 0 )
    {
        ipPort += ":" + tics::portability::stl::string(port);
    }

    switch( runMode )
    {
    case RunMode_Enumerate:
        return Enumerate();;
    case RunMode_Find:
    {
        return Find( argv[argc - 1] );
    }
    case RunMode_SetBridgeConfig:
    {
        CheckIpAddr(ipAddr);
        char attr[32];
        char value[32];
        memset( attr, 0, 32 );
        memset( value, 0, 32 );
        parse_args_config( argc, argv, attr, 32, value, 32 );
        return SetBridgeConfig( ipPort, attr, value );
    }
    case RunMode_GetBridgeConfig:
    {
        CheckIpAddr(ipAddr);
        return GetBridgeConfig( ipPort );
    }
    case RunMode_UpdateBridgeFirmware:
    {
        CheckIpAddr(ipAddr);
        char fwpath[256];
        memset( fwpath,0, 256 );
        parse_args_fwpath( argc, argv, fwpath, 256 );
        return UpdateBridgeFirmware( ipPort, fwpath );
    }
    case RunMode_GetBridgeFirmwareVersion:
    {
        CheckIpAddr(ipAddr);
        return GetBridgeFirmwareVersion( ipPort );
    }
    case RunMode_ControlTargetPower:
    {
        CheckIpAddr(ipAddr);
        BridgeTargetPowerState powerState;
        bool force;
        parse_args_power_command(argc, argv, &powerState, &force);
        if(powerState == TargetPowerState_Query)
        {
            printf("No valid command found.\n");
            return -1;
        }
        return ControlTargetPower( ipPort, powerState, force );
    }
    case RunMode_ControlCradle:
    {
        CheckIpAddr(ipAddr);
        if(argc != 5)
        {
            printf("The number of arguments for ControlCradle must be 5 (*.exe --ControlCradle --ip <bridge_ip> <0|1>.\n");
            return -1;
        }
        bool attached;
        if( tics::util::STRCASECMP( argv[4], "0" ) == 0 )
        {
            attached = false;
        }
        else if( tics::util::STRCASECMP( argv[4], "1" ) == 0 )
        {
            attached = true;
        }
        else
        {
            printf("The fifth argument must be 0 or 1.\n");
            return -1;
        }
        return ControlCradle( ipPort, attached );
    }
    case RunMode_GetTargetStatus:
    {
        CheckIpAddr(ipAddr);
        return GetTargetStatus( ipPort );
    }
    case RunMode_CreateEepromBin:
    {
        char macAddr[32];
        uint8_t hwVersion = 1; // 1 is for X7(SDEV1.5)
        memset( macAddr, 0, 32);
        parse_args_mac( argc, argv, macAddr, 32 );
        parse_args_hw_version( argc, argv, &hwVersion);
        tics::portability::stl::string formattedMacAddr = FormatMacAddr(macAddr);
        if( formattedMacAddr.size() == 0)
        {
            PRINT_ERROR_MSG("Invalid Mac Address : %s\n", macAddr );
            return -1;
        }
        uint8_t macAddrArr[6];
        TryParseMacAddr( formattedMacAddr, macAddrArr, 6);

        BridgeEeprom::Load();
        BridgeEeprom::SetMacAddr(macAddrArr);
        BridgeEeprom::SetHwVersion( hwVersion );
        BridgeEeprom::Initialize();
        if( BridgeEeprom::Store() == BridgeEeprom::Result_Success )
        {
            PRINT_OK_MSG("Succeeded to create EEPROM.bin\n");
            return 0;
        }
        else
        {
            PRINT_ERROR_MSG("Failed to create EEPROM.bin\n");
            return -1;
        }
    }
    default:
        PRINT_ERROR_MSG("Please select mode.\n");
        PrintUsage();
        return -1;
    }
} // NOLINT(impl/function_size)

bool TryParseMacAddr( const tics::portability::stl::string& str, uint8_t* arr, int arr_size )
{
    tics::portability::stl::vector< tics::portability::stl::string > vec;
    tics::portability::algorithm::split(vec, str, "-:");

    if(vec.size() != 6)
    {
        return false;
    }

    int count = 0;
    for( tics::portability::stl::vector< tics::portability::stl::string >::const_iterator it = vec.begin(); it != vec.end(); it++ )
    {
        char* endptr = NULL;
        unsigned long value = strtoul( it->c_str(), &endptr, 16 );
        if( (*endptr) != '\0' || value > 0xFF )
        {
            return false;
        }

        if( count < arr_size )
        {
            arr[count] = static_cast<uint8_t>( value );
        }
        count++;
    }
    return true;
}

tics::portability::stl::string FormatMacAddr( const tics::portability::stl::string& macAddr )
{
    for( unsigned int i = 0; i < macAddr.length(); i++ )
    {
        if (!( (macAddr[i] >= '0' && macAddr[i] <= '9')
            || (macAddr[i] >= 'a' && macAddr[i] <= 'f')
            || (macAddr[i] >= 'A' && macAddr[i] <= 'F')
            || ( (i % 3) == 2 && macAddr[i] == '-' ) ))
        {
            return "";
        }
    }

    if( macAddr.length() == 12 )
    {
        tics::portability::stl::string mac = macAddr.substr(0, 2) + "-" + macAddr.substr(2, 2) + "-" + macAddr.substr(4, 2) + "-"
            + macAddr.substr(6, 2) + "-" + macAddr.substr(8, 2) + "-" + macAddr.substr(10, 2);
        std::transform( mac.begin(), mac.end(), mac.begin(), ::toupper );
        return mac;

    }
    else if( macAddr.length() == 17 )
    {
        return macAddr;
    }
    return "";
}
