﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Threading.Tasks;
using System.Diagnostics;
using System.ComponentModel;
using YamlDotNet;
using YamlDotNet.Serialization;
using YamlDotNet.Serialization.NamingConventions;

namespace Nintendo.FsAccessLogAnalysis
{
    public interface IFsAccessLogParserListener
    {
        void SetUp(FsAccessLogAnalyzer analyzer);
        void OnParseLine(string line, FsAccessLog log);
        void TearDown();
    }

    public enum FsApiType
    {
        Misc,
        Mount,
        Access,
    }

    [Flags]
    public enum FsApiAccessType
    {
        None = 0,
        Read = 1 << 0,
        Write = 1 << 1,
    }

    public class HexUInt32
    {
        public uint Value { get; set; }
        public static implicit operator uint(HexUInt32 v)
        {
            return v.Value;
        }

        public static implicit operator HexUInt32(uint v)
        {
            HexUInt32 h = new HexUInt32();
            h.Value = v;
            return h;
        }
        public static explicit operator HexUInt32(string v)
        {
            try
            {
                HexUInt32 h = new HexUInt32();
                h.Value = Convert.ToUInt32(v, 16);
                return h;
            }
            catch
            {
                throw;
            }
        }
        public override string ToString()
        {
            return "0x" + Value.ToString("X");
        }
    }
    public class HexUInt64
    {
        public ulong Value { get; set; }
        public static implicit operator ulong(HexUInt64 v)
        {
            return v.Value;
        }

        public static implicit operator HexUInt64(ulong v)
        {
            HexUInt64 h = new HexUInt64();
            h.Value = v;
            return h;
        }
        public static explicit operator HexUInt64(string v)
        {
            try
            {
                HexUInt64 h = new HexUInt64();
                h.Value = Convert.ToUInt64(v, 16);
                return h;
            }
            catch
            {
                throw;
            }
        }
        public override string ToString()
        {
            return "0x" + Value.ToString("X");
        }
    }

    public class OptionalUInt64
    {
        public ulong Value { get; set; }
        public bool Invalid { get; set; } = false;

        public static implicit operator ulong(OptionalUInt64 v)
        {
            return v.Value;
        }

        public static implicit operator OptionalUInt64(ulong v)
        {
            OptionalUInt64 h = new OptionalUInt64();
            h.Value = v;
            return h;
        }
        public static explicit operator OptionalUInt64(string v)
        {
            try
            {
                OptionalUInt64 h = new OptionalUInt64();
                ulong value;
                if (ulong.TryParse(v, out value))
                {
                    h.Value = value;
                }
                else
                {
                    h.Value = (ulong)long.Parse(v);
                    h.Invalid = true;
                }
                return h;
            }
            catch
            {
                throw;
            }
        }

        public override string ToString()
        {
            return Value.ToString();
        }
    }

    public class UserId : IEquatable<UserId>, IComparable<UserId>, IComparable
    {
        public UserId()
        {
            Upper = 0;
            Lower = 0;
        }
        public UserId(ulong upper, ulong lower)
        {
            Upper = upper;
            Lower = lower;
        }
        public static UserId InvalidUserId()
        {
            return new UserId();
        }
        public static implicit operator UserId(ulong v)
        {
            UserId h = new UserId();
            h.Lower = v;
            return h;
        }
        public static explicit operator UserId(string v)
        {
            try
            {
                UserId h = new UserId();
                string normalized = v.ToUpper();
                if (normalized.StartsWith("0X"))
                {
                    normalized = normalized.Remove(0, 2);
                }
                string str128 = normalized.PadLeft(32, '0');
                string upper = str128.Substring(0, 16);
                string lower = str128.Substring(16);
                h.Upper = Convert.ToUInt64(upper, 16);
                h.Lower = Convert.ToUInt64(lower, 16);
                return h;
            }
            catch
            {
                throw;
            }
        }
        public int CompareTo(object obj)
        {
            return CompareTo((UserId)obj);
        }
        public int CompareTo(UserId other)
        {
            int result = Upper.CompareTo(other.Upper);
            if (result == 0)
            {
                result = Lower.CompareTo(other.Lower);
            }
            return result;
        }
        public override bool Equals(object obj)
        {
            return Equals((UserId)obj);
        }
        public bool Equals(UserId other)
        {
            return (Upper == other.Upper) && (Lower == other.Lower);
        }

        public override string ToString()
        {
            return string.Format("{0:X16}{1:X16}", Upper, Lower);
        }
        public override int GetHashCode()
        {
            return ToString().GetHashCode();
        }

        public ulong Upper { get; set; }
        public ulong Lower { get; set; }
    }

    public class EnumId<Type> : IEquatable<EnumId<Type>>
        where Type : struct
    {
        public EnumId()
        {
        }

        public EnumId(Type id)
        {
            Id = id;
            Name = id.ToString();
        }

        public static implicit operator EnumId<Type>(Type value)
        {
            EnumId<Type> id = new EnumId<Type>(value);
            return id;
        }

        public static explicit operator EnumId<Type>(string value)
        {
            EnumId<Type> id = new EnumId<Type>();
            Type result;
            if (Enum.TryParse<Type>(value, true, out result))
            {
                id.Id = result;
                id.Name = result.ToString();
            }
            else
            {
                id.Name = value;
                id.Id = (Type)Enum.Parse(typeof(Type), value.GetHashCode().ToString());
            }
            return id;
        }

        public override bool Equals(object obj)
        {
            return Equals((EnumId<Type>)obj);
        }
        public bool Equals(EnumId<Type> other)
        {
            return Id.Equals(other.Id);
        }

        public Type Id { get; set; }
        public string Name { get; set; } = null;
        public override string ToString()
        {
            return Name ?? Id.ToString();
        }
        public override int GetHashCode()
        {
            return ToString().GetHashCode();
        }
    }

    public class FsAccessLog : IComparable<FsAccessLog>
    {
        public FsAccessLog()
        {
            Type = FsApiType.Misc;
            MountTarget = FsMountTarget.Unknown;
            Offset = 0;
            Size = 0;
            Result = 0;
            Handle = 0;
            StartTag = false;
            EndTag = false;
            FetchExceeded = false;
        }

        public class FetchedLog
        {
            public OptionalUInt64 Offset { get; set; }
            public OptionalUInt64 Size { get; set; }
        }

        // アクセスログの情報
        public long Start { get; set; }
        public long End { get; set; }
        public HexUInt32 Result { get; set; }
        public HexUInt64 Handle { get; set; }
        public string Priority { get; set; }
        public string Function { get; set; }
        public string ClassName { get; set; }
        public string NameSpace { get; set; }

        public OptionalUInt64 Offset { get; set; }
        public OptionalUInt64 Size { get; set; }
        public string Path { get; set; }
        public string NewPath { get; set; }
        public HexUInt32 OpenMode { get; set; }
        public FetchedLog[] Fetched { get; set; }
        public bool? FetchExceeded { get; set; }
        public string Name { get; set; }
        public int? Index { get; set; }
        [YamlAlias("userid")]
        public UserId UserId { get; set; }
        [YamlAlias("dataid")]
        public HexUInt64 DataId { get; set; }
        [YamlAlias("systemdataid")]
        public HexUInt64 SystemDataId { get; set; }
        [YamlAlias("applicationid")]
        public HexUInt64 ApplicationId { get; set; }
        [YamlAlias("bispartitionid")]
        public EnumId<FsFunction.BisPartitionId> BisPartitionId { get; set; }
        [YamlAlias("imagedirectoryid")]
        public EnumId<FsFunction.ImageDirectoryId> ImageDirectoryId { get; set; }
        [YamlAlias("cloudbackupworkstorageid")]
        public EnumId<FsFunction.CloudBackupWorkStorageId> CloudBackupWorkStorageId { get; set; }
        [YamlAlias("savedataid")]
        public HexUInt64 SaveDataId { get; set; }
        [YamlAlias("savedataspaceid")]
        public EnumId<FsFunction.SaveDataSpaceId> SaveDataSpaceId { get; set; }
        [YamlAlias("programid")]
        public HexUInt64 ProgramId { get; set; }
        [YamlAlias("contentstorageid")]
        public EnumId<FsFunction.ContentStorageId> ContentStorageId { get; set; }
        public EnumId<FsFunction.ContentType> ContentType { get; set; }
        public HexUInt64 FileHandle { get; set; }
        public EnumId<FsFunction.FileSystemProxyType> ProxyType { get; set; }
        public HexUInt32 GamecardHandle { get; set; }
        public EnumId<FsFunction.GameCardPartition> GamecardPartition { get; set; }

        public FsFunction.WriteOptionFlag WriteOption { get; set; }
        public string SdkVersion { get; set; }
        public string Spec { get; set; }
        public bool ForSystem { get; set; }
        public bool StartTag { get; set; }
        public bool EndTag { get; set; }

        public long? SaveDataFlags { get; set; }
        public HexUInt64 SaveDataOwnerId { get; set; }
        public long? SaveDataSize { get; set; }
        public long? SaveDataJournalSize { get; set; }
        public long? SaveDataTimeStamp { get; set; }
        public HexUInt64 SaveDataCommitId { get; set; }

        public long? CachestoragelistHandle { get; set; }
        public long? Infobuffercount { get; set; }

        public long? FileSize { get; set; }
        public long? BufferSize { get; set; }

        // nn::nfp
        public int? ModelType { get; set; }
        public int? NfpMountTarget { get; set; }
        public long? AccessId { get; set; }
        public int? DataSize { get; set; }

        // nn::album
        public ulong? ImageDataSize { get; set; }
        public int? Width { get; set; }
        public int? Height { get; set; }
        public int? ImageOrientation { get; set; }
        public HexUInt32 AlbumReportOption { get; set; }

        // 解析ツール側で設定する項目
        public FsApiType Type { get; set; }
        public FsApiAccessType AccessType { get; set; }
        public FsMountTarget MountTarget { get; set; }
        public bool MountForSystem { get; set; }
        public int MountHash { get; set; }
        public string ParserError { get; set; }
        public string AnalyzerError { get; set; }
        public string OutputStateError { get; set; }
        public string AccessInfoName { get; set; }

        public string GetFunctionFullName()
        {
            if (ClassName != null)
            {
                return ClassName + "::" + Function;
            }
            if (NameSpace != null)
            {
                return NameSpace + "::" + Function;
            }
            return Function;
        }

        public string GetAccessInfoName()
        {
            return AccessInfoName ?? GetFunctionFullName();
        }

        public MillisecondsPeriod Period
        {
            get
            {
                return new MillisecondsPeriod(Start, End);
            }
        }

        public int WriteCount
        {
            get
            {
                if (IsResultSuccess() || IsNfpAccess())
                {
                    FsFunction.GroupAccess groupAccess;
                    if (FsFunction.TryGetWriteGroupAccess(this, out groupAccess))
                    {
                        return groupAccess.WriteCount;
                    }
                    int count = AccessType.HasFlag(FsApiAccessType.Write) ? 1 : 0;
                    if (WriteOption.HasFlag(FsFunction.WriteOptionFlag.Flush))
                    {
                        ++count;
                    }
                    return count;
                }
                return 0;
            }
        }

        public ulong ActuallyWrittenSize
        {
            get
            {
                if (IsResultSuccess() || IsNfpAccess())
                {
                    FsFunction.GroupAccess groupAccess;
                    if (FsFunction.TryGetWriteGroupAccess(this, out groupAccess))
                    {
                        return groupAccess.WriteSize;
                    }
                    if (FsFunction.IsWriteSizeSameAsWritten(GetFunctionFullName()) && !Size.Invalid)
                    {
                        return Size;
                    }
                }
                return 0;
            }
        }

        public ulong ActuallyReadedSize
        {
            get
            {
                if (IsResultSuccess() || IsNfpAccess())
                {
                    FsFunction.GroupAccess groupAccess;
                    if (FsFunction.TryGetReadGroupAccess(this, out groupAccess))
                    {
                        return groupAccess.ReadSize;
                    }
                    if (AccessType.HasFlag(FsApiAccessType.Read) && !Size.Invalid)
                    {
                        return Size;
                    }
                }
                return 0;
            }
        }

        public bool IsResultSuccess()
        {
            return Result == 0;
        }
        public bool IsNfpAccess()
        {
            return NameSpace == "nfp";
        }

        public bool HasLogError()
        {
            return (ParserError != null) || (AnalyzerError != null) || (OutputStateError != null);
        }

        public bool IsAnalyzable()
        {
            return (Function != null) && (!HasLogError());
        }

        public bool IsStartLog()
        {
            return (SdkVersion != null) && (Spec != null);
        }

        public bool IsOutputStateLog()
        {
            return StartTag || EndTag;
        }

        public string GetMountArgumentsString()
        {
            return FsMountUtility.GetMountArgumentsString(this);
        }

        public int CompareTo(FsAccessLog other)
        {
            long diff = Start - other.Start;
            if (diff > 0)
            {
                return 1;
            }
            if (diff < 0)
            {
                return -1;
            }
            return 0;
        }

        public override int GetHashCode()
        {
            string path = (Path != null) ? FsMountUtility.RemoveMountName(Path) : string.Empty;
            // ファイルの同一性検証に利用します
            // ファイルパス + マウントターゲット + MountHash + Index + userId
            return (path + MountTarget + MountHash + (Index ?? 0) + UserId?.ToString() + AccessInfoName).GetHashCode();
        }
        public FsAccessLog GetWriteElement()
        {
            FsAccessLog newLog = (FsAccessLog)MemberwiseClone();
            newLog.AccessType = FsApiAccessType.Write;
            return newLog;
        }
        public FsAccessLog GetReadElement()
        {
            FsAccessLog newLog = (FsAccessLog)MemberwiseClone();
            newLog.AccessType = FsApiAccessType.Read;
            return newLog;
        }

        public FsAccessLog CreateFetchedLog(int fetchedIndex)
        {
            FsAccessLog newReadLog = (FsAccessLog)MemberwiseClone();
            newReadLog.Offset = Fetched[fetchedIndex].Offset;
            newReadLog.Size = Fetched[fetchedIndex].Size;
            newReadLog.Fetched = null;
            return newReadLog;
        }

        internal bool IsFetchedAscendingAndNoOverlap()
        {
            if (Fetched.Length == 0)
            {
                return true;
            }
            OptionalUInt64 nextMinOffset = Fetched[0].Offset + Fetched[0].Size;
            for (int i = 1; i < Fetched.Length; ++i)
            {
                if (Fetched[i].Offset < nextMinOffset)
                {
                    return false;
                }
                nextMinOffset = Fetched[i].Offset + Fetched[i].Size;
            }
            return true;
        }

        internal bool IsFetchedValueOutOfRange()
        {
            foreach (FetchedLog log in Fetched)
            {
                if (log.Offset < 0 || log.Size < 0
                    || log.Offset.Invalid || log.Size.Invalid
                    || (log.Offset > log.Offset + log.Size))
                {
                    return true;
                }
            }
            return false;
        }

        internal bool IsAllFetchedLogValid()
        {
            foreach (var fetched in Fetched)
            {
                if (fetched.Offset == null || fetched.Size == null)
                {
                    return false;
                }
            }
            return true;
        }

        internal bool IsFetchExceedValid()
        {
            if (FetchExceeded == null)
            {
                return false;
            }
            else if ((bool)FetchExceeded)
            {
                return Fetched != null;
            }
            return true;
        }

        internal bool IsCacheableLog()
        {
            return Function == "ReadFile";
        }
    }

    public class FsAccessLogParser
    {
        private const int MaxFetchedLogCount = 8;

        public IFsAccessLogParserListener Listener { get; set; }

        public static FsAccessLogParser Open(string filepath)
        {
            if (filepath.Length == 0)
            {
                return null;
            }
            FileStream stream = new FileStream(filepath, FileMode.Open, FileAccess.Read, FileShare.Read);
            return new FsAccessLogParser(stream);
        }

        public FsAccessLogParser(Stream stream)
        {
            Stream = stream;
        }

        private enum OutputState
        {
            NO_TAG,
            START_TAG,
            END_TAG
        }

        public List<FsAccessLogAnalyzer> Parse()
        {
            if (Stream == null)
            {
                return null;
            }

            List<FsAccessLogAnalyzer> list = new List<FsAccessLogAnalyzer>();
            FsAccessLogAnalyzer analyzer = new FsAccessLogAnalyzer();
            Listener?.SetUp(analyzer);
            using (FsAccessLogStreamSorter sorter = new FsAccessLogStreamSorter(Stream, Path.GetTempFileName()))
            {
                using (StreamReader sr = sorter.GetReader())
                {
                    OutputState lastTag = OutputState.NO_TAG;
                    bool isFirst = true;
                    while (sr.Peek() > 0)
                    {
                        string line = sr.ReadLine();
                        FsAccessLog log = null;
                        int markerIndex = line.IndexOf(FsLogMarker);
                        if (markerIndex >= 0)
                        {
                            log = Deserialize(line.Substring(markerIndex));
                            if (log != null)
                            {
                                if (log.StartTag)
                                {
                                    if (lastTag == OutputState.START_TAG)
                                    {
                                        // start_tag が連続している
                                        log.OutputStateError = "Access log is incomplete.";
                                        if (analyzer != null && !analyzer.IsEmpty())
                                        {
                                            analyzer.IsCompleteLog = false;
                                        }
                                    }
                                    lastTag = OutputState.START_TAG;
                                }
                                else if (log.EndTag)
                                {
                                    if (lastTag != OutputState.START_TAG)
                                    {
                                        // start_tag が無い or end_tag が連続している
                                        // 意図的にログを改変しないとこの状態にならない想定
                                        log.OutputStateError = "Invalid data. \"FS_ACCESS: { start_tag: true }\" is not found.";
                                    }
                                    lastTag = OutputState.END_TAG;
                                }
                                else if (log.IsStartLog() && !log.HasLogError())
                                {
                                    if (!isFirst || !analyzer.IsEmpty())
                                    {
                                        // ログの先頭以外で開始ログを発見した場合、analyzer を再生成
                                        Listener?.TearDown();
                                        list.Add(analyzer);
                                        analyzer = new FsAccessLogAnalyzer();
                                        Listener?.SetUp(analyzer);
                                    }
                                    isFirst = false;
                                }
                                analyzer.RecordLog(log);
                            }
                        }
                        if (log != null
                            && (log.Fetched != null || log.FetchExceeded != false)
                            && !log.IsCacheableLog())
                        {
                        }
                        else if (log != null && !log.IsFetchExceedValid())
                        {
                            log.AnalyzerError = "Invalid data. unexpected 'fetch_exceeded' found.";
                            Listener?.OnParseLine(line, log);
                        }
                        else if (log != null
                            && log.Fetched != null)
                        {
                            if (!log.IsAllFetchedLogValid())
                            {
                                log.AnalyzerError = "Invalid data. unexpected log in 'fetched'.";
                                Listener?.OnParseLine(line, log);
                            }
                            else if (!log.IsFetchedAscendingAndNoOverlap()
                                || log.IsFetchedValueOutOfRange()
                                || log.Fetched.Length > MaxFetchedLogCount)
                            {
                                log.AnalyzerError = "Invalid data. unexpected data in 'fetched'.";
                                Listener?.OnParseLine(line, log);
                            }
                            else if (log.FetchExceeded != null && (bool)log.FetchExceeded)
                            {
                                if (log.Fetched.Length != MaxFetchedLogCount)
                                {
                                    log.AnalyzerError = "Invalid data. unexpected count of 'fetched'.";
                                }
                                else
                                {
                                    // Exceeded のときは、Fetched の間全てにアクセスしたと判定する
                                    // この時点で、Fetched が昇順且つ重複無し且つ MaxFetchedLogCount 個である事が保証されている
                                    OptionalUInt64 newOffset = log.Fetched[0].Offset;
                                    OptionalUInt64 newSize = log.Fetched[MaxFetchedLogCount - 1].Offset + log.Fetched[MaxFetchedLogCount - 1].Size - newOffset;
                                    log.Offset = newOffset;
                                    log.Size = newSize;
                                }
                                Listener?.OnParseLine(line, log);
                            }
                            else if (log.Fetched.Length == 0)
                            {
                                log.AccessType = FsApiAccessType.None;
                                Listener?.OnParseLine(line, log);
                            }
                            else
                            {
                                for (int i = 0; i < log.Fetched.Length; ++i)
                                {
                                    FsAccessLog newReadLog = (FsAccessLog)log.CreateFetchedLog(i);
                                    Listener?.OnParseLine(i == 0 ? line : string.Empty, newReadLog);
                                }
                            }
                        }
                        else if (log != null
                         && log.IsNfpAccess()
                         && log.AccessType.HasFlag(FsApiAccessType.Write | FsApiAccessType.Read))
                        {
                            FsAccessLog writeLog = log.GetWriteElement();
                            FsAccessLog readLog = log.GetReadElement();
                            writeLog.MountTarget = FsMountTarget.Nand;
                            readLog.MountTarget = FsMountTarget.Nfp;
                            Listener?.OnParseLine(line, writeLog);
                            Listener?.OnParseLine(string.Empty, readLog);
                        }
                        else
                        {
                            Listener?.OnParseLine(line, log);
                        }
                        if ((log != null) && log.HasLogError())
                        {
                            analyzer.ErrorLogList.Add(log);
                        }
                    }
                    if (lastTag == OutputState.START_TAG)
                    {
                        analyzer.IsCompleteLog = false;
                    }
                }
            }
            Listener?.TearDown();
            list.Add(analyzer);
            return list;
        }

        private class FsAccessLogTop
        {
            [YamlAlias("FS_ACCESS")]
            public FsAccessLog FsAccess { get; set; } = new FsAccessLog();
        }

        private FsAccessLog Deserialize(string log)
        {
            StringReader stream = new StringReader(log.Replace('\\', '/'));
            return Deserialize(stream);
        }

        private FsAccessLog Deserialize(StringReader stream)
        {
            try
            {
                return YamlDeserializer.Deserialize<FsAccessLogTop>(stream).FsAccess;
            }
            catch (YamlDotNet.Core.YamlException e)
            {
                Debug.WriteLine(e.Message);
                Debug.WriteLine(e.StackTrace);
                FsAccessLog log = new FsAccessLog();
                string endMarkString = e.End.ToString() + "): ";
                int messageIndex = e.Message.LastIndexOf(endMarkString);
                log.ParserError = e.Message.Substring(messageIndex + endMarkString.Length);
                return log;
            }
            catch (Exception e)
            {
                Debug.WriteLine(e.Message);
                Debug.WriteLine(e.StackTrace);
                FsAccessLog log = new FsAccessLog();
                log.ParserError = e.Message;
                return log;
            }
        }

        internal static readonly string FsLogMarker = "FS_ACCESS: {";
        private Stream Stream = null;
        // YamlDotNet 3.1.0 未満ではマッチしなかったキーは無視できない
        // ignoreUnmatched:true
        private Deserializer YamlDeserializer = new Deserializer(namingConvention: new UnderscoredNamingConvention());
    }
}
