﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Text.RegularExpressions;
using System.Collections;
using System.Runtime.InteropServices;

namespace ErrorMessageDatabaseConverter
{
    /// <summary>
    /// 入力用の XML ファイルのデータ構造を定義します。
    /// </summary>
    public class InputXmlDataModel
    {
        public class MetaInfo
        {
            [System.Xml.Serialization.XmlElement("ocean")]
            public string Ocean { get; set; }
        }

        /// <summary>
        /// エラーメッセージを含むXMLファイルそのもののデータ構造
        /// </summary>
        [System.Xml.Serialization.XmlRoot("error_list")]
        public class ErrorListElement
        {
            [System.Xml.Serialization.XmlElement("meta")]
            public MetaInfo MetaInfo { get; set; }

            [System.Xml.Serialization.XmlElement("error")]
            public List<ErrorElement> ErrorElements { get; set; }

            public ErrorListElement()
            {
                ErrorElements = new List<ErrorElement>();
            }
        }

        /// <summary>
        /// 各エラーメッセージに含まれる情報
        /// </summary>
        public class ErrorElement
        {
            [System.Xml.Serialization.XmlAttribute("id")]
            public string ErrorCodeText { get; set; }

            [System.Xml.Serialization.XmlElement("lang")]
            public List<LanguageElement> LanguageElements { get; set; }

            [System.Xml.Serialization.XmlElement("dialog_button_action")]
            public List<byte> DialogViewButtonActions { get; set; }

            [System.Xml.Serialization.XmlElement("fullviewer_button_action")]
            public List<byte> FullScreenViewButtonActions { get; set; }

            [System.Xml.Serialization.XmlElement("server_failure")]
            public bool ServerFailure { get; set; }

            [System.Xml.Serialization.XmlElement("server_code")]
            public string ServerCode { get; set; }

            [System.Xml.Serialization.XmlElement("attribute")]
            public UInt32 Attribute { get; set; }

            // 以下、変換元の XML に直接は存在しない値です。
            // 読み込み後・変換実行時にメッセージの内容から算出します。

            [System.Xml.Serialization.XmlIgnore]
            public SystemDataModel.ErrorCode ErrorCode
            {
                get
                {
                    if (IsReferenceOnly)
                    {
                        // TORIAEZU : 参照専用のエラー（"100-comm" のようなもの）はシステムデータ化する事はなく、
                        //            値が必要とされることもないので例外を返しておく。
                        throw new NotSupportedException($"Unable to get ErrorCode from {ErrorCodeText}.");
                    }
                    return new SystemDataModel.ErrorCode(ErrorCodeText);
                }
            }

            [System.Xml.Serialization.XmlIgnore]
            public SystemDataModel.ErrorCode DialogViewMessageReference { get; set; }

            [System.Xml.Serialization.XmlIgnore]
            public SystemDataModel.ErrorCode DialogViewButtonMessageReference { get; set; }

            [System.Xml.Serialization.XmlIgnore]
            public SystemDataModel.ErrorCode FullScreenViewMessageReference { get; set; }

            [System.Xml.Serialization.XmlIgnore]
            public SystemDataModel.ErrorCode FullScreenViewButtonMessageReference { get; set; }

            public bool IsReferenceOnly
            {
                get
                {
                    Match m = Regex.Match(ErrorCodeText, @"[a-zA-Z]");
                    return m.Success;
                }
            }

            /// <summary>
            /// メッセージの参照先を設定します
            /// </summary>
            /// <param name="propertyKey">参照先を設定するプロパティ</param>
            /// <param name="errorCode">参照先のエラーコード</param>
            public void SetLanguageElementReference(LanguageElement.MessageElementKey propertyKey, SystemDataModel.ErrorCode errorCode)
            {
                switch (propertyKey)
                {
                    case LanguageElement.MessageElementKey.DlgMsg:
                        DialogViewMessageReference = errorCode;
                        return;
                    case LanguageElement.MessageElementKey.DlgBtn:
                        DialogViewButtonMessageReference = errorCode;
                        return;
                    case LanguageElement.MessageElementKey.FlvMsg:
                        FullScreenViewMessageReference = errorCode;
                        return;
                    case LanguageElement.MessageElementKey.FlvBtn:
                        FullScreenViewButtonMessageReference = errorCode;
                        return;
                    default:
                        throw new ArgumentException("key must be selected from predefined values", nameof(propertyKey));
                }
            }

            /// <summary>
            /// 自身が持つメッセージのデータをコピーします
            /// </summary>
            /// <param name="dst">コピー先</param>
            /// <param name="propertyKey">コピーするプロパティ</param>
            public void CopyLanguageValuesTo(ErrorElement dst, LanguageElement.MessageElementKey propertyKey)
            {
                foreach (var srcLang in LanguageElements)
                {
                    var dstLang = dst.GetLanguageElement(srcLang.Language);
                    if (dstLang == null)
                    {
                        var l = LanguageElement.MakeEmptyElement();
                        l.Language = srcLang.Language;
                        l[propertyKey] = srcLang[propertyKey];
                        dst.LanguageElements.Add(l);
                    }
                    else
                    {
                        dstLang[propertyKey] = srcLang[propertyKey];
                    }
                }
            }

            public LanguageElement GetLanguageElement(Language lang)
            {
                return LanguageElements.Where(l => l.Language == lang).FirstOrDefault();
            }
        }

        /// <summary>
        /// 各エラーに含まれる情報で、言語毎に設定されるもの（テキストメッセージの類）
        /// </summary>
        public class LanguageElement
        {
            public static LanguageElement MakeEmptyElement()
            {
                LanguageElement l = new LanguageElement();
                l.DialogViewButtonMessage = new List<string> { string.Empty, string.Empty, string.Empty };
                l.FullScreenViewButtonMessage = new List<string> { string.Empty, string.Empty, string.Empty };
                return l;
            }

            public enum MessageElementKey
            {
                DlgMsg,
                DlgBtn,
                DlgBtn0,
                DlgBtn1,
                DlgBtn2,
                FlvMsg,
                FlvBtn,
                FlvBtn0,
                FlvBtn1,
                FlvBtn2,
            }

            [System.Xml.Serialization.XmlAttribute("name")]
            public Language Language { get; set; }

            [System.Xml.Serialization.XmlElement("dialog_message")]
            public string DialogViewMessage { get; set; }

            [System.Xml.Serialization.XmlElement("dialog_button_message")]
            public List<string> DialogViewButtonMessage { get; set; }

            [System.Xml.Serialization.XmlElement("fullviewer_message")]
            public string FullScreenViewMessage { get; set; }

            [System.Xml.Serialization.XmlElement("fullviewer_button_message")]
            public List<string> FullScreenViewButtonMessage { get; set; }

            public string this[MessageElementKey key]
            {
                get
                {
                    switch (key)
                    {
                        case MessageElementKey.DlgMsg:
                            return DialogViewMessage;
                        case MessageElementKey.DlgBtn:
                            // ボタンのテキストは改行を含まないことになっているので改行区切りでひとまとめに。
                            return string.Join("\n", DialogViewButtonMessage);
                        case MessageElementKey.DlgBtn0:
                            return (DialogViewButtonMessage.Count > 0) ? DialogViewButtonMessage[0] : "";
                        case MessageElementKey.DlgBtn1:
                            return (DialogViewButtonMessage.Count > 1) ? DialogViewButtonMessage[1] : "";
                        case MessageElementKey.DlgBtn2:
                            return (DialogViewButtonMessage.Count > 2) ? DialogViewButtonMessage[2] : "";
                        case MessageElementKey.FlvMsg:
                            return FullScreenViewMessage;
                        case MessageElementKey.FlvBtn:
                            return string.Join("\n", FullScreenViewButtonMessage);
                        case MessageElementKey.FlvBtn0:
                            return (FullScreenViewButtonMessage.Count > 0) ? FullScreenViewButtonMessage[0] : "";
                        case MessageElementKey.FlvBtn1:
                            return (FullScreenViewButtonMessage.Count > 1) ? FullScreenViewButtonMessage[1] : "";
                        case MessageElementKey.FlvBtn2:
                            return (FullScreenViewButtonMessage.Count > 2) ? FullScreenViewButtonMessage[2] : "";
                        default:
                            throw new ArgumentException("key must be selected from predefined values", nameof(key));
                    }
                }
                set
                {
                    switch (key)
                    {
                        case MessageElementKey.DlgMsg:
                            DialogViewMessage = value;
                            return;
                        case MessageElementKey.DlgBtn:
                            var dlgBtns = value.Split('\n');
                            DialogViewButtonMessage.Clear();
                            for (var i = 0; i < dlgBtns.Length; i++)
                            {
                                DialogViewButtonMessage.Add(dlgBtns[i]);
                            }
                            return;
                        case MessageElementKey.DlgBtn0:
                            DialogViewButtonMessage[0] = value;
                            return;
                        case MessageElementKey.DlgBtn1:
                            DialogViewButtonMessage[1] = value;
                            return;
                        case MessageElementKey.DlgBtn2:
                            DialogViewButtonMessage[2] = value;
                            return;
                        case MessageElementKey.FlvMsg:
                            FullScreenViewMessage = value;
                            return;
                        case MessageElementKey.FlvBtn:
                            var flvBtns = value.Split('\n');
                            FullScreenViewButtonMessage.Clear();
                            for (var i = 0; i < flvBtns.Length; i++)
                            {
                                FullScreenViewButtonMessage.Add(flvBtns[i]);
                            }
                            return;
                        case MessageElementKey.FlvBtn0:
                            FullScreenViewButtonMessage[0] = value;
                            return;
                        case MessageElementKey.FlvBtn1:
                            FullScreenViewButtonMessage[1] = value;
                            return;
                        case MessageElementKey.FlvBtn2:
                            FullScreenViewButtonMessage[2] = value;
                            return;
                        default:
                            throw new ArgumentException("key must be selected from predefined values", nameof(key));
                    }
                }
            }
        }

        /// <summary>
        /// メッセージ中の URL置換タグを置換するための URL情報のXMLファイルのデータ構造
        /// </summary>
        [System.Xml.Serialization.XmlRoot("url_list")]
        public class UrlListElement
        {
            [System.Xml.Serialization.XmlElement("url")]
            public List<UrlElement> UrlElements { get; set; }
        }

        public class UrlElement
        {
            [System.Xml.Serialization.XmlAttribute("id")]
            public string Id { get; set; }
            [System.Xml.Serialization.XmlElement("lang")]
            public List<LanguageElementInUrlElement> LanguageElements { get; set; }

            public class LanguageElementInUrlElement
            {
                [System.Xml.Serialization.XmlAttribute("name")]
                public Language Language { get; set; }
                [System.Xml.Serialization.XmlText]
                public string Url { get; set; }
            }
        }

        /// <summary>
        /// データベース自身の情報を指定するための XML ファイルのデータ構造
        /// </summary>
        [System.Xml.Serialization.XmlRoot("database_info")]
        public class DatabaseInfoElement
        {
            [System.Xml.Serialization.XmlElement("major_version")]
            public UInt16 MajorVersion { get; set; }
            [System.Xml.Serialization.XmlElement("minor_version")]
            public UInt16 MinorVersion { get; set; }
        }

        /// <summary>
        /// メッセージデータ
        /// </summary>
        [System.Xml.Serialization.XmlRoot("message_list")]
        public class MessageListElement
        {
            [System.Xml.Serialization.XmlElement("meta")]
            public MetaInfo MetaInfo { get; set; }

            [System.Xml.Serialization.XmlElement("message")]
            public List<MessageElement> MessageElements { get; set; }
        }

        public class MessageElement
        {
            [System.Xml.Serialization.XmlAttribute("id")]
            public string Id { get; set; }
            [System.Xml.Serialization.XmlElement("lang")]
            public List<LanguageElementInMessageElement> LanguageElements { get; set; }

            public class LanguageElementInMessageElement
            {
                [System.Xml.Serialization.XmlAttribute("name")]
                public Language Language { get; set; }
                [System.Xml.Serialization.XmlText]
                public string Message { get; set; }
            }
        }
    }
}
