﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using CommandUtility;
using MakeFirmwareArchive;
using Nintendo.Foundation.IO;
using System.Text.RegularExpressions;
using System.Security.Cryptography;

namespace DumpNandInfo
{
    public class DumpNandInfoArguments
    {
        [CommandLineOption('o', "output", Description = "output answer path", IsRequired = true)]
        public string OutputPath { get; set; }

        [CommandLineOption('i', "input", Description = "input nfa path", IsRequired = true)]
        public string InputNfaPath { get; set; }

        [CommandLineOption("keyconfig", Description = "keyconfig path", IsRequired = true)]
        public string KeyPath { get; set; }

        //要 ExFat が有効状態か設定するフラグ
        [CommandLineOption("exfat", Description = "is enabled exfat", DefaultValue = false, IsRequired = false)]
        public bool IsEnabledExFat { get; set; }
    }

    class Program
    {
        public static readonly Dictionary<string, string> directoryMapper = new Dictionary<string, string>()
        {
            {"BCPKG2-1-Normal-Main", "BootConfigAndPackage2Part1"},
            {"BCPKG2-2-Normal-Sub", "BootConfigAndPackage2Part2"},
            {"BCPKG2-3-SafeMode-Main", "BootConfigAndPackage2Part3"},
            {"BCPKG2-4-SafeMode-Sub", "BootConfigAndPackage2Part4"},
            {"BCPKG2-5-Repair-Main", "BootConfigAndPackage2Part5"},
            {"BCPKG2-6-Repair-Sub", "BootConfigAndPackage2Part6"},

            {"BootPartition1-Bct-1st", "BootPartition1Root"},
            {"BootPartition1-Bct-2nd", "BootPartition1Root"},
            {"BootPartition1-Bct-Normal", "BootPartition1Root"},
            {"BootPartition1-Bct-SafeMode", "BootPartition1Root"},
            {"BootPartition1-Normal-Package1-Main", "BootPartition1Root"},
            {"BootPartition1-Normal-Package1-Sub", "BootPartition1Root"},

            {"BootPartition2-SafeMode-Package1-Main", "BootPartition2Root"},
            {"BootPartition2-SafeMode-Package1-Sub", "BootPartition2Root"},

            {"PRODINFOF", "CalibrationFile"},
            {"SAFE", "SafeMode"},
            {"SYSTEM", "System"},
            {"USER", "User"},
        };

        static void Main(string[] args)
        {
            var parameters = new DumpNandInfoArguments();
            if (CommandLineParser.Default.ParseArgs(args, out parameters))
            {
                DumpNandInfo(new FileInfo(parameters.OutputPath),
                             new FileInfo(parameters.InputNfaPath),
                             new FileInfo(parameters.KeyPath),
                             parameters.IsEnabledExFat);
            }
            else
            {
                return;
            }
        }

        //対象外のディレクトリかどうか判定する(BootConfig, config)
        static bool IsExcludedDirectoryName(string directoryName)
        {
            return Regex.IsMatch(directoryName, ".*(BootConfig|config).*$");
        }

        //.cnmt.nca は対象外なので除外しつつ nca ファイルを見つける
        static bool IsNcaFile(string fileName)
        {
            return Regex.IsMatch(fileName, ".*(.nca)$");
        }

        static bool IsNspFile(string fileName)
        {
            return Regex.IsMatch(fileName, ".*(.nsp)$");
        }

        static bool IsBctFile(string fileName)
        {
            return Regex.IsMatch(fileName, ".*(bct)$");
        }

        static bool IsBootLoaderFile(string fileName)
        {
            return Regex.IsMatch(fileName, ".*(bl|package1)$");
        }

        static bool IsPackage2File(string fileName)
        {
            return Regex.IsMatch(fileName, ".*(package2).*");
        }

        static bool IsBinaryFile(string fileName)
        {
            return Regex.IsMatch(fileName, ".*(.bl|.bin|.img|.bct)$");
        }

        static bool IsCertificationFile(string fileName)
        {
            return Regex.IsMatch(fileName, ".*(.png)$");
        }

        static bool IsIgnoredDirectory(string dirName)
        {
            return Regex.IsMatch(dirName, ".*(Boot).*");
        }

        static bool IsSafeDirectory(string dirName)
        {
            return Regex.IsMatch(dirName, ".*(Safe|2nd).*");
        }

        static bool IsBootImagePackage(string fileName)
        {
            return Regex.IsMatch(fileName, ".*BootImagePackage-.*");
        }

        static bool IsBootImagePackageSafe(string fileName)
        {
            return Regex.IsMatch(fileName, ".*BootImagePackageSafe-.*");
        }

        static bool IsBootImagePackageExFat(string fileName)
        {
            return Regex.IsMatch(fileName, ".*BootImagePackageExFat-.*");
        }

        static bool IsBootImagePackageExFatSafe(string fileName)
        {
            return Regex.IsMatch(fileName, ".*BootImagePackageExFatSafe-.*");
        }

        //Bct の読み取り
        static byte[] ReadBct(FileStream fs)
        {
            int readSize = 4096;
            var buffer = new byte[readSize];
            fs.Seek(4096, SeekOrigin.Begin);
            fs.Read(buffer, 0, readSize);
            return buffer;
        }

        //BootLoader の読み取り
        static byte[] ReadBootLoader(FileStream fs)
        {
            int readSize = 128 * 1024;
            var buffer = new byte[readSize];

            fs.Read(buffer, 0, readSize);

            return buffer;
        }

        //Package2 の読み取り
        static byte[] ReadPackage2(FileStream fs)
        {
            int readSize = 1500 * 1024;
            var buffer = new byte[readSize];

            fs.Read(buffer, 0, readSize);

            return buffer;
        }

        static public string CalculateHash(FileInfo file)
        {
            var crypto = new SHA256CryptoServiceProvider();
            crypto.Initialize();

            byte[] buffer = null;
            using (var fs = new FileStream(file.FullName, FileMode.Open, FileAccess.Read))
            {
                if (IsBctFile(file.Name))
                {
                    buffer = ReadBct(fs);
                }
                else if (IsPackage2File(file.Name))
                {
                    buffer = ReadPackage2(fs);
                }
                else if(IsBootLoaderFile(file.Name))
                {
                    buffer = ReadBootLoader(fs);
                }

                if (buffer == null)
                {
                    buffer = crypto.ComputeHash(fs);
                }
                else
                {
                    buffer = crypto.ComputeHash(buffer);
                }
            }

            var hashText = new StringBuilder();

            //contentsId の計算方法と合わせる ( sha256 の先頭 128 bit )
            for (int i = 0; i < buffer.Length / 2; i++)
            {
                hashText.AppendFormat("{0:x2}", buffer[i]);
            }

            return hashText.ToString();
        }

        //特定のディレクトリを起点として以下のファイルを列挙する
        static List<FileInfo> EnumerateFilesRecursively(DirectoryInfo currentDirectory)
        {
            var list = new List<FileInfo>();

            foreach (var directory in currentDirectory.EnumerateDirectories())
            {
                list.AddRange(EnumerateFilesRecursively(directory));
            }

            list.AddRange(currentDirectory.EnumerateFiles());

            return list;
        }

        private static void DumpNandInfo(FileInfo outputPath, FileInfo inputNfa, FileInfo keyPath, bool IsEnabledExfat)
        {
            using (var tempHolder = new TemporaryFileHolder("DumpNandInfo"))
            {
                var extractedNfa = tempHolder.CreateTemporaryDirectory("ExtractedNfa");

                var hashList = new Dictionary<string, List<Tuple<string, string>>>();
                //最初にマッピング後のディレクトリ名を作成しておく
                foreach (var directoryName in directoryMapper)
                {
                    if (!hashList.ContainsKey(directoryName.Value))
                    {
                        hashList.Add(directoryName.Value, new List<Tuple<string, string>>());
                    }
                }

                using (var w = new StreamWriter(outputPath.FullName))
                {
                    bool bipSafeFound = false;
                    bool bipExfatFound = false;
                    bool bipExFatSafeFound = false;

                    var archiveDirectory = FirmwareArchiveDirectory.FromFirmwareArchive(extractedNfa, inputNfa);

                    foreach (var dir in archiveDirectory.Directory.EnumerateDirectories())
                    {
                        if (IsExcludedDirectoryName(dir.Name))
                        {
                            continue;
                        }

                        var allFiles = EnumerateFilesRecursively(dir);

                        foreach (var file in allFiles)
                        {
                            if (IsBootImagePackageExFatSafe(file.Name))
                            {
                                bipExFatSafeFound = true;
                            }

                            if (IsBootImagePackageExFat(file.Name))
                            {
                                bipExfatFound = true;
                            }

                            if (IsBootImagePackageSafe(file.Name))
                            {
                                bipSafeFound = true;
                            }
                        }
                    }

                    // Exfat 有効化を計算するにもかかわらず BIP-Exfat が無い場合はエラーとする
                    if (IsEnabledExfat && !bipExfatFound)
                    {
                        Console.WriteLine("[error] enabled --exfat option but BootImagePackageExfat is not found in nfa");
                        throw new Exception("");
                    }

                    // Exfat 有効化を計算するにもかかわらず BIP-Exfat が無い場合はエラーとする
                    if (IsEnabledExfat && !bipExFatSafeFound && bipSafeFound)
                    {
                        Console.WriteLine("[error] enabled --exfat option but BootImagePackageExfatSafe is not found in nfa");
                        throw new Exception("");
                    }

                    foreach (var dir in archiveDirectory.Directory.EnumerateDirectories())
                    {
                        if (IsExcludedDirectoryName(dir.Name))
                        {
                            continue;
                        }

                        //ここで Dirname を取っておくことにする
                        var dirName = dir.Name;
                        if (directoryMapper.ContainsKey(dirName))
                        {
                            dirName = directoryMapper[dirName];
                        }

                        var allFiles = EnumerateFilesRecursively(dir);

                        foreach (var file in allFiles)
                        {
                            var targetFiles = new List<FileInfo>();

                            //nsp の展開
                            if(IsNspFile(file.Name))
                            {
                                var tempDirectory = tempHolder.CreateTemporaryDirectory(file.Name);
                                ExtractNsp(tempDirectory, file);

                                var extractFiles = tempDirectory.EnumerateFiles();
                                //展開された Nsp から nca ファイルを探す
                                foreach (var extractFile in extractFiles)
                                {
                                    if (IsNcaFile(extractFile.Name))
                                    {
                                        //開発版の初期化イメージでは Exfat 有効化、非有効化版の両方が存在し
                                        //Exfat 無効の場合は、操作によって Target 上の BootImagePackageExfat が削除されているので、スキップする
                                        if (IsBootImagePackageExFat(file.Name) && !IsEnabledExfat)
                                        {
                                            continue;
                                        }

                                        if (IsBootImagePackageExFatSafe(file.Name) && !IsEnabledExfat)
                                        {
                                            continue;
                                        }

                                        targetFiles.Add(extractFile);

                                        //BootImagePackage なら展開して中身のハッシュ値を求める
                                        if ((IsBootImagePackage(file.Name) && !IsEnabledExfat) ||
                                            (IsBootImagePackageExFat(file.Name) && IsEnabledExfat) ||
                                             IsBootImagePackageSafe(file.Name) && !IsEnabledExfat ||
                                             IsBootImagePackageExFatSafe(file.Name) && IsEnabledExfat)
                                        {
                                            var temp2 = tempHolder.CreateTemporaryDirectory("BootImagePackage");
                                            ExtractBootImagePackage(temp2, file, keyPath);

                                            if (!Regex.IsMatch(extractFile.Name, ".*cnmt.*"))
                                            {
                                                string baseDirectory = temp2.FullName + "/" + extractFile.Name + "/fs0/nx/";
                                                string hashText = CalculateHash(new FileInfo(baseDirectory + "package2"));

                                                //BootImagePackageSafe の場合は 3,4 に格納される
                                                if (IsBootImagePackageSafe(file.Name) || IsBootImagePackageExFatSafe(file.Name))
                                                {
                                                    hashList["BootConfigAndPackage2Part3"].Add(new Tuple<string, string>(hashText, file.Name));
                                                    hashList["BootConfigAndPackage2Part4"].Add(new Tuple<string, string>(hashText, file.Name));
                                                }
                                                //BootImagePackage(Exfat) の場合は 1,2 に格納される
                                                else
                                                {
                                                    hashList["BootConfigAndPackage2Part1"].Add(new Tuple<string, string>(hashText, file.Name));
                                                    hashList["BootConfigAndPackage2Part2"].Add(new Tuple<string, string>(hashText, file.Name));
                                                }

                                                // Main と Sub の 2 つを格納する
                                                hashText = CalculateHash(new FileInfo(baseDirectory + "package1"));

                                                //BootImagePackageSafe の場合は 2 に格納される
                                                if (IsBootImagePackageSafe(file.Name) || IsBootImagePackageExFatSafe(file.Name))
                                                {
                                                    hashList["BootPartition2Root"].Add(new Tuple<string, string>(hashText, file.Name));
                                                    hashList["BootPartition2Root"].Add(new Tuple<string, string>(hashText, file.Name));
                                                }
                                                //BootImagePackage(Exfat) の場合は 1 に格納される
                                                else
                                                {
                                                    hashList["BootPartition1Root"].Add(new Tuple<string, string>(hashText, file.Name));
                                                    hashList["BootPartition1Root"].Add(new Tuple<string, string>(hashText, file.Name));
                                                }

                                                // Main と Sub の 2 つを格納する
                                                hashText = CalculateHash(new FileInfo(baseDirectory + "bct"));
                                                hashList["BootPartition1Root"].Add(new Tuple<string, string>(hashText, file.Name));
                                                hashList["BootPartition1Root"].Add(new Tuple<string, string>(hashText, file.Name));
                                            }
                                        }
                                    }
                                }
                            }
                            else
                            {
                                targetFiles.Add(file);
                            }

                            foreach (var targetFile in targetFiles)
                            {
                                if (IsNcaFile(targetFile.Name) || IsBinaryFile(targetFile.Name) || IsCertificationFile(targetFile.Name))
                                {
                                    //修理用ディレクトリか更新されるディレクトリ場合はハッシュ値を記録しない
                                    if (!IsIgnoredDirectory(dirName) ||
                                       ((IsSafeDirectory(dir.Name) && !bipSafeFound) &&
                                        (IsSafeDirectory(dir.Name) && !bipExFatSafeFound)))
                                    {
                                        hashList[dirName].Add(new Tuple<string, string>(CalculateHash(targetFile), file.Name));
                                    }
                                }
                            }
                        }
                    }

                    foreach (var directory in hashList)
                    {
                        var list = directory.Value;
                        w.WriteLine("{0} {1}", directory.Key, list.Count);

                        list.Sort();
                        foreach (var file in list)
                        {
                            w.WriteLine("{0} {1}", file.Item1, file.Item2);
                        }
                    }
                }
            }
        }

        private static void ExtractNsp(DirectoryInfo replacedNspDir, FileInfo systemNsp)
        {
            SdkTool.Execute(
               SdkPath.FindToolPath("AuthoringTool", "AuthoringTool/AuthoringTool.exe", "AuthoringTool/AuthoringTool.exe"),
                new string[] {
                    "extractnsp",
                    "-o", replacedNspDir.FullName,
                    systemNsp.FullName,
                }
            );

            Console.WriteLine(replacedNspDir.FullName);
        }

        private static void ExtractBootImagePackage(DirectoryInfo extractDir, FileInfo bootImagePackageNsp, FileInfo keyPath)
        {
            SdkTool.Execute(
                SdkPath.FindToolPath("AuthoringTool", "AuthoringTool/AuthoringTool.exe", "AuthoringTool/AuthoringTool.exe"),
                new string[] {
                    "extract",
                    "--keyconfig", keyPath.FullName,
                    "-o", extractDir.FullName,
                    bootImagePackageNsp.FullName,
                }
            );
        }
    }
}
