﻿
#define GET_DATA_FROM_SERVER

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.Net.Http;
using System.Collections;

using System.Diagnostics;

// for Json
using System.Runtime.Serialization;
using System.Runtime.Serialization.Json;
using System.IO;

namespace CrashReport
{
    class Program
    {
        [DataContract]
        private class CrashSummaryItem
        {
            [DataMember(Name = "application_version")]
            public string application_version { get; set; }
            [DataMember(Name = "crash_report_hash")]
            public string crash_report_hash { get; set; }
            [DataMember(Name = "error_code")]
            public string error_code { get; set; }
            [DataMember(Name = "count")]
            public ulong count { get; set; }
            [DataMember(Name = "serial_number_count")]
            public ulong serial_number_count { get; set; }
            [DataMember(Name = "exception_info")]
            public string exception_info { get; set; }
        }

        [DataContract]
        private class CrashDetailItem
        {
            [DataMember(Name = "application_version")]
            public string application_version { get; set; }
            [DataMember(Name = "crash_report_hash")]
            public string crash_report_hash { get; set; }
            [DataMember(Name = "error_code")]
            public string error_code { get; set; }
            [DataMember(Name = "occurrence_timestamp_net")]
            public string occurrence_timestamp_net { get; set; }
            [DataMember(Name = "os_version")]
            public string os_version { get; set; }
            [DataMember(Name = "exception_info")]
            public string exception_info { get; set; }
            [DataMember(Name = "dying_message")]
            public string dying_message { get; set; }
        }

        [DataContract]
        private class ErrorMessage
        {
            [DataMember(Name = "error_code")]
            public string error_code { get; set; }
            [DataMember(Name = "message")]
            public string message { get; set; }
        }

        static private List<CrashSummaryItem> m_CrashSummaryItems = new List<CrashSummaryItem>();
        static private List<CrashInformation> m_CrashInfo = new List<CrashInformation>();

        static private async Task<string> QueryCrashReportData(CommandLineOption option)
        {
#if (GET_DATA_FROM_SERVER)
            var client = new HttpClient();
            Hashtable ServerFqdn = new Hashtable
            {
                ["lp1"] = "crash-report.mng.nintendo.net",
                ["sp1"] = "sp1-crash-report.mng.nintendo.net",
                ["jd1"] = "jd1-crash-report.mng.nintendo.net",
                ["td1"] = "td1-crash-report.mng.nintendo.net",
                ["dd1"] = "dd1-crash-report.mng.nintendo.net"
            };

            // タイムアウトを設定
            client.Timeout = TimeSpan.FromSeconds(option.RequestTimeout);

            string fqdn = (string)ServerFqdn["lp1"];
            if (option.ServiceDiscovery != null)
            {
                fqdn = (string)ServerFqdn[option.ServiceDiscovery];
            }

            if (option.Fqdn != null)
            {
                fqdn = option.Fqdn;
            }

            string query;
            if (option.DetailMode)
            {
                query = string.Format("https://{0}/api/v1/titles/{1}/crash_reports/?access-token={2}", fqdn, option.ApplicationId, option.AccessToken);

                // 詳細取得時はサーバー側でクラッシュハッシュで絞り込む
                if (option.FilterHash != null)
                {
                    byte[] hash = new byte[option.FilterHash.Length / 2];
                    for (int i = 0; i < option.FilterHash.Length / 2; i++)
                    {
                        hash[i] = Convert.ToByte(option.FilterHash.Substring(i * 2, 2), 16);
                    }
                    query += string.Format("&crash_report_hash={0}", Uri.EscapeDataString(Convert.ToBase64String(hash)));
                }
            }
            else
            {
                query = string.Format("https://{0}/api/v1/titles/{1}/crash_reports/summary?access-token={2}", fqdn, option.ApplicationId, option.AccessToken);
            }

            if (option.ApplicationVersion != null)
            {
                query += string.Format("&application_version={0}", Uri.EscapeDataString(option.ApplicationVersion));
            }

            if (option.ReportOffset != 0)
            {
                query += string.Format("&offset={0}", option.ReportOffset);
            }

            if (option.FilterDateBegin != null)
            {
                query += string.Format("&start={0}", option.FilterDateBegin);
            }

            if (option.FilterDateEnd != null)
            {
                query += string.Format("&end={0}", option.FilterDateEnd);
            }

            try
            {
                var response = await client.GetAsync(query);
                string result = await response.Content.ReadAsStringAsync();

                if (response.StatusCode != System.Net.HttpStatusCode.OK)
                {
                    //Console.WriteLine(query);
                    //Console.WriteLine(response);
                    //Console.WriteLine(result);

                    var serializer = new DataContractJsonSerializer(typeof(ErrorMessage));
                    Stream stream = new MemoryStream(Encoding.Unicode.GetBytes(result));
                    try
                    {
                        var e = (ErrorMessage)serializer.ReadObject(stream);
                        Console.WriteLine("{0} {1}", e.error_code, e.message);
                    }
                    catch (Exception)
                    {
                        Console.WriteLine("{0} {1}", (int)response.StatusCode, response.StatusCode);
                    }

                    Environment.Exit(1);
                }
                return result;
            }
            catch (TaskCanceledException)
            {
                Console.WriteLine("Request timed out.");
                Console.WriteLine("Try --timeout option to extend timeout.");
                Console.WriteLine("");
                Environment.Exit(1);
            }
            catch (Exception)
            {
                throw;
            }
            return "";
#else
            // テスト用：サーバーの代わりにファイルから json を読み込む
            using (FileStream sourceStream = new FileStream("D:\\home\\sako\\siglo\\sdk5\\Programs\\Eris\\Sources\\Tools\\CrashReport\\response_sample.json",
                FileMode.Open, FileAccess.Read, FileShare.Read,
                bufferSize: 4096, useAsync: true))
            {
                StringBuilder sb = new StringBuilder();

                byte[] buffer = new byte[0x1000];
                int numRead;
                while ((numRead = await sourceStream.ReadAsync(buffer, 0, buffer.Length)) != 0)
                {
                    string text = Encoding.UTF8.GetString(buffer, 0, numRead);
                    sb.Append(text);
                }

                return sb.ToString();
            }
#endif
        }

        static private string GetCrashReportDataFromBinaryStdin(CommandLineOption option)
        {
            List<CrashSummaryItem> crashList = new List<CrashSummaryItem>();
            CrashSummaryItem ci = new CrashSummaryItem();

            // 標準入力からクラッシュダンプ情報を取得
            Stream stdin = Console.OpenStandardInput();
            byte[] buffer = new byte[1024 * 1024];
            int bytes = stdin.Read(buffer, 0, buffer.Length);
            buffer = buffer.Take(bytes).ToArray();
            ci.exception_info = Convert.ToBase64String(buffer);

            ci.application_version = "1.0.0";
            ci.crash_report_hash = "";
            ci.error_code = "2168-0002";
            ci.count = 100;
            ci.serial_number_count = 100;
            crashList.Add(ci);

            // クラッシュレポートサーバーから取得した形式に合わせる
            Stream stream = new MemoryStream();
            var serializer = new DataContractJsonSerializer(typeof(List<CrashSummaryItem>));
            serializer.WriteObject(stream, crashList);

            stream.Position = 0;
            StreamReader sr = new StreamReader(stream);

            return sr.ReadToEnd();
        }

        static private void ParseCrashReportSummary(string data, CommandLineOption option)
        {
            var serializer = new DataContractJsonSerializer(typeof(List<CrashSummaryItem>));
            Stream stream = new MemoryStream(Encoding.Unicode.GetBytes(data));
            var deserialized = (List<CrashSummaryItem>)serializer.ReadObject(stream);

            foreach (CrashSummaryItem item in deserialized)
            {
                var ci = new CrashInformation();

                ci.SetApplicationVersion(item.application_version);
                ci.SetErrorCode(item.error_code);
                ci.SetSerialNumberCount(item.serial_number_count);
                ci.SetTotalCount(item.count);
                ci.SetCrashHash(Convert.FromBase64String(item.crash_report_hash));
                bool result = ci.ParseCrashInfo(Convert.FromBase64String(item.exception_info), option);

                if (result)
                {
                    m_CrashInfo.Add(ci);
                }
            }
        }

        static private void ParseCrashReportDetail(string data, CommandLineOption option)
        {
            var serializer = new DataContractJsonSerializer(typeof(List<CrashDetailItem>));
            Stream stream = new MemoryStream(Encoding.Unicode.GetBytes(data));
            var deserialized = (List<CrashDetailItem>)serializer.ReadObject(stream);

            int count = 0;
            foreach (CrashDetailItem item in deserialized)
            {
                var ci = new CrashInformation();

                ci.SetApplicationVersion(item.application_version);
                ci.SetErrorCode(item.error_code);
                ci.SetOccurrenceTimeStamp(item.occurrence_timestamp_net);
                ci.SetOsVersion(item.os_version);
                ci.SetCrashHash(Convert.FromBase64String(item.crash_report_hash));
                ci.SetDyingMessage(item.dying_message);
                ci.SetSerialNumberCount(1);
                ci.SetTotalCount(1);
                bool result = ci.ParseCrashInfo(Convert.FromBase64String(item.exception_info), option);

                if (result)
                {
                    m_CrashInfo.Add(ci);
                    count++;
                    if (option.FilterTop > 0 && count >= option.FilterTop)
                    {
                        break;
                    }
                }
            }
        }

        static private void OutputJson(List<CrashInformation> data)
        {
            var serializer = new DataContractJsonSerializer(typeof(List<CrashInformation>));
            Stream stream = new MemoryStream();

            serializer.WriteObject(stream, data);
            stream.Position = 0;
            StreamReader sr = new StreamReader(stream);
            Console.WriteLine(sr.ReadToEnd());
        }

        static private void ShowCrashReportSummary(CommandLineOption option)
        {
            if (option.OutputInJson)
            {
                List<CrashInformation> data = new List<CrashInformation>();

                int count = 0;
                foreach (var ci in m_CrashInfo)
                {
                    if (ci.IsShown(option))
                    {
                        data.Add(ci);
                        count++;
                        if (option.FilterTop > 0 && count >= option.FilterTop)
                        {
                            break;
                        }
                    }
                }
                OutputJson(data);
            }
            else
            {
                int showCount = 0;
                ulong showSerialCount = 0;
                ulong showTotalCount = 0;

                if (option.CountOnlyFlag == false && option.BinaryInput == false)
                {
                    Console.WriteLine("Hash                                                             ErrorCode Serial/Total");
                }

                if (!option.BinaryInput)
                {
                    foreach (var ci in m_CrashInfo)
                    {
                        if (showCount >= option.FilterTop)
                        {
                            break;
                        }
                        var result = ci.ShowSummary(option);
                        if (result.isShown)
                        {
                            showCount++;
                            showSerialCount += result.serialCount;
                            showTotalCount += result.totalCount;
                        }
                    }
                }

                if (option.CountOnlyFlag)
                {
                    Console.WriteLine("{0}", showSerialCount);
                }

                if (option.SummaryOnlyFlag == false && option.CountOnlyFlag == false)
                {
                    if (!option.BinaryInput)
                    {
                        Console.WriteLine("");
                    }
                    showCount = 0;
                    foreach (var ci in m_CrashInfo)
                    {
                        if (showCount >= option.FilterTop)
                        {
                            break;
                        }
                        var result = ci.ShowCrashDump(option, false);
                        if (result.isShown)
                        {
                            showCount++;
                        }
                    }
                }
            }
        }

        static private void ShowCrashReportDetail(CommandLineOption option)
        {
            if (option.OutputInJson)
            {
                OutputJson(m_CrashInfo);
            }
            else
            {
                foreach (var ci in m_CrashInfo)
                {
                    ci.ShowCrashDump(option, true);
                }
            }
        }

        static void Main(string[] args)
        {
            string data = "";
            var option = CommandLineOption.Parse(args);

            try
            {
                SymbolResolver.Prepare(option.Addr2LinePath, option.ApplicationNssPath);

                if (option.DetailMode)
                {
                    if (option.BinaryInput)
                    {
                        // 標準入力からバイナリでダンプ情報を入力
                        data = GetCrashReportDataFromBinaryStdin(option);
                    }
                    else
                    {
                        // クラッシュレポートサーバからデータを取得
                        data = QueryCrashReportData(option).Result;
                    }

                    // クラッシュレポート情報のパース
                    ParseCrashReportDetail(data, option);

                    // クラッシュレポート表示
                    ShowCrashReportDetail(option);
                }
                else
                {
                    if (option.BinaryInput)
                    {
                        // 標準入力からバイナリでダンプ情報を入力
                        data = GetCrashReportDataFromBinaryStdin(option);
                    }
                    else
                    {
                        // クラッシュレポートサーバからデータを取得
                        data = QueryCrashReportData(option).Result;
                        //Console.WriteLine(data);
                    }

                    // クラッシュレポート情報のパース
                    ParseCrashReportSummary(data, option);

                    // クラッシュレポート表示
                    ShowCrashReportSummary(option);
                }
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e);
                Environment.Exit(1);
            }
        }
    }
}
