﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace CrashReport
{
    public class CommandLineOption
    {
        public string ServiceDiscovery { get; private set; } = "lp1";
        public string ApplicationId { get; private set; } = null;
        public string ApplicationVersion { get; private set; } = null;
        public string ApplicationNssPath { get; private set; } = null;
        public string AccessToken { get; private set; } = null;
        public bool SummaryOnlyFlag { get; private set; } = false;
        public string FilterHash { get; private set; } = null;
        public ulong FilterMinimumCount { get; private set; } = 1;
        public int FilterTop { get; private set; } = 1000;
        public string FilterDateBegin { get; private set; } = null;
        public string FilterDateEnd { get; private set; } = null;
        public string Fqdn { get; private set; } = null;
        public string Addr2LinePath { get; private set; } = null;
        public bool CountOnlyFlag { get; private set; } = false;
        public string NintendoSdkRoot { get; private set; } = System.Environment.GetEnvironmentVariable("NINTENDO_SDK_ROOT");
        public string BuildType { get; private set; } = "Release";
        public ulong ReportOffset { get; private set; } = 0;
        public bool BinaryInput { get; private set; } = false;
        public bool DetailMode { get; private set; } = false;
        public bool OutputInJson { get; private set; } = false;
        public ulong RequestTimeout { get; private set; } = 180;

        private string GetNextArg(string[] args, int index)
        {
            if (index + 1 >= args.Length)
            {
                Console.WriteLine("Missing an argument for {0}.", args[index]);
                ShowUsage();
                Environment.Exit(1);
            }
            return args[index + 1];
        }

        private void ParseImpl(string[] args)
        {
            for (int i = 0; i < args.Length; i += 2)
            {
                if (args[i] == "\n")
                {
                    --i;
                    continue;
                }

                switch (args[i])
                {
                    case "--application_id":
                        {
                            ApplicationId = System.Text.RegularExpressions.Regex.Replace(GetNextArg(args, i), @"^0x", "");
                        }
                        break;

                    case "--application_version":
                        {
                            ApplicationVersion = GetNextArg(args, i);
                        }
                        break;

                    case "--access_token":
                        {
                            AccessToken = GetNextArg(args, i);
                        }
                        break;

                    case "--nss":
                        {
                            ApplicationNssPath = GetNextArg(args, i);
                            if (System.IO.File.Exists(ApplicationNssPath) == false)
                            {
                                Console.WriteLine("File not found: {0}", ApplicationNssPath);
                                Environment.Exit(1);
                            }
                        }
                        break;

                    case "--summary_only":
                        {
                            SummaryOnlyFlag = true;
                            --i;
                        }
                        break;

                    case "--count_only":
                        {
                            CountOnlyFlag = true;
                            --i;
                        }
                        break;

                    case "--date_begin":
                        {
                            FilterDateBegin = GetNextArg(args, i);
                        }
                        break;

                    case "--date_end":
                        {
                            FilterDateEnd = GetNextArg(args, i);
                        }
                        break;

                    case "--top":
                        {
                            FilterTop = Convert.ToInt32(GetNextArg(args, i));
                        }
                        break;

                    case "--hash":
                        {
                            FilterHash = GetNextArg(args, i);
                        }
                        break;

                    case "--sdk_root":
                        {
                            NintendoSdkRoot = GetNextArg(args, i);
                        }
                        break;

                    case "--min_count":
                        {
                            FilterMinimumCount = Convert.ToUInt64(GetNextArg(args, i));
                        }
                        break;

                    case "--offset":
                        {
                            ReportOffset = Convert.ToUInt64(GetNextArg(args, i));
                        }
                        break;

                    case "--service_discovery":
                        {
                            ServiceDiscovery = GetNextArg(args, i);
                        }
                        break;

                    case "--fqdn":
                        {
                            Fqdn = GetNextArg(args, i);
                        }
                        break;

                    case "--addr2line":
                        {
                            Addr2LinePath = GetNextArg(args, i);
                            if (System.IO.File.Exists(Addr2LinePath) == false)
                            {
                                Console.WriteLine("File not found: {0}", Addr2LinePath);
                                Environment.Exit(1);
                            }
                        }
                        break;

                    case "--build_type":
                        {
                            BuildType = GetNextArg(args, i);
                        }
                        break;

                    case "--timeout":
                        {
                            RequestTimeout = Convert.ToUInt64(GetNextArg(args, i));
                        }
                        break;

                    case "--binary":
                        {
                            BinaryInput = true;
                            --i;
                        }
                        break;

                    case "--default":
                        {
                            DetailMode = false;
                            --i;
                        }
                        break;

                    case "--detail":
                        {
                            DetailMode = true;
                            --i;
                        }
                        break;

                    case "--text":
                        {
                            OutputInJson = false;
                            --i;
                        }
                        break;

                    case "--json":
                        {
                            OutputInJson = true;
                            --i;
                        }
                        break;

                    case "--version":
                        {
                            var app = System.Reflection.Assembly.GetExecutingAssembly().GetName();
                            Console.WriteLine("{0}: Version {1}", app.Name, app.Version);
                            Environment.Exit(1);
                        }
                        break;

                    case "--help":
                        {
                            ShowUsage();
                            Environment.Exit(1);
                        }
                        break;

                    default:
                        {
                            Console.WriteLine("Unknown option: {0}", args[i]);
                            ShowUsage();
                            Environment.Exit(1);
                        }
                        break;
                }
            }
        }

        static private void ShowUsage()
        {
            Console.WriteLine("CrashReport.exe");
            Console.WriteLine("  --application_id       <id>       [REQUIRED] The application ID");
            Console.WriteLine("  --application_version  <version>  [REQUIRED] The version of the application (DisplayVersion)");
            Console.WriteLine("  --access_token         <token>    [REQUIRED] The obtained access token");
            Console.WriteLine("");
            Console.WriteLine("Mode options:");
            Console.WriteLine("  --default                         [DEFAULT] Display a summary and the exception thread dumps");
            Console.WriteLine("  --detail                          Display each crash report in detail (all threads dump, dying message)");
            Console.WriteLine("");
            Console.WriteLine("Output format options:");
            Console.WriteLine("  --text                            [DEFAULT] Display results in text format");
            Console.WriteLine("  --json                            Display results in JSON format");
            Console.WriteLine("");
            Console.WriteLine("Other options:");
            Console.WriteLine("  --nss                  <path>     The path for the application's NSS file");
            Console.WriteLine("  --date_begin           <date>     The starting date and time of the period in ISO8601");
            Console.WriteLine("  --date_end             <date>     The ending date and time of the period in ISO8601");
            Console.WriteLine("  --top                  <N>        Only display the first <N> results");
            Console.WriteLine("  --hash                 <hash>     Only display reports for which the hash value matches <hash>");
            Console.WriteLine("  --addr2line            <path>     The path of the addr2line tool for resolving symbols");
            Console.WriteLine("  --summary_only                    Only display a summary (*1)");
            Console.WriteLine("  --count_only                      Only display the total number of systems on which the crash occurred (*1)");
            Console.WriteLine("  --min_count            <N>        Only display reports which occurred <N> times or more (*1)");
            Console.WriteLine("  --offset               <N>        Get crash report results from <N>-th offset (*2)");
            Console.WriteLine("  --timeout              <second>   Set the second to wait before the request times out");
            Console.WriteLine("  --version                         Display version information");
            Console.WriteLine("  --help                            Display Help");
            Console.WriteLine("                                    *1: default mode only, *2: detail mode only");
        }

        static public CommandLineOption Parse(string[] args)
        {
            var option = new CommandLineOption();
            option.ParseImpl(args);

            if (!option.BinaryInput)
            {
                if (option.ApplicationId == null)
                {
                    Console.WriteLine("--application_id must be specified.");
                    ShowUsage();
                    Environment.Exit(1);
                }

                if (option.ApplicationVersion == null)
                {
                    Console.WriteLine("--application_version must be specified.");
                    ShowUsage();
                    Environment.Exit(1);
                }

                if (option.AccessToken == null)
                {
                    Console.WriteLine("--access_token must be specified.");
                    ShowUsage();
                    Environment.Exit(1);
                }
            }

            if (option.Addr2LinePath == null || System.IO.File.Exists(option.Addr2LinePath) == false)
            {
                string path = option.NintendoSdkRoot + "\\Compilers\\NX\\nx\\aarch64\\bin\\aarch64-nintendo-nx-elf-addr2line.exe";
                if (System.IO.File.Exists(path))
                {
                    option.Addr2LinePath = path;
                }
                else
                {
                    if (option.ApplicationNssPath != null)
                    {
                        Console.WriteLine("Failed to find addr2line.exe. Try --addr2line option.");
                        ShowUsage();
                        Environment.Exit(1);
                    }
                }
            }

            return option;
        }
    }
}
