﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "OpusBasicInfo.h"
#include "OpusMultiStreamInfo.h"
#include "detail/OpusCodecImpl.h"

using namespace System;

namespace Nintendo {
namespace CodecTool {

public ref class OpusEncoder
{
public:

    enum class OpusBitRateControl
    {
        Invalid,
        Constant,
        Variable,
        ConstrainedVariable,
    };

    enum class OpusCodingMode
    {
        Invalid,
        Silk,
        Celt,
    };

    ref struct UnexpectedCodingModeException : public System::Exception
    {
    };

    ref struct UnexpectedInternalErrorException : public System::Exception
    {
    };

    OpusEncoder(int sampleRate, int channelCount, uint32_t channelMask);
    ~OpusEncoder();
    !OpusEncoder();

    array<Byte>^ Encode(array<short>^ input, int sampleCount);

    OpusBasicInfo^ GetBasicInfo();
    OpusMultiStreamInfo^ GetMultiStreamInfo();

private:
    void Initialize(int sampleRate, int channelCount, uint32_t channelMask);

public:
    property int SampleRate
    {
        int get()
        {
            return this->m_pOpusEncoderImpl->GetSampleRate();
        }
    }
    property int ChannelCount
    {
        int get()
        {
            return this->m_pOpusEncoderImpl->GetChannelCount();
        }
    }
    property array<uint32_t>^ FinalRanges
    {
        array<uint32_t>^ get()
        {
            array<uint32_t>^ finalRanges = gcnew array<uint32_t>(this->m_pOpusEncoderImpl->GetTotalStreamCount());
            pin_ptr<uint32_t> ptr = &finalRanges[0];
            this->m_pOpusEncoderImpl->GetFinalRanges(ptr);
            return finalRanges;
        }
    }
    property int BitRate
    {
        void set(int bitRate)
        {
            this->m_pOpusEncoderImpl->SetBitRate(bitRate);
        }
        int get()
        {
            return this->m_pOpusEncoderImpl->GetBitRate();
        }
    }

    property OpusBitRateControl BitRateControl
    {
        void set(OpusBitRateControl bitRateControl)
        {
            switch (bitRateControl)
            {
            case OpusBitRateControl::Constant:
                m_pOpusEncoderImpl->SetBitRateControl(detail::OpusEncoderImpl::BitRateControl::Constant);
                break;
            case OpusBitRateControl::Variable:
                m_pOpusEncoderImpl->SetBitRateControl(detail::OpusEncoderImpl::BitRateControl::Variable);
                break;
            case OpusBitRateControl::ConstrainedVariable:
                m_pOpusEncoderImpl->SetBitRateControl(detail::OpusEncoderImpl::BitRateControl::ConstrainedVariable);
                break;
            default:
                // Do nothing.
                break;
            }
        }
        OpusBitRateControl get()
        {
            switch (this->m_pOpusEncoderImpl->GetBitRateControl())
            {
            case detail::OpusEncoderImpl::BitRateControl::Constant:
                return OpusBitRateControl::Constant;
            case detail::OpusEncoderImpl::BitRateControl::Variable:
                return OpusBitRateControl::Variable;
            case detail::OpusEncoderImpl::BitRateControl::ConstrainedVariable:
                return OpusBitRateControl::ConstrainedVariable;
            default:
                return OpusBitRateControl::Invalid;
            }
        }
    }

    property OpusCodingMode CodingMode
    {
        void set(OpusCodingMode codingMode)
        {
            switch (codingMode)
            {
            case OpusCodingMode::Silk:
                m_pOpusEncoderImpl->BindCodingMode(detail::OpusEncoderImpl::CodingMode::Silk);
                break;
            case OpusCodingMode::Celt:
                m_pOpusEncoderImpl->BindCodingMode(detail::OpusEncoderImpl::CodingMode::Celt);
                break;
            default:
                // Do nothing.
                break;
            }
        }
        OpusCodingMode get()
        {
            switch (this->m_pOpusEncoderImpl->GetCodingMode())
            {
            case detail::OpusEncoderImpl::CodingMode::Silk:
                return OpusCodingMode::Silk;
            case detail::OpusEncoderImpl::CodingMode::Celt:
                return OpusCodingMode::Celt;
            default:
                return OpusCodingMode::Invalid;
            }
        }
    }
    property int PreSkipSampleCount
    {
        int get()
        {
            return this->m_pOpusEncoderImpl->GetPreSkipSampleCount();
        }
    }

    property int TotalStreamCount
    {
        int get()
        {
            return this->m_pOpusEncoderImpl->GetTotalStreamCount();
        }
    }

internal:
    detail::OpusEncoderImpl* m_pOpusEncoderImpl;
};

}  // namespace CodecTool
}  // namespace Nintendo
