﻿using System;
using System.Diagnostics;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.Properties;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor
{
    public class StringHelper
    {
        private readonly Container _diContainer;

        public StringHelper(Container diContainer)
        {
            _diContainer = diContainer;
        }

        public string AppModeName => GetAppModeName(_diContainer.GetInstance<AppProfile>().AppMode);

        public string GetAppModeName(AppModeType appModeType)
        {
            var name = Resources.ResourceManager.GetString($"AppMode_{appModeType}");
            Debug.Assert(string.IsNullOrEmpty(name) == false);
            return name;
        }

        public string GetCardClockRateReadableString(CardSpec cardSpec, bool isNspMode = true)
        {
            if (isNspMode)
            {
                var autoSet = cardSpec.IsAutomaticSettingClockRate ? $" ({Resources.CardSpec_AutoSet})" : string.Empty;
                return $"{cardSpec.ClockRate}{Resources.MHz}{autoSet}";
            }
            return cardSpec.IsAutomaticSettingClockRate
                ? Resources.CardSpec_AutoSet
               : $"{cardSpec.ClockRate}{Resources.MHz}";
        }

        public string GetCardSizeReadableString(CardSpec cardSpec, bool isNspMode = true)
        {
            if (isNspMode)
            {
                var autoSet = cardSpec.IsAutomaticSettingSize ? $" ({Resources.CardSpec_AutoSet})" : string.Empty;
                return $"{cardSpec.Size}{Resources.GB}{autoSet}";
            }
            return cardSpec.IsAutomaticSettingSize
                ? Resources.CardSpec_AutoSet
               : $"{cardSpec.Size}{Resources.GB}";
        }

        public string GetNspHandleResultMessage(NspHandleResult result)
        {
            var message =
                Resources.ResourceManager.GetString($"NspHandleResult_{result}");
            Debug.Assert(string.IsNullOrEmpty(message) == false);
            return message;
        }

        public string CreateNew => string.Format(Resources.OP_CreateNew, AppModeName);
        public string Import => string.Format(Resources.OP_Import, AppModeName);

        public string ImportFromType(ImportableFileType type)
        {
            var name =
                Resources.ResourceManager
                    .GetString($"ImportableFileType_{type}");

            Debug.Assert(string.IsNullOrEmpty(name) == false);

            return string.Format(Resources.OP_Import, name);
        }

        public static string GetWorkingMessage(WorkingKind kind, WorkingState state)
        {
            switch (state)
            {
                case WorkingState.Waiting:
                    return string.Empty;

                case WorkingState.Canceling:
                    return Resources.Canceling;

                case WorkingState.Working:
                case WorkingState.Succeeded:
                case WorkingState.Failed:
                case WorkingState.Canceled:
                    return Resources.ResourceManager.GetString($"{kind}_{state}", Resources.Culture);

                default:
                    throw new ArgumentOutOfRangeException(nameof(state), state, null);
            }
        }
    }
}
