﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using BezelEditor.Mvvm;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reactive.Concurrency;
using System.Reactive.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Windows;
using BezelEditor.Foundation;
using BezelEditor.Foundation.Types;
using Livet.Messaging.Windows;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.SelectTwoNspFilesWindow
{
    public class SelectTwoFilesWindowVm : ViewModelBase
    {
        private class ComparableFile
        {
            public PatchContentMeta PatchContent { get; set; }
            public ComparableFileType FileType { get; set; }

            // 初版アプリケーションの指定が必要なのは PatchNsp の場合に限る
            // PatchNspExtractedDirectory の場合は既に必要なファイルが展開済みで、初版アプリは不要なため
            public bool IsRequiredOriginalVersion =>
                FileType == ComparableFileType.PatchNsp &&
                PatchContent != null && PatchContent.IsProduction == false;
        }

        public ReactiveCommand CloseCommand { get; }
        public ReactiveCommand OkCommand { get; }

        public FilePathStringParamVm LeftNspFile { get; }
        public FilePathStringParamVm RightNspFile { get; }

        public FilePathStringParamVm LeftOriginalApplicationNspFile { get; }
        public FilePathStringParamVm RightOriginalApplicationNspFile { get; }

        public DialogResultType Result { get; private set; } = DialogResultType.Cancel;

        public ReactiveProperty<ComparableFileType> LeftType { get; }
        public ReactiveProperty<ComparableFileType> RightType { get; }
        public ReactiveProperty<ComparableFileType> LeftOriginalApplicationType { get; }
        public ReactiveProperty<ComparableFileType> RightOriginalApplicationType { get; }

        // ダミー。検索系コントロールのために用意
        public ReactiveProperty<string> DelaySearchWord { get; set; }

        private readonly Container _diContainer;

        private readonly ManualResetEventSlim _disposeSync = new ManualResetEventSlim(false);

        public SelectTwoFilesWindowVm(Container diContainer, ComparisonTargetFiles model)
        {
            _diContainer = diContainer;

            CompositeDisposable.Add(() => _disposeSync.Set());

            var file0Rp = model.ToReactivePropertyAsSynchronized(x => x.File0,
                ReactivePropertyMode.RaiseLatestValueOnSubscribe);
            var file1Rp = model.ToReactivePropertyAsSynchronized(x => x.File1,
                ReactivePropertyMode.RaiseLatestValueOnSubscribe);

            var originalAppNspFile0Rp = model.ToReactivePropertyAsSynchronized(x => x.OriginalApplicationNspFile0,
                ReactivePropertyMode.RaiseLatestValueOnSubscribe);
            var originalAppNspFile1Rp = model.ToReactivePropertyAsSynchronized(x => x.OriginalApplicationNspFile1,
                ReactivePropertyMode.RaiseLatestValueOnSubscribe);

            LeftNspFile = new FilePathStringParamVm(
                nameof(Resources.ComparisonTargetFile),
                null,
                file0Rp,
                nameof(Resources.DialogFilter_ComparableFile),
                null,
                false
            ).AddTo(CompositeDisposable);

            RightNspFile = new FilePathStringParamVm(
                nameof(Resources.ComparisonTargetFile),
                null,
                file1Rp,
                nameof(Resources.DialogFilter_ComparableFile),
                null,
                false
            ).AddTo(CompositeDisposable);

            LeftOriginalApplicationNspFile = new FilePathStringParamVm(
                nameof(Resources.ComparisonTargetFile_OriginalApplicatioNsp),
                null,
                originalAppNspFile0Rp,
                nameof(Resources.DialogFilter_Nsp),
                null,
                false
            ).AddTo(CompositeDisposable);
            LeftOriginalApplicationNspFile.Visibility.Value = Visibility.Collapsed;

            RightOriginalApplicationNspFile = new FilePathStringParamVm(
                nameof(Resources.ComparisonTargetFile_OriginalApplicatioNsp),
                null,
                originalAppNspFile1Rp,
                nameof(Resources.DialogFilter_Nsp),
                null,
                false
            ).AddTo(CompositeDisposable);
            RightOriginalApplicationNspFile.Visibility.Value = Visibility.Collapsed;

            LeftType = model.ToReactivePropertyAsSynchronized(x => x.FileType0).AddTo(CompositeDisposable);
            RightType = model.ToReactivePropertyAsSynchronized(x => x.FileType1).AddTo(CompositeDisposable);

            LeftOriginalApplicationType = new ReactiveProperty<ComparableFileType>().AddTo(CompositeDisposable);
            RightOriginalApplicationType = new ReactiveProperty<ComparableFileType>().AddTo(CompositeDisposable);

            file0Rp
                .ObserveOn(ThreadPoolScheduler.Instance)
                .Subscribe(p =>
                {
                    LeftType.Value = ComparableFileType.Detecting;
                    var t = GetFileType(p);
                    LeftType.Value = t.FileType;

                    if (t.IsRequiredOriginalVersion)
                    {
                        LeftOriginalApplicationNspFile.Visibility.Value = Visibility.Visible;
                        originalAppNspFile0Rp.ForceNotify();
                    }
                    else
                    {
                        LeftOriginalApplicationType.Value = t.FileType == ComparableFileType.PatchNsp
                            ? ComparableFileType.AppNsp
                            : ComparableFileType.NotSet;
                        LeftOriginalApplicationNspFile.Visibility.Value = Visibility.Collapsed;
                    }
                })
                .AddTo(CompositeDisposable);

            file1Rp
                .ObserveOn(ThreadPoolScheduler.Instance)
                .Subscribe(p =>
                {
                    RightType.Value = ComparableFileType.Detecting;
                    var t = GetFileType(p);
                    RightType.Value = t.FileType;

                    if (t.IsRequiredOriginalVersion)
                    {
                        RightOriginalApplicationNspFile.Visibility.Value = Visibility.Visible;
                        originalAppNspFile1Rp.ForceNotify();
                    }
                    else
                    {
                        RightOriginalApplicationType.Value = t.FileType == ComparableFileType.PatchNsp
                            ? ComparableFileType.AppNsp
                            : ComparableFileType.NotSet;
                        RightOriginalApplicationNspFile.Visibility.Value = Visibility.Collapsed;
                    }
                })
                .AddTo(CompositeDisposable);

            EnsurePatchNspValidHistory(LeftOriginalApplicationType, originalAppNspFile0Rp, file0Rp);
            EnsurePatchNspValidHistory(RightOriginalApplicationType, originalAppNspFile1Rp, file1Rp);

            CloseCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            CloseCommand
                .Subscribe(_ => Close(DialogResultType.Cancel))
                .AddTo(CompositeDisposable);

            OkCommand =
                Observable
                    .Merge(LeftType.ToUnit())
                    .Merge(RightType.ToUnit())
                    .Merge(LeftOriginalApplicationType.ToUnit())
                    .Merge(RightOriginalApplicationType.ToUnit())
                    .Select(_ =>
                    {
                        if (LeftType.Value == ComparableFileType.NotSet)
                            return false;

                        if (LeftType.Value == ComparableFileType.Unknown)
                            return false;

                        if (LeftType.Value.IsComparable(RightType.Value) == false)
                            return false;

                        // 片方がパッチなら、常にパッチとして比較する
                        model.FileType = RightType.Value.IsPatch() ? ComparableFileType.PatchNsp : LeftType.Value;

                        // パッチ nsp の場合、初版アプリケーションが指定されている必要がある
                        // 展開済みパッチ nsp ディレクトリ (PatchNspExtractedDirectory) の場合は初版アプリケーション不要
                        if (model.FileType == ComparableFileType.PatchNsp)
                        {
                            if (LeftType.Value == ComparableFileType.PatchNsp)
                            {
                                if (LeftOriginalApplicationType.Value != ComparableFileType.AppNsp)
                                    return false;
                            }
                            if (RightType.Value == ComparableFileType.PatchNsp)
                            {
                                if (RightOriginalApplicationType.Value != ComparableFileType.AppNsp)
                                    return false;
                            }
                        }

                        return true;
                    })
                    .ToReactiveCommand()
                    .AddTo(CompositeDisposable);

            OkCommand
                .Subscribe(_ => Close(DialogResultType.Ok))
                .AddTo(CompositeDisposable);

            _disposeSync.AddTo(CompositeDisposable);
        }

        private void EnsurePatchNspValidHistory(
            ReactiveProperty<ComparableFileType> fileTypeProperty,
            ReactiveProperty<string> originalNspProperty,
            ReactiveProperty<string> patchNspProperty)
        {
            originalNspProperty
                .Select(p =>
                {
                    fileTypeProperty.Value = ComparableFileType.Detecting;
                    return new { PatchNspFile = patchNspProperty.Value, OriginalNspFile = p };
                })
                .ObserveOn(ThreadPoolScheduler.Instance)
                .SelectMany(p =>
                {
                    if (p == null)
                        return Task.FromResult(NspHandleResult.Error);
                    // パッチとアプリのアプリケーション ID が等しく、アプリが初版であることをチェック
                    return NspBuilder.IsValidPatchAndOriginalApplication(p.PatchNspFile, p.OriginalNspFile);
                })
                .ObserveOnUIDispatcher()
                .Subscribe(r =>
                {
                    if (_disposeSync.IsSet)
                        return;
                    fileTypeProperty.Value = r == NspHandleResult.Ok
                        ? ComparableFileType.AppNsp
                        : ComparableFileType.NotOriginalAppNspFile;
                })
                .AddTo(CompositeDisposable);
        }

        private void Close(DialogResultType result)
        {
            Result = result;
            Messenger.Raise(new WindowActionMessage(WindowAction.Close, "WindowAction"));
        }

        private static bool IsExistsReadableFile(string filePath)
        {
            if (string.IsNullOrEmpty(filePath))
                return false;

            if (File.Exists(filePath) == false)
                return false;

            return true;
        }

        private static ComparableFile GetFileTypeFromXmlContainsDirectory(string filePath)
        {
            // パスに .xml を指定した場合、3PT からダウンロードしてきた xml ディレクトリ構造があるとみなす
            if (File.Exists(filePath) &&
                Path.GetExtension(filePath)?.Equals(".xml", StringComparison.OrdinalIgnoreCase) == true)
            {
                filePath = Path.GetDirectoryName(filePath);
            }

            if (Directory.Exists(filePath) == false)
                return new ComparableFile {FileType = ComparableFileType.Unknown};

            var nspFile = new NspExtractedDirectory(filePath);
            var contentMetaType = Task.Run(async () => await NspImporter.DetectContentMetaTypeAsync(nspFile).ConfigureAwait(false)).Result;
            switch (contentMetaType)
            {
                case ContentMetaType.AddOnContent:
                    return new ComparableFile {FileType = ComparableFileType.AocNsp};
                case ContentMetaType.Application:
                    return new ComparableFile {FileType = ComparableFileType.AppNsp};
                case ContentMetaType.Patch:
                    return new ComparableFile
                    {
                        PatchContent = NspImporter.ReadContentMeta<PatchContentMeta>(nspFile),
                        FileType = ComparableFileType.PatchNspExtractedDirectory
                    };
                default:
                    return new ComparableFile {FileType = ComparableFileType.Unknown};
            }
        }

        private ComparableFile GetFileType(string filePath)
        {
            var t = GetFileTypeInternal(filePath);

            // パッチ作成不可の環境 (NX Addon 0.12 系) において、パッチ nsp は不明なファイルタイプ扱い
            if (t.FileType.IsPatch() &&
                _diContainer.GetInstance<ApplicationCapability>().IsSupportMakingPatch == false)
            {
                return new ComparableFile {FileType = ComparableFileType.Unknown};
            }

            return t;
        }

        private static ComparableFile GetFileTypeInternal(string filePath)
        {
            if (string.IsNullOrEmpty(filePath))
                return new ComparableFile {FileType = ComparableFileType.NotSet};

            {
                var t = GetFileTypeFromXmlContainsDirectory(filePath);
                if (t.FileType != ComparableFileType.Unknown)
                    return t;
            }

            if (IsExistsReadableFile(filePath) == false)
                return new ComparableFile {FileType = ComparableFileType.NotFound};

            AppModeType mode;
            try
            {
                mode = TypeHelper.ToAppMode(filePath);
            }
            catch
            {
                return new ComparableFile {FileType = ComparableFileType.Unknown};
            }

            switch (mode)
            {
                case AppModeType.ApplicationMeta:
                    return new ComparableFile {FileType = ComparableFileType.AppMeta};
                case AppModeType.AocMeta:
                    return new ComparableFile {FileType = ComparableFileType.AocMeta};

                case AppModeType.ApplicationNsp:
                    return new ComparableFile {FileType = ComparableFileType.AppNsp};
                case AppModeType.AocNsp:
                    return new ComparableFile {FileType = ComparableFileType.AocNsp};
                case AppModeType.PatchNsp:
                    return new ComparableFile
                    {
                        PatchContent = NspImporter.ReadContentMeta<PatchContentMeta>(new NspFile(filePath)),
                        FileType = ComparableFileType.PatchNsp
                    };

                default:
                    return new ComparableFile {FileType = ComparableFileType.Unknown};
            }
        }

    }
}
