﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Reactive.Linq;
using System.Threading;
using System.Threading.Tasks;
using BezelEditor.Foundation.Extentions;
using BezelEditor.Foundation.Utilities;
using BezelEditor.Mvvm;
using Microsoft.WindowsAPICodePack.Taskbar;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.SubPanel
{
    public class WorkingPanelVm : ViewModelBase
    {
        public ReadOnlyReactiveProperty<WorkingState> WorkingState { get; }
        public ReadOnlyReactiveProperty<string> WorkingProgress { get; }

        public ReactiveCommand CancelExportNspCommand { get; private set; }
        public ReactiveCommand CloseWorkingPanelCommand { get; private set; }

        public ReadOnlyReactiveProperty<bool> IsWaiting { get; }

        public ReactiveProperty<bool> IsFadeoutWorkingPanel { get; }
        public ReactiveProperty<string> ErrorSummary { get;  }
        public ReactiveProperty<string> ErrorMessages { get; }

        public ReactiveProperty<string> WorkingMessage { get; }

        public ReadOnlyReactiveProperty<string> OpenGeneratedObjectMessage { get; }

        public ReactiveProperty<bool> IsOpenGeneratedObject { get; }

        public ReadOnlyReactiveProperty<Action<string>> OpenGeneratedObjectAction { get; }

        public WorkingPanelVm(App app)
        {
            IsWaiting = app.ObserveProperty(x => x.WorkingState)
                .Select(s => s == Core.WorkingState.Waiting)
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);

            IsFadeoutWorkingPanel = new ReactiveProperty<bool>().AddTo(CompositeDisposable);
            IsOpenGeneratedObject = new ReactiveProperty<bool>(true).AddTo(CompositeDisposable);
            ErrorSummary = new ReactiveProperty<string>().AddTo(CompositeDisposable);
            ErrorMessages = new ReactiveProperty<string>().AddTo(CompositeDisposable);

            OpenGeneratedObjectMessage = app.ObserveProperty(x => x.OpenGeneratedMessageTag)
                .Select(t =>
                    string.IsNullOrEmpty(t) ? string.Empty : Resources.ResourceManager.GetString(t))
                .ToReadOnlyReactiveProperty(nameof(Resources.OpenGeneratedObjectContainsFolder))
                .AddTo(CompositeDisposable);

            OpenGeneratedObjectAction = app.ObserveProperty(x => x.OpenGeneratedObjectAction)
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);

            WorkingState = app.ObserveProperty(x => x.WorkingState)
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);

            WorkingProgress = app.ObserveProperty(x => x.WorkingProgress)
                .Select(p => $"{p}%")
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);

            WorkingState
                .Subscribe(s =>
                {
                    try
                    {
                        switch (s)
                        {
                            case Core.WorkingState.Waiting:
                                TaskbarManager.Instance.SetProgressState(TaskbarProgressBarState.Normal);
                                TaskbarManager.Instance.SetProgressValue(0, 100);
                                break;

                            case Core.WorkingState.Working:
                                TaskbarManager.Instance.SetProgressState(TaskbarProgressBarState.Indeterminate);
                                break;

                            case Core.WorkingState.Succeeded:
                                TaskbarManager.Instance.SetProgressState(TaskbarProgressBarState.Normal);
                                break;

                            case Core.WorkingState.Failed:
                                TaskbarManager.Instance.SetProgressState(TaskbarProgressBarState.Error);
                                TaskbarManager.Instance.SetProgressState(TaskbarProgressBarState.Error);
                                break;

                            case Core.WorkingState.Canceling:
                            case Core.WorkingState.Canceled:
                                TaskbarManager.Instance.SetProgressState(TaskbarProgressBarState.Paused);
                                break;

                            default:
                                throw new ArgumentOutOfRangeException(nameof(s), s, null);
                        }
                    }
                    catch
                    {
                        // ignored
                    }
                }).AddTo(CompositeDisposable);

            WorkingMessage = new ReactiveProperty<string>().AddTo(CompositeDisposable);
            Observable
                .Merge(app.ObserveProperty(x => x.WorkingKind).ToUnit())
                .Merge(app.ObserveProperty(x => x.WorkingState).ToUnit())
                .Subscribe(_ => UpdateWorkingMessage(app))
                .AddTo(CompositeDisposable);

            SetupCommands(app);
        }

        private void UpdateWorkingMessage(App app)
        {
            WorkingMessage.Value = StringHelper.GetWorkingMessage(app.WorkingKind, app.WorkingState);
        }

        private void SetupCommands(App app)
        {
            //////////////////////////////////////////////////////////////////////////////////////
            CancelExportNspCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            CancelExportNspCommand.Subscribe(_ =>
            {
                app.WorkingState = Core.WorkingState.Canceling;

                Debug.Assert(app.WorkingCancellationTokenSource != null);
                app.WorkingCancellationTokenSource.Cancel();
            }).AddTo(CompositeDisposable);

            //////////////////////////////////////////////////////////////////////////////////////
            CloseWorkingPanelCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            CloseWorkingPanelCommand
                .Subscribe(async _ =>
                {
                    var isOpenGeneratedObject = app.WorkingState == Core.WorkingState.Succeeded &&
                                                IsOpenGeneratedObject.Value;
                    var filePath = app.WorkingFilePath;

                    IsFadeoutWorkingPanel.Value = true;
                    await Task.Delay(TimeSpan.FromMilliseconds(300));
                    IsFadeoutWorkingPanel.Value = false;

                    app.WorkingState = Core.WorkingState.Waiting;

                    if (isOpenGeneratedObject)
                        OpenGeneratedObjectAction.Value?.Invoke(filePath);
                })
                .AddTo(CompositeDisposable);
        }
    }
}
