﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Reactive;
using System.Reactive.Linq;
using System.Windows;
using BezelEditor.Foundation.Extentions;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel
{
    public class ProjectEditPanelVm : ViewModelBase
    {
        public ReactiveProperty<string> SearchWord { get; }
        public ReactiveProperty<string> DelaySearchWord { get; }
        public ReactiveProperty<int> SearchHitPageCount { get; }

        public ReactiveProperty<PageVmBase[]> AllPages { get; }
        public ReactiveProperty<PageVmBase> AllSelectedPage { get; }

        private IEnumerable<ReadOnlyReactiveProperty<bool>> AllHasErrors
        {
            get
            {
                yield return HasErrorsOverviewPages;
                yield return HasErrorsBasicPages;
                yield return HasErrorsGeneralPages;
            }
        }

        public ReactiveProperty<PageVmBase[]> OverviewPages { get; }
        public ReactiveProperty<PageVmBase[]> BasicPages { get; }
        public ReactiveProperty<PageVmBase[]> GeneralPages { get; }
        public ReactiveProperty<PageVmBase[]> ErrorPages { get; }

        public ReadOnlyReactiveProperty<PageVmBase> OverviewSelectedPage { get; }
        public ReadOnlyReactiveProperty<PageVmBase> BasicSelectedPage { get; }
        public ReadOnlyReactiveProperty<PageVmBase> GeneralSelectedPage { get; }
        public ReadOnlyReactiveProperty<PageVmBase> ErrorSelectedPage { get; }

        public ReadOnlyReactiveProperty<bool> HasErrorsOverviewPages { get; } = null;
        public ReadOnlyReactiveProperty<bool> HasErrorsBasicPages { get; private set; }
        public ReadOnlyReactiveProperty<bool> HasErrorsGeneralPages { get; private set; }

        public ProjectEditPanelVm(Container diContainer, App app)
        {
            ////////////////////////////////////////////////////////////////////////////////////////////////////////////
            AllPages = new ReactiveProperty<PageVmBase[]>().AddTo(CompositeDisposable);
            ErrorPages = new ReactiveProperty<PageVmBase[]>(new PageVmBase[0]).AddTo(CompositeDisposable);
            OverviewPages = new ReactiveProperty<PageVmBase[]>(new PageVmBase[0]).AddTo(CompositeDisposable);
            BasicPages = new ReactiveProperty<PageVmBase[]>(new PageVmBase[0]).AddTo(CompositeDisposable);
            GeneralPages = new ReactiveProperty<PageVmBase[]>(new PageVmBase[0]).AddTo(CompositeDisposable);

            Observable
                .Merge(app.ObserveProperty(x => x.Project).ToUnit())
                .Subscribe(_ =>
                {
                    SetupPages(diContainer, app);
                })
                .AddTo(CompositeDisposable);

            CompositeDisposable.Add(() => AllPages.Value?.ForEach(x => x.Dispose()));
            CompositeDisposable.Add(() => AllHasErrors?.ForEach(x => x?.Dispose()));

            ////////////////////////////////////////////////////////////////////////////////////////////////////////////
            AllSelectedPage = AllPages.Select(_ => AllPages.Value.FirstOrDefault(x => x.PageGroup != PageGroups.Errors))
                .ToReactiveProperty()
                .AddTo(CompositeDisposable);

            OverviewSelectedPage =
                AllSelectedPage.Select(p => OverviewPages.Value.FirstOrDefault(x => x == p))
                    .ToReadOnlyReactiveProperty()
                    .AddTo(CompositeDisposable);

            BasicSelectedPage =
                AllSelectedPage.Select(p => BasicPages.Value.FirstOrDefault(x => x == p))
                    .ToReadOnlyReactiveProperty()
                    .AddTo(CompositeDisposable);

            GeneralSelectedPage =
                AllSelectedPage.Select(p => GeneralPages.Value.FirstOrDefault(x => x == p))
                    .ToReadOnlyReactiveProperty()
                    .AddTo(CompositeDisposable);

            ErrorSelectedPage =
                AllSelectedPage.Select(p => ErrorPages.Value.FirstOrDefault(x => x == p))
                    .ToReadOnlyReactiveProperty()
                    .AddTo(CompositeDisposable);

            ////////////////////////////////////////////////////////////////////////////////////////////////////////////
            SearchHitPageCount = new ReactiveProperty<int>().AddTo(CompositeDisposable);
            SearchWord = new ReactiveProperty<string>(string.Empty).AddTo(CompositeDisposable);

            DelaySearchWord = SearchWord
                .Throttle(TimeSpan.FromMilliseconds(250))
                .ToReactiveProperty()
                .AddTo(CompositeDisposable);

            // 検索なしは即時に評価する
            SearchWord
                .Where(string.IsNullOrEmpty)
                .Subscribe(s =>
                {
                    DelaySearchWord.Value = string.Empty;
                    SearchWord.Value = string.Empty;
                })
                .AddTo(CompositeDisposable);

            SearchWord
                .Where(word => word.IsEmpty() == false)
                .Subscribe(word => AllSelectedPage.Value = null)
                .AddTo(CompositeDisposable);

            DelaySearchWord
                .Subscribe(UpdatePageVisibility)
                .AddTo(CompositeDisposable);

            CultureService.Instance.ObserveProperty(x => x.Resources)
                .Subscribe(_ => SearchWord.Value = string.Empty)
                .AddTo(CompositeDisposable);

            ////////////////////////////////////////////////////////////////////////////////////////////////////////////
            UpdatePageVisibility(string.Empty);
        }

        private void SetupPages(Container diContainer, App app)
        {
            if (diContainer.GetInstance<AppProfile>().IsAocAppMode)
                SetupPages_AocMeta(diContainer);
            else
                SetupPages_Basic(diContainer, app);

            ErrorPages.Value.ForEach(p => p.PageGroup = PageGroups.Errors);
            OverviewPages.Value.ForEach(p => p.PageGroup = PageGroups.Overview);
            BasicPages.Value.ForEach(p => p.PageGroup = PageGroups.Basic);
            GeneralPages.Value.ForEach(p => p.PageGroup = PageGroups.General);

            // ReSharper disable ExplicitCallerInfoArgument
            HasErrorsBasicPages = SetupHasErros(app.Project.Meta, BasicPages.Value);
            HasErrorsGeneralPages = SetupHasErros(app.Project.Meta, GeneralPages.Value);

            RaisePropertyChanged(nameof(HasErrorsBasicPages));
            RaisePropertyChanged(nameof(HasErrorsGeneralPages));
            // ReSharper restore ExplicitCallerInfoArgument

            AllPages.Value?.ForEach(x => x.Dispose());
            AllHasErrors.ForEach(x => x?.Dispose());

            Observable
                .Merge(Observable.Return(Unit.Default).ToUnit())
                .Subscribe(_ =>
                {
                    AllPages.Value =
                        ErrorPages.Value
                            .Concat(OverviewPages.Value)
                            .Concat(BasicPages.Value)
                            .Concat(GeneralPages.Value)
                            .ToArray();

                    // ReSharper disable once PossibleNullReferenceException
                    foreach (var p in AllPages.Value)
                        p.Height
                            .Subscribe(__ => UpdatePageMargin())
                            .AddTo(p.CompositeDisposable);
                }).AddTo(CompositeDisposable);

        }

        private void SetupPages_Basic(Container diContainer, App app)
        {
            var appProfile = diContainer.GetInstance<AppProfile>();

            var isEnableDeviceSaveDataGroup = app.Project.Meta.Application.IsSpecifiedDeviceSaveDataSize &&
                                              app.Project.Meta.Application.IsSpecifiedDeviceSaveDataJournalSize;
            var isEnableCorePageGroup = appProfile.IsEnableCorePageGroup;
            var isEnableApplicationInfo = appProfile.AppMode == AppModeType.ApplicationNsp ||
                                          appProfile.AppMode == AppModeType.PatchNsp;
            var isEnableBcatPage = app.Project.AppCapability.IsSupportBcat;
            var isEnableTempDataAreaPage = app.Project.AppCapability.IsSupportTempAndCacheStorage;
            var isSupportPlayLogs = app.Project.AppCapability.IsSupportPlayLogs;

            ////////////////////////////////////////////////////////////////////////////////////////////////////////////
            var appInfo = isEnableApplicationInfo ? diContainer.GetInstance<ApplicationPageVm>() : null;
            var nspSizeInfo = isEnableApplicationInfo ? diContainer.GetInstance<NspSizePageVm>() : null;
            //
            var basic = diContainer.GetInstance<BasicPageVm>();
            var language = diContainer.GetInstance<LanguagePageVm>();
            var title = diContainer.GetInstance<TitlePageVm>();
            var limitation = diContainer.GetInstance<LimitationPageVm>();
            //
            var general = diContainer.GetInstance<GeneralPageVm>();
            var core = isEnableCorePageGroup ? diContainer.GetInstance<CorePageVm>() : null;
            var bcat = isEnableBcatPage ? diContainer.GetInstance<BcatPageVm>() : null;
            var account = diContainer.GetInstance<AccountPageVm>();
            var userSave = diContainer.GetInstance<SaveDataPageVm>();
            var tempDataArea = isEnableTempDataAreaPage ? diContainer.GetInstance<TemporaryDataAreaPageVm>() : null;
            var deviceSave = isEnableDeviceSaveDataGroup ? diContainer.GetInstance<DeviceSaveDataPageVm>() : null;
            var fsAccess = diContainer.GetInstance<FsAccessControlPageVm>();
            var feature = diContainer.GetInstance<FeaturePageVm>();
            var localComm = diContainer.GetInstance<LocalCommunicationPageVm>();
            var playLogs = isSupportPlayLogs ? diContainer.GetInstance<PlayLogsPageVm>() : null;
            var cardSpec = diContainer.GetInstance<CardSpecPageVm>();

            var appFormatError = isEnableApplicationInfo && app.Project.Meta.FileFormatErrors.Any()
                ? diContainer.GetInstance<ApplicationMetaErrorPageVm>()
                : null;
            var appUnpublishableError = isEnableApplicationInfo && app.Project.Meta.UnpublishableErrors.Any()
                ? diContainer.GetInstance<ApplicationUnpublishableErrorPageVm>()
                : null;

            ErrorPages.Value = ToValidPageArray(new PageVmBase[]
            {
                appFormatError,
                appUnpublishableError
            });

            BasicPages.Value = ToValidPageArray(new PageVmBase[]
            {
                appInfo,
                nspSizeInfo,
                basic,
                language,
                title,
                limitation
            });

            GeneralPages.Value = ToValidPageArray(new PageVmBase[]
            {
                general,
                core,
                account,
                bcat,
                userSave,
                deviceSave,
                tempDataArea,
                fsAccess,
                feature,
                localComm,
                playLogs,
                cardSpec
            });
        }

        private static PageVmBase[] ToValidPageArray(IEnumerable<PageVmBase> pages) =>
            pages.Where(x => x != null && x.Params.Any()).ToArray();

        private void SetupPages_AocMeta(Container diContainer)
        {
            var overview = diContainer.GetInstance<AocOverviewPageVm>();
            var contents = diContainer.GetInstance<AocContentsPageVm>();

            OverviewPages.Value = new PageVmBase[]
            {
                overview,
            };

            GeneralPages.Value = new PageVmBase[]
            {
                contents
            };
        }

        private ReadOnlyReactiveProperty<bool> SetupHasErros(ApplicationMeta appMeta, PageVmBase[] source)
        {
            return source
                .Select(x => x.HasErrors)
                .CombineLatestValuesAreAllFalse()
                .Inverse()
                .Where(hasErrors => hasErrors && appMeta.IsReadOnly == false)
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);
        }

        private void UpdatePageVisibility(string keyword)
        {
            var c = 0;

            foreach (var page in AllPages.Value)
            {
                if (page.PageGroup == PageGroups.Errors)
                    continue; // エラーページの表示制御はその他のページとは独立して行う

                page.UpdateVisibility(keyword);

                if (page.Visibility.Value == Visibility.Visible)
                    ++c;
            }

            SearchHitPageCount.Value = c;

            UpdatePageMargin();
        }

        private double _oldPageHeight;

        public void UpdatePageMargin()
        {
            UpdatePageMargin(_oldPageHeight);
        }

        public void UpdatePageMargin(double pageHeight)
        {
            // ReSharper disable once CompareOfFloatsByEqualityOperator
            if (pageHeight == 0.0)
                return;

            _oldPageHeight = pageHeight;

            var last = AllPages.Value.LastOrDefault(x => x.Visibility.Value == Visibility.Visible);

            foreach (var p in AllPages.Value)
            {
                var margin = p.Margin.Value;

                margin.Bottom =
                    p != last
                        ? PageVmBase.PageBottomMargin
                        : Math.Max(PageVmBase.PageBottomMargin,
                            pageHeight - p.Height.Value - PageVmBase.PageBottomMargin);

                p.Margin.Value = margin;
            }
        }
    }
}
