﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;
using System.Windows.Media.Animation;
using BezelEditor.Control.Basic;
using BezelEditor.Foundation.Extentions;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel
{
    /// <summary>
    /// ProjectEditPanel.xaml の相互作用ロジック
    /// </summary>
    public partial class ProjectEditPanel
    {
        public ProjectEditPanel()
        {
            InitializeComponent();
        }

        private void Button_Click(object sender, RoutedEventArgs e)
        {
            var button = (Button) sender;

            var border = MainContent
                .Descendants<Border>()
                .FirstOrDefault(x => button.DataContext == x.DataContext);

            if (border == null)
                return;

            var relativePoint = border.TransformToAncestor(MainContent).Transform(new Point(0, 0));
            ScrollToPosition(relativePoint.X, relativePoint.Y - 16);
            UpdatePageSelectionFromScroll(ScrollViewer.VerticalOffset);
        }

        private void ScrollToPosition(double x, double y)
        {
            var distance =
                Math.Sqrt(
                    (x - ScrollViewer.CurrentHorizontalOffset)*(x - ScrollViewer.CurrentHorizontalOffset) +
                    (y - ScrollViewer.CurrentVerticalOffset)*(y - ScrollViewer.CurrentVerticalOffset)
                );

            var durationMs = distance;

            durationMs = Math.Min(durationMs, 400);
            durationMs = Math.Max(durationMs, 150);

            var vertAnim = new DoubleAnimation
            {
                From = ScrollViewer.VerticalOffset,
                To = y,
                AccelerationRatio = 0.25,
                DecelerationRatio = 0.75,
                EasingFunction = new CubicEase
                {
                    EasingMode = EasingMode.EaseInOut,
                },
                Duration = new Duration(TimeSpan.FromMilliseconds(durationMs))
            };

            var horzAnim = new DoubleAnimation
            {
                From = ScrollViewer.HorizontalOffset,
                To = x,
                AccelerationRatio = 0.25,
                DecelerationRatio = 0.75,
                EasingFunction = new CubicEase
                {
                    EasingMode = EasingMode.EaseInOut,
                },
                Duration = new Duration(TimeSpan.FromMilliseconds(durationMs))
            };

            Timeline.SetDesiredFrameRate(vertAnim, 60);
            Timeline.SetDesiredFrameRate(horzAnim, 60);

            var sb = new Storyboard();
            sb.Children.Add(vertAnim);
            sb.Children.Add(horzAnim);

            Storyboard.SetTarget(vertAnim, ScrollViewer);
            Storyboard.SetTargetProperty(vertAnim, new PropertyPath(AnimationScrollViewer.CurrentVerticalOffsetProperty));
            Storyboard.SetTarget(horzAnim, ScrollViewer);
            Storyboard.SetTargetProperty(horzAnim,
                new PropertyPath(AnimationScrollViewer.CurrentHorizontalOffsetProperty));

            sb.Begin();
        }

        private void ScrollViewer_ScrollChanged(object sender, ScrollChangedEventArgs e)
        {
            var dataContext = (ProjectEditPanelVm) DataContext;

            if (string.IsNullOrEmpty(dataContext.SearchWord.Value) == false)
                return;

            // ReSharper disable once CompareOfFloatsByEqualityOperator
            if (e.VerticalChange != 0)
                UpdatePageSelectionFromScroll(e.VerticalOffset);
        }

        private void ScrollViewer_SizeChanged(object sender, SizeChangedEventArgs e)
        {
            UpdatePageMargin(e.NewSize.Height);
        }

        private void UpdatePageMargin(double height)
        {
            ((ProjectEditPanelVm) DataContext).UpdatePageMargin(height);
        }

        // スクロール値から選択ページを更新する
        private void UpdatePageSelectionFromScroll(double scrollY)
        {
            var dataContext = (ProjectEditPanelVm) DataContext;

            var y = 0.0;

            PageVmBase old = null;

            foreach (var i in dataContext.AllPages.Value.Where(x => x.Visibility.Value == Visibility.Visible))
            {
                var top = y;
                var bottom = y + i.Height.Value;

                if (old != null)
                {
                    var back = (old.Height.Value + old.Margin.Value.Top + old.Margin.Value.Bottom)*0.25;

                    top -= back;
                    bottom -= back;
                }
                else
                    bottom *= 0.25;

                if ((scrollY >= top) && (scrollY <= bottom))
                {
                    dataContext.AllSelectedPage.Value = i;
                    break;
                }

                y += i.Height.Value + i.Margin.Value.Top + i.Margin.Value.Bottom;

                old = i;
            }
        }

        private void Grid_DataContextChanged(object sender, DependencyPropertyChangedEventArgs e)
        {
            (e.OldValue as IDisposable)?.Dispose();
            if (e.NewValue == null)
                return;

            var dataContext = (ProjectEditPanelVm) DataContext;

            dataContext.SearchWord
                .Subscribe(_ => ScrollViewer.ScrollToTop())
                .AddTo(dataContext.CompositeDisposable);

            dataContext.AllPages
                .Subscribe(appPages =>
                {
                    UpdatePageMargin(ScrollViewer.ActualHeight);

                    _lastPageHeightDisposable?.Dispose();

                    var last = appPages.LastOrDefault();
                    if (last != null)
                        _lastPageHeightDisposable = last.Height
                            .Subscribe(_ => UpdatePageMargin(ScrollViewer.ActualHeight))
                            .AddTo(dataContext.CompositeDisposable);
                })
                .AddTo(dataContext.CompositeDisposable);

            dataContext.CompositeDisposable.Add(() => _lastPageHeightDisposable?.Dispose());
        }

        private IDisposable _lastPageHeightDisposable;

        private void Button_MouseRightButtonDown(object sender, MouseButtonEventArgs e)
        {
            e.Handled = true;
        }

        private void StackPanel_MouseWheel(object sender, MouseWheelEventArgs e)
        {
            ScrollViewer.RaiseEvent(e);
        }

        private void ClippingBorder_PreviewMouseLeftButtonDown(object sender, MouseButtonEventArgs e)
        {
            var dataContext = (ProjectEditPanelVm) DataContext;

            dataContext.SearchWord.Value = string.Empty;
        }

        private void Grid_PreviewKeyDown(object sender, KeyEventArgs e)
        {
            if (e.IsDown == false)
                return;

            if (e.Key == Key.F && Keyboard.Modifiers == ModifierKeys.Control)
                FocusManager.SetFocusedElement(FocusManager.GetFocusScope(SearchTextBox), SearchTextBox);
        }
    }
}
