﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Reactive.Linq;
using System.Threading.Tasks;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using Livet.Messaging;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Controls;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params
{
    public class IconPreviewParamVm : ParamVm
    {
        public ReactiveCommand ShowIconLargePreviewCommand { get; }
        public ReactiveCommand ShowNxIconPreviewCommand { get; }

        public ReactiveProperty<Brush> IconBrush { get; }
        public ReactiveProperty<Brush> NxIconBrush { get; }

        private IconActualSizePreviewWindowVm _iconLargePreviewVm;
        private IconActualSizePreviewWindowVm _nxIconPreviewVm;

        private class PreviewIcon
        {
            public string Raw { get; set; }
            public string Nx { get; set; }
        }

        public IconPreviewParamVm(Container diContainer, string captionTag, Title model,
            DisposableDirectory disposableDirectory)
            : base(captionTag, null, null)
        {
            ShowIconLargePreviewCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            ShowIconLargePreviewCommand
                .Subscribe(_ => ShowIconLargeSizePreviewWindow(model))
                .AddTo(CompositeDisposable);

            ShowNxIconPreviewCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            ShowNxIconPreviewCommand
                .Subscribe(_ => ShowNxIconPreviewWindow(model))
                .AddTo(CompositeDisposable);

            IconBrush = new ReactiveProperty<Brush>().AddTo(CompositeDisposable);
            NxIconBrush = new ReactiveProperty<Brush>().AddTo(CompositeDisposable);

            var project = diContainer.GetInstance<App>().Project;
            var builder = new NspBuilder();

            Observable
                //
                .Merge(model.ObserveProperty(x => x.IsReplaceIcon).ToUnit())
                .Merge(model.IconFilePath.PropertyChangedAsObservable().ToUnit())
                //
                .Merge(model.ObserveProperty(x => x.IsReplaceNxIcon).ToUnit())
                .Merge(model.NxIconFilePath.PropertyChangedAsObservable().ToUnit())
                //
                .Merge(model.ObserveProperty(x => x.OriginalIconFilePath).ToUnit())
                .Merge(model.ObserveProperty(x => x.OriginalNxIconFilePath).ToUnit())
                //
                .Throttle(TimeSpan.FromMilliseconds(100))
                .Select(async _ =>
                {
                    var preview = new PreviewIcon();

                    if (model.IsReplaceIcon &&
                        model.ValidationIconFilePath == Title.IconFilePathValidationType.Ok)
                    {
                        await ConvertPreviewIcon(model, disposableDirectory, project, builder, preview).ConfigureAwait(false);
                    }

                    if (model.IsReplaceIcon == false)
                    {
                        if (model.ValidationOriginalIconFilePath == Title.IconFilePathValidationType.Ok)
                            preview.Raw = model.OriginalIconFilePath;
                        if (model.ValidationNxIconFilePath == Title.NxIconFilePathValidationType.Ok)
                            preview.Nx = model.OriginalNxIconFilePath;
                    }

                    return preview;
                })
                .ObserveOnUIDispatcher()
                .Subscribe(async previewTask =>
                {
                    var preview = await previewTask;

                    // icon
                    if (File.Exists(preview.Raw))
                    {
                        var bitmap = new BitmapImage(new Uri(preview.Raw));
                        bitmap.Freeze();
                        IconBrush.Value = new ImageBrush(bitmap);
                    }
                    else
                    {
                        IconBrush.Value = null;
                    }

                    // nx-icon
                    if (File.Exists(preview.Nx))
                    {
                        var bitmap = new BitmapImage(new Uri(preview.Nx));
                        bitmap.Freeze();
                        NxIconBrush.Value = new ImageBrush(bitmap);
                    }
                    else
                    {
                        NxIconBrush.Value = null;
                    }
                }).AddTo(CompositeDisposable);
        }

        private async Task ConvertPreviewIcon(Title model, DisposableDirectory disposableDirectory, Project project, NspBuilder builder,
            PreviewIcon preview)
        {
            var iconFilePath = project.ToAbsolutePath(model.IconFilePath);

            preview.Raw = preview.Nx = iconFilePath;

            if (project.AppCapability.IsSupportNspIconConvert == false)
                return;

            var outputDir = Path.Combine(disposableDirectory.RootPath, model.Language.ToString());
            PrepareOutputDir(outputDir);

            var result = await builder.ConvertIconAsync(outputDir, iconFilePath).ConfigureAwait(false);
            if (result.Result != NspHandleResult.Ok)
                return;

            preview.Raw = result.RawIconPath;
            preview.Nx = result.NxIconPath;
        }

        protected override bool IsVisibled(string keyword)
        {
            if (base.IsVisibled(keyword))
                return true;

            var texts = new[]
            {
                Resources.Icon,
                Resources.NxIcon,
            };

            foreach (var text in texts)
                if (text.IndexOf(keyword, StringComparison.OrdinalIgnoreCase) != -1)
                    return true;

            return false;
        }

        private void ShowIconLargeSizePreviewWindow(Title model)
        {
            _iconLargePreviewVm?.Dispose();
            _iconLargePreviewVm = new IconActualSizePreviewWindowVm(
                Resources.Icon,
                model,
                IconBrush.ToReadOnlyReactiveProperty(),
                Title.OffDeviceIconWidth, Title.OffDeviceIconHeight
            ).AddTo(CompositeDisposable);

            Messenger.Raise(new TransitionMessage(_iconLargePreviewVm, "ShowIconPreviewWindow"));
        }

        private void ShowNxIconPreviewWindow(Title model)
        {
            _nxIconPreviewVm?.Dispose();
            _nxIconPreviewVm = new IconActualSizePreviewWindowVm(
                Resources.NxIcon,
                model,
                NxIconBrush.ToReadOnlyReactiveProperty(),
                Title.NxIconWidth, Title.NxIconHeight
            ).AddTo(CompositeDisposable);

            Messenger.Raise(new TransitionMessage(_nxIconPreviewVm, "ShowIconPreviewWindow"));
        }

        private void PrepareOutputDir(string outputDir)
        {
            if (Directory.Exists(outputDir) == false)
            {
                try
                {
                    Directory.CreateDirectory(outputDir);
                }
                catch (IOException) { } // ignore
            }
        }
    }
}
