﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Reactive.Linq;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;
using static Nintendo.Authoring.AuthoringEditor.Properties.Resources;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages
{
    public class FeaturePageVm : PageVmBase
    {
        public BoolTwoSelectionParamVm Screenshot { get; }
        public EnumParamVm VideoCapture { get; }
        public EnumParamVm RuntimeAddOnContentInstall { get; }
        public EnumParamVm CrashReport { get; }

        public FeaturePageVm(Container diContainer, Project project, ApplicationMeta appMeta)
            : base(nameof(Feature))
        {
            var isReadOnlyRp = appMeta.ToReactivePropertyAsSynchronized(x => x.IsReadOnly).AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            Screenshot = new BoolTwoSelectionParamVm(
                nameof(Screenshot_Caption),
                nameof(Screenshot_Comment),
                appMeta.Application.ToReactivePropertyAsSynchronized(x => x.IsAllowScreenshot),
                nameof(Allow),
                nameof(Deny)
            ).AddTo(CompositeDisposable);
            Screenshot.IsReadOnly = isReadOnlyRp;

            /////////////////////////////////////////////////////////

            if (IsEditableVideoCapture(diContainer, project))
            {
                VideoCapture = EnumParamVm.Factory(
                    nameof(VideoCapture_Caption),
                    nameof(VideoCapture_Comment),
                    appMeta.Application.ToReactivePropertyAsSynchronized(x => x.VideoCapture),
                    appMeta.Application.VideoCapture == VideoCaptureType.Manual
                        ? new object[]
                        {
                            VideoCaptureType.Disable,
                            VideoCaptureType.Manual,
                            VideoCaptureType.Enable
                        }
                        : new object[]
                        {
                            VideoCaptureType.Disable,
                            VideoCaptureType.Enable
                        }
                ).AddTo(CompositeDisposable);
                VideoCapture.IsReadOnly = isReadOnlyRp;

                var info = new InformationBlockParamVm(nameof(VideoCapture_IsDenyScreenshot))
                    .AddTo(CompositeDisposable);
                new[]
                {
                    appMeta.Application.ObserveProperty(x => x.VideoCapture).Select(x => x == VideoCaptureType.Enable || x == VideoCaptureType.Manual),
                    appMeta.Application.ObserveProperty(x => x.IsAllowScreenshot).Select(x => x == false)
                }.CombineLatest().Subscribe(x =>
                {
                    info.Visibility.Value = x.All(p => p)
                        ? System.Windows.Visibility.Visible
                        : System.Windows.Visibility.Collapsed;
                });

                VideoCapture.Additional = info;
            }

            /////////////////////////////////////////////////////////
            if (project.AppCapability.IsSupportRuntimeAddOnContentInstall)
            {
                RuntimeAddOnContentInstall = EnumParamVm.Factory(
                    nameof(RuntimeAddOnContentInstall_Caption),
                    nameof(RuntimeAddOnContentInstall_Comment),
                    appMeta.Application.ToReactivePropertyAsSynchronized(x => x.RuntimeAddOnContentInstall),
                    new object[]
                    {
                        RuntimeAddOnContentInstallType.Deny,
                        RuntimeAddOnContentInstallType.AllowAppend
                    }
                ).AddTo(CompositeDisposable);
                RuntimeAddOnContentInstall.IsReadOnly = isReadOnlyRp;
            }

            /////////////////////////////////////////////////////////
            if (project.AppCapability.IsSupportCrashReport)
            {
                CrashReport = EnumParamVm.Factory(
                    nameof(CrashReport_Caption),
                    nameof(CrashReport_Comment),
                    appMeta.Application.ToReactivePropertyAsSynchronized(x => x.CrashReport),
                    new object[]
                    {
                        CrashReportType.Deny,
                        CrashReportType.Allow
                    }
                ).AddTo(CompositeDisposable);
                CrashReport.IsReadOnly = isReadOnlyRp;
            }

            /////////////////////////////////////////////////////////
            Params = new ParamVm[]
            {
                Screenshot,
                VideoCapture,
                RuntimeAddOnContentInstall,
                CrashReport
            }.Where(x => x != null).ToArray();
        }

        // VideoCaptrue 項目が現在開いている アプリケーション meta/nsp において表示・編集可能なら true を返す
        // 表示・編集は次のすべての条件を満たす場合に可能になる:
        // * AuthoringEditor が属する SDK がVideoCapture をサポート
        // * アプリケーション nsp 内の XML にある <VideoCapture> 項目が Disable, Manual, Enable
        //   またはアプリケーション meta を開いている
        private static bool IsEditableVideoCapture(Container diContainer, Project project)
        {
            if (project.AppCapability.IsSupportVideoCapture == false)
                return false;
            var appMeta = project.Meta;
            var isAppNspMode = diContainer.GetInstance<AppProfile>().AppMode == AppModeType.ApplicationNsp;
            return isAppNspMode == false || IsValidVideoCaptureItem(appMeta);
        }

        // VideoCapture 項目が有効な値 (>= NX Addon 4.2.0 から利用できる値) なら true を返す
        private static bool IsValidVideoCaptureItem(ApplicationMeta appMeta)
        {
            switch (appMeta.Application.VideoCapture)
            {
                case VideoCaptureType.Disable:
                case VideoCaptureType.Manual:
                case VideoCaptureType.Enable:
                    return true;
                default:
                    return false;
            }
        }
    }
}
