﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.Linq;
using System.Reactive.Linq;
using System.Windows.Input;
using BezelEditor.Foundation.Types;
using BezelEditor.Mvvm;
using BezelEditor.Mvvm.Messages;
using Livet.Messaging.Windows;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using Disposable = System.Reactive.Disposables.Disposable;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Controls
{
    public class SaveDataSizeSimulatorWindowVm : ViewModelBase
    {
        public ReactiveCommand CloseCommand { get; }
        public ReactiveCommand SetResultAndCloseCommand { get; }
        public ReactiveCommand AddFileCommand { get; }
        public ReactiveCommand ResetCommand { get; }
        public ReactiveCommand MeasureEmulationDirectoryCommand { get; }
        public ReactiveCommand<FileData> RemoveFileCommand { get; }

        public ObservableCollection<FileData> Files { get; } = new ObservableCollection<FileData>();
        public ReactiveProperty<int> DirectoryCount { get; }

        public ReactiveProperty<string> ResultString { get; }
        public int SaveDataSize { get; private set; }

        public DialogResultType DialogResult { get; private set; } = DialogResultType.Cancel;

        public SaveDataSizeSimulatorWindowVm(ApplicationMeta appMeta)
        {
            DirectoryCount = new ReactiveProperty<int>().AddTo(CompositeDisposable);

            CloseCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            CloseCommand
                .Subscribe(_ => Close())
                .AddTo(CompositeDisposable);

            SetResultAndCloseCommand =
                appMeta.ObserveProperty(x => x.IsReadOnly)
                    .Select(x => x == false)
                    .ToReactiveCommand()
                    .AddTo(CompositeDisposable);
            SetResultAndCloseCommand
                .Subscribe(_ => SetResultAndClose())
                .AddTo(CompositeDisposable);

            AddFileCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            AddFileCommand
                .Subscribe(_ => AddFile())
                .AddTo(CompositeDisposable);

            ResetCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            ResetCommand
                .Subscribe(_ =>
                {
                    Reset();
                    UpdateResult();
                })
                .AddTo(CompositeDisposable);

            RemoveFileCommand = new ReactiveCommand<FileData>().AddTo(CompositeDisposable);
            RemoveFileCommand
                .Subscribe(RemoveFile)
                .AddTo(CompositeDisposable);

            MeasureEmulationDirectoryCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            MeasureEmulationDirectoryCommand
                .Subscribe(_ => MeasureEmulationDirectory())
                .AddTo(CompositeDisposable);

            ResultString = new ReactiveProperty<string>().AddTo(CompositeDisposable);
            UpdateResult();

            Observable
                .Merge(Files.CollectionChangedAsObservable().ToUnit())
                .Merge(Files.ObserveElementPropertyChanged().ToUnit())
                .Merge(DirectoryCount.ToUnit())
                .Throttle(TimeSpan.FromMilliseconds(250))
                .Subscribe(_ =>
                {
                    if (_isInMeasureEmulationDirectory == false)
                        UpdateResult();
                })
                .AddTo(CompositeDisposable);
        }

        private void Close()
        {
            Messenger.Raise(new WindowActionMessage(WindowAction.Close, "WindowAction"));
        }

        private void SetResultAndClose()
        {
            DialogResult = DialogResultType.Ok;
            Close();
        }

        private void RemoveFile(FileData file)
        {
            Files.Remove(file);
        }

        private void AddFile()
        {
            Files.Insert(0, new FileData(RemoveFileCommand));
        }

        private void Reset()
        {
            Files.Clear();
            DirectoryCount.Value = 0;
        }

        private bool _isInMeasureEmulationDirectory;

        private void MeasureEmulationDirectory()
        {
            using (Disposable.Create(() => _isInMeasureEmulationDirectory = false))
            {
                _isInMeasureEmulationDirectory = true;

                var r = Messenger.GetResponse(new FolderSelectionMessage(GuiConstants.MessageKey_DirectoryOpen));
                if (r?.Response == null)
                    return;

                Reset();

                var targetDir = r.Response;

                var counter = new Dictionary<int, int>();
                {
                    var sizes = FileSystemHelper.EnumerateAllFiles(targetDir)
                        .Select(x => (int) new FileInfo(x).Length)
                        .OrderByDescending(x => x);

                    foreach (var size in sizes)
                    {
                        if (counter.ContainsKey(size))
                            ++counter[size];
                        else
                            counter[size] = 1;
                    }
                }

                Files.Clear();

                foreach (var c in counter)
                    Files.Add(new FileData(RemoveFileCommand)
                        {
                            Size = c.Key,
                            Count = c.Value
                        }
                    );

                DirectoryCount.Value = FileSystemHelper.EnumerateAllDirectories(targetDir).Count();

                UpdateResult();
            }
        }

        private void UpdateResult()
        {
            var c = new SaveDataSizeCalculator();

            c.AddDirectoryEntries(DirectoryCount.Value);
            c.AddFileEntries(Files.SelectMany(x => Enumerable.Repeat(x.Size, x.Count)));

            SaveDataSize = c.SaveDataSize;

            var kib = ((decimal) SaveDataSize/1024).ToString("0.###");
            var mib = ((decimal) SaveDataSize/1024/1024).ToString("0.###");

            ResultString.Value = $"{mib}MB ({kib}KB, {SaveDataSize}Bytes)";
        }
    }

    public class FileData : ModelBase
    {
        #region Size

        private int _Size;

        public int Size
        {
            get { return _Size; }
            set { SetProperty(ref _Size, value); }
        }

        #endregion

        #region Count

        private int _Count = 1;

        public int Count
        {
            get { return _Count; }
            set { SetProperty(ref _Count, value); }
        }

        #endregion

        public ICommand RemoveFileCommand { get; }

        public FileData(ICommand removeFileCommand)
        {
            RemoveFileCommand = removeFileCommand;
        }
    }
}
