﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;

namespace Nintendo.Authoring.AuthoringEditor.Controls
{
    public class BalloonBox : Canvas
    {
        #region PlacementTarget

        public FrameworkElement PlacementTarget
        {
            get { return (FrameworkElement) GetValue(PlacementTargetProperty); }
            set { SetValue(PlacementTargetProperty, value); }
        }

        public static readonly DependencyProperty PlacementTargetProperty =
            DependencyProperty.Register(
                nameof(PlacementTarget),
                typeof(FrameworkElement),
                typeof(BalloonBox),
                new FrameworkPropertyMetadata
                {
                    PropertyChangedCallback = OnPlacementTargetChanged,
                    DefaultValue = default(FrameworkElement),
                    BindsTwoWayByDefault = true
                }
            );

        private static void OnPlacementTargetChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            Debug.Assert(d is BalloonBox);
            var self = (BalloonBox) d;

            var oldFe = e.OldValue as FrameworkElement;
            var newFe = e.NewValue as FrameworkElement;

            if (oldFe != null)
                oldFe.SizeChanged -= self.OnSizeChanged;

            if (newFe != null)
                newFe.SizeChanged += self.OnSizeChanged;

            self._placementTargetControl = newFe;
            self.UpdateLocation();
        }


        #endregion

        public enum PlacementMode
        {
            Top,
            Bottom
        }

        #region Placement

        public PlacementMode Placement
        {
            get { return (PlacementMode) GetValue(PlacementProperty); }
            set { SetValue(PlacementProperty, value); }
        }

        public static readonly DependencyProperty PlacementProperty =
            DependencyProperty.Register(
                nameof(Placement),
                typeof(PlacementMode),
                typeof(BalloonBox),
                new FrameworkPropertyMetadata
                {
                    PropertyChangedCallback = OnPlacementChanged,
                    DefaultValue = PlacementMode.Bottom,
                    BindsTwoWayByDefault = true
                }
            );

        private static void OnPlacementChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            Debug.Assert(d is BalloonBox);
            var self = (BalloonBox) d;

            self.UpdateLocation();
        }

        #endregion

        #region Content

        public object Content
        {
            get { return GetValue(ContentProperty); }
            set { SetValue(ContentProperty, value); }
        }

        public static readonly DependencyProperty ContentProperty =
            DependencyProperty.Register(
                nameof (Content),
                typeof (object),
                typeof (BalloonBox),
                new FrameworkPropertyMetadata
                {
                    DefaultValue            = default(object),
                    BindsTwoWayByDefault    = true
                }
            );

        #endregion

        #region ContentLabelStyle

        public Style ContentLabelStyle
        {
            get { return (Style)GetValue(ContentLabelStyleProperty); }
            set { SetValue(ContentLabelStyleProperty, value); }
        }

        public static readonly DependencyProperty ContentLabelStyleProperty =
            DependencyProperty.Register(
                nameof (ContentLabelStyle),
                typeof (Style),
                typeof (BalloonBox),
                new FrameworkPropertyMetadata
                {
                    DefaultValue            = default(Style),
                    BindsTwoWayByDefault    = true
                }
            );

        #endregion

        #region Sinking

        public double Sinking
        {
            get { return (double)GetValue(SinkingProperty); }
            set { SetValue(SinkingProperty, value); }
        }

        public static readonly DependencyProperty SinkingProperty =
            DependencyProperty.Register(
                nameof (Sinking),
                typeof (double),
                typeof (BalloonBox),
                new FrameworkPropertyMetadata
                {
                    DefaultValue            = default(double),
                    BindsTwoWayByDefault    = true
                }
            );

        #endregion

        private FrameworkElement _placementTargetControl;
        private FrameworkElement _childControl;

        public BalloonBox()
        {
            SetZIndex(this, 1);

            Loaded += (_, __) =>
            {
               Children.Add(new Label {Style = ContentLabelStyle});
            };

            Unloaded += (_, __) =>
            {
                if (_placementTargetControl != null)
                    _placementTargetControl.SizeChanged -= OnSizeChanged;

                if (_childControl != null)
                {
                    _childControl.SizeChanged -= OnSizeChanged;
                    _childControl.MouseEnter -= OnMouseEnter;
                }
            };
        }

        protected override void OnVisualChildrenChanged(DependencyObject newContent, DependencyObject oldContent)
        {
            base.OnVisualChildrenChanged(newContent, oldContent);

            var oldFe = oldContent as FrameworkElement;
            var newFe = newContent as FrameworkElement;

            if (oldFe != null)
            {
                oldFe.SizeChanged -= OnSizeChanged;
                oldFe.MouseEnter -= OnMouseEnter;
            }

            if (newFe != null)
            {
                newFe.SizeChanged += OnSizeChanged;
                newFe.MouseEnter += OnMouseEnter;
            }

            _childControl = newFe;

            UpdateLocation();
        }

        private void OnMouseEnter(object sender, MouseEventArgs mouseEventArgs)
        {
            Placement =
                Placement == PlacementMode.Bottom
                    ? PlacementMode.Top
                    : PlacementMode.Bottom;
        }

        private void OnSizeChanged(object sender, SizeChangedEventArgs sizeChangedEventArgs)
        {
            UpdateLocation();
        }

        private void UpdateLocation()
        {
            double bottom = 0;

            if (Children.Count == 0)
                return;

            var child = Children[0];

            if (child is FrameworkElement)
            {
                var f = child as FrameworkElement;
                bottom = (f.ActualHeight + f.Margin.Top + f.Margin.Bottom)*-1;
            }

            switch (Placement)
            {
                case PlacementMode.Top:
                {
                    if (PlacementTarget == null)
                    {
                        Margin = new Thickness(0, 0, 0, bottom);
                        break;
                    }

                    const double topMargin = 4;

                    var h = PlacementTarget.ActualHeight + PlacementTarget.Margin.Top + PlacementTarget.Margin.Bottom -
                            bottom + topMargin;

                    Margin = new Thickness(0, -h + Sinking + 1, 0, bottom);
                    break;
                }

                case PlacementMode.Bottom:
                {
                    Margin = new Thickness(0, -Sinking, 0, bottom);
                    break;
                }

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }
    }
}
