﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Xml.Linq;
using System.Xml.Serialization;
using BezelEditor.Foundation;

namespace Nintendo.Authoring.AuthoringEditor.Foundation
{
    public class NspExtractedDirectoryEnumerable : INspFileEnumerable
    {
        public PathString NspPath { get; }
        public NspFileEnumeration EnumerationMode { get; }

        public NspExtractedDirectoryEnumerable(INspFile nspFile, NspFileEnumerateParameter parameter)
        {
            NspPath = nspFile.NspPath;
            EnumerationMode = parameter.EnumerationType;
        }

        IEnumerator IEnumerable.GetEnumerator()
        {
            return GetEnumerator();
        }

        public IEnumerator<NspFileEntry> GetEnumerator()
        {
            var nspPath = NspPath.ToString();

            return Directory.EnumerateFiles(nspPath, "*.*", ToSearchOption())
                .Select(x =>
                {
                    var fileInfo = new FileInfo(x);
                    return new NspFileEntry
                    {
                        FilePath = x.Substring(nspPath.Length),
                        FileSize = (ulong)fileInfo.Length
                    };
                })
                .GetEnumerator();
        }

        private SearchOption ToSearchOption()
        {
            switch (EnumerationMode)
            {
                case NspFileEnumeration.All:
                    return SearchOption.AllDirectories;
                case NspFileEnumeration.RootOnly:
                    return SearchOption.TopDirectoryOnly;
                default:
                    throw new ArgumentException(nameof(EnumerationMode));
            }
        }
    }

    public class NspExtractedDirectory : INspFile
    {
        public NspFileEnumeration EnumerationMode { get; set; }

        private PathString _NspPath;

        public PathString NspPath
        {
            get { return _NspPath; }
            set { _NspPath = value.DirectoryWithoutLastSeparator + Path.DirectorySeparatorChar; }
        }
        public INspFile OriginalNspFile { get; set; }
        public bool IsExists => Directory.Exists(NspPath);
        public bool IsAuthoringToolOperational => false;

        public NspExtractedDirectory(string nspPath)
        {
            NspPath = nspPath;
        }

        public INspFileEnumerable Files(NspFileEnumerateParameter parameter) => new NspExtractedDirectoryEnumerable(this, parameter);

        public string ReadAllText(string pathInNsp, Encoding encoding)
            => File.ReadAllText(Path.Combine(NspPath, pathInNsp), encoding);

        public string ReadAllText(string pathInNsp) => ReadAllText(pathInNsp, Encoding.UTF8);

        public Task<string> ReadAllTextAsync(string pathInNsp, Encoding encoding) =>
            Task.FromResult(ReadAllText(pathInNsp, encoding));

        public Task<string> ReadAllTextAsync(string pathInNsp) =>
            ReadAllTextAsync(pathInNsp, Encoding.UTF8);

        public Task<T> ReadXmlAsync<T>(string filePath, Encoding encoding)
        {
            var xmlText = ReadAllText(filePath, encoding);
            var serializer = new XmlSerializer(typeof(T));
            using (var reader = new StringReader(xmlText))
            {
                return Task.FromResult((T) serializer.Deserialize(reader));
            }
        }

        public Task<T> ReadXmlAsync<T>(string filePath) => ReadXmlAsync<T>(filePath, Encoding.UTF8);

        public Task ExtractAsync(string pathInNsp, string path) =>
            Task.Run(() => File.Copy(Path.Combine(NspPath, pathInNsp), path, true));

        public Task ExtractFilesAsync(IDictionary<string, string> targets)
        {
            return Task.Run(() =>
            {
                var files = Directory.EnumerateFiles(NspPath, "*.*", SearchOption.AllDirectories);
                foreach (var p in files)
                {
                    var outputSubPath = p.Substring(NspPath.ToString().Length)
                        .Replace(Path.DirectorySeparatorChar, Path.AltDirectorySeparatorChar);
                    string outputPath;
                    if (targets.TryGetValue(outputSubPath, out outputPath))
                    {
                        var outputDirectory = Path.GetDirectoryName(outputPath);
                        if (string.IsNullOrEmpty(outputDirectory) == false && Directory.Exists(outputDirectory) == false)
                            Directory.CreateDirectory(outputDirectory);
                        File.Copy(p, outputPath, true);
                    }
                }
            });
        }

        public Task ExtractContentAsync(ContentType contentType, string path)
        {
            return Task.Run(() =>
            {
                var files = Directory.EnumerateFiles(NspPath, "*.cnmt.xml", SearchOption.TopDirectoryOnly);
                foreach (var file in files)
                {
                    var contentMetaXml = XDocument.Parse(File.ReadAllText(file, Encoding.UTF8));
                    var contentElement = contentMetaXml.Root?
                        .Elements("Content")
                        .FirstOrDefault(x => x.Element("Type")?.Value == contentType.ToString());
                    var contentId = contentElement?.Element("Id")?.Value;
                    if (string.IsNullOrEmpty(contentId))
                        throw new NspFileReadException(NspPath, $"File not found: {contentId}.nca ({contentType})");
                    var contentNcaPath = Path.Combine(NspPath, $"{contentId}.nca");
                    if (File.Exists(contentNcaPath))
                        Copy(contentNcaPath, path);
                }
            });
        }

        public Task ExtractAllAsync(PathString outputDir, Action<int> progressChanged,
            CancellationToken cancellationToken)
        {
            return Task.Run(() =>
            {
                var files = Directory.EnumerateFiles(NspPath, "*.*", SearchOption.AllDirectories);
                foreach (var p in files)
                {
                    if (cancellationToken.IsCancellationRequested)
                        return;
                    Copy(p, outputDir);
                }
            });
        }

        public NcaCompareResult GetNcaIdenticalResult(INspFile target, string contentType, string prefixPath = null,
            Action<float> onProgressChange = null, CancellationToken token = default(CancellationToken))
        {
            var targetNsp = target as NspExtractedDirectory;
            if (targetNsp == null)
                throw new ArgumentException(nameof(target));

            var sourceNcaPath = GetComparedNcaPath(contentType);
            var targetNcaPath = targetNsp.GetComparedNcaPath(contentType);

            if (sourceNcaPath == null && targetNcaPath == null)
                return NcaCompareResult.Identical;

            {
                var flags = NcaCompareResult.ContainsDifference;

                if (sourceNcaPath != null)
                    flags |= NcaCompareResult.SourceNcaExists;
                if (targetNcaPath != null)
                    flags |= NcaCompareResult.TargetNcaExists;

                if (sourceNcaPath == null || targetNcaPath == null)
                    return flags;
            }

            var resultFlags = NcaCompareResult.SourceNcaExists | NcaCompareResult.TargetNcaExists;

            if (string.IsNullOrEmpty(prefixPath))
            {
                resultFlags |= FileSystemHelper.IsEqualDirectory(sourceNcaPath, targetNcaPath)
                    ? NcaCompareResult.Identical
                    : NcaCompareResult.ContainsDifference;
            }
            else
            {
                resultFlags |= FileSystemHelper.IsEqualDirectory(sourceNcaPath, targetNcaPath,
                    x => x.StartsWith(prefixPath))
                    ? NcaCompareResult.Identical
                    : NcaCompareResult.ContainsDifference;
            }

            onProgressChange?.Invoke(100.0f);

            return resultFlags;
        }

        private string GetComparedNcaPath(string contentType)
        {
            var contentMetaXmlPath = Directory.EnumerateFiles(NspPath, "*.cnmt.xml", SearchOption.TopDirectoryOnly).FirstOrDefault();
            if (File.Exists(contentMetaXmlPath) == false)
                return null;
            var xml = XDocument.Parse(File.ReadAllText(contentMetaXmlPath));
            var contentElement = xml.Root?
                .Elements("Content")
                .FirstOrDefault(x => x.Element("Type")?.Value == contentType);
            var contentId = contentElement?.Element("Id")?.Value;
            if (contentId == null)
                return null;
            return Path.Combine(NspPath, $"{contentId}.nca");
        }

        private void Copy(string sourcePath, string outputDirectory)
        {
            var outputSubPath = sourcePath.Substring(NspPath.ToString().Length);
            var outputPath = Path.Combine(outputDirectory, outputSubPath);
            var outputDir = Path.GetDirectoryName(outputPath);
            if (Directory.Exists(outputDir) == false)
                Directory.CreateDirectory(outputDir);
            File.Copy(sourcePath, outputPath, true);
        }
    }
}
