﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Threading.Tasks;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    public class NspExtraResourceImporter : INspExtraResourceImporter
    {
        private readonly string _tempDir;
        private readonly ContentMeta _contentMeta;
        private readonly INspFile _nspFile;
        private readonly Application _application;

        public NspExtraResourceImporter(Application application, string tempDir, INspFile nspFile, ContentMeta contentMeta)
        {
            _application = application;
            _tempDir = tempDir;
            _nspFile = nspFile;
            _contentMeta = contentMeta;
        }

        public async Task<bool> ReadHtmlDocumentAsync()
        {
            var documentPath = await ExtractResourcesAsync(ContentType.HtmlDocument).ConfigureAwait(false);
            if (documentPath == null)
                return false;

            var app = _application;

            var htmlDocumentDir = Path.Combine(documentPath, @"fs0\html-document");
            if (Directory.Exists(htmlDocumentDir))
            {
                app.OriginalHtmlDocumentPath = new ExpandablePath(htmlDocumentDir);
            }

            var accessibleUrlsDir = Path.Combine(documentPath, @"fs0\accessible-urls");
            if (Directory.Exists(accessibleUrlsDir))
            {
                var extractedAccesibleUrlFile = Directory.EnumerateFiles(accessibleUrlsDir, "*.*", SearchOption.TopDirectoryOnly).FirstOrDefault();
                app.OriginalAccessibleUrlsFilePath = new ExpandablePath(extractedAccesibleUrlFile);
            }

            return true;
        }

        public async Task<bool> ReadLegalInformationAsync()
        {
            var legalInfoDirPath = await ExtractResourcesAsync(ContentType.LegalInformation).ConfigureAwait(false);
            if (legalInfoDirPath == null)
                return false;

            _application.OriginalLegalInformationPath = new ExpandablePath(Path.Combine(legalInfoDirPath, "fs0"));

            return true;
        }

        private async Task<string> ExtractResourcesAsync(ContentType contentType)
        {
            Debug.Assert(Directory.Exists(_tempDir));
            var contentId = _contentMeta.GetContentId(contentType);
            if (contentId == null)
                return null;
            try
            {
                await _nspFile.ExtractContentAsync(contentType, _tempDir).ConfigureAwait(false);
                return Path.Combine(_tempDir, $"{contentId}.nca");
            }
            catch
            {
                return null; // 展開に失敗した
            }
        }
    }
}
