﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using BezelEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.Foundation;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    public enum NspTypes
    {
        Application,
        SystemProgram
    }

    public enum OutputTypes
    {
        Nsp
    }

    public class NspBuilder
    {
        #region MakePatch

        public class PatchResultType : NspHandleResultType
        {
            public ApplicationContentMeta Original { get; set; }
            public ApplicationContentMeta Current { get; set; }
            public PatchContentMeta Previous { get; set; }
        }

        public class MakePatchOption
        {
            public string OutputPath { get; set; }
            public string InputDescFilePath { get; set; } = NintendoSdkHelper.ApplicationDescFilePath;
            public string OriginalNspFilePath { get; set; }
            public string CurrentNspFilePath { get; set; }
            public string PreviousNspFilePath { get; set; }

            public Action<int> ProgressChanged { get; set; }

            public CancellationToken CancellationToken { get; set; }
        }

        public async Task<PatchResultType> MakePatchNspAsync(MakePatchOption option)
        {
            var result = await TestMakeNspPatchError(option).ConfigureAwait(false);
            if (result.Result != NspHandleResult.Ok)
                return result;

            var args = GetMakeNspPatchArguments(option);
            using (var job = AuthoringToolWrapper.Run(args, option.ProgressChanged))
            {
                NspHandleResultType r;
                try
                {
                    r = await job.WaitForExitAsync(x => x.MakeResultType(), option.CancellationToken).ConfigureAwait(false);
                }
                catch (TaskCanceledException)
                {
                    r = job.MakeResultType(NspHandleResult.Canceled);
                }
                return new PatchResultType()
                {
                    Result = r.Result,
                    ErrorMessages = r.ErrorMessages,
                    Original = result.Original,
                    Current = result.Current,
                    Previous = result.Previous
                };
            }
        }

        public static async Task<PatchResultType> TestMakeNspPatchError(MakePatchOption option)
        {
            // エラーがない場合に返す Result
            var okResult = new PatchResultType { Result = NspHandleResult.Ok };

            if (File.Exists(AuthoringToolHelper.AuthoringToolExe) == false)
                return new PatchResultType { Result = NspHandleResult.NotFoundAuthoringToolExe};

            if (File.Exists(Environment.ExpandEnvironmentVariables(option.InputDescFilePath ?? string.Empty)) == false)
                return new PatchResultType { Result = NspHandleResult.NotFoundDescFile};

            {
                var r = await TestPatchOriginalNspError(option).ConfigureAwait(false);
                if (r.Result != NspHandleResult.Ok)
                    return r;
                okResult.Original = r.Original;
            }

            {
                var r = await TestPatchCurrentNspError(option).ConfigureAwait(false);
                if (r.Result != NspHandleResult.Ok)
                {
                    r.Original = okResult.Original;
                    return r;
                }
                okResult.Current = r.Current;
            }

            if (okResult.Original.Id != okResult.Current.Id)
            {
                return new PatchResultType
                {
                    Result = NspHandleResult.NspIsNotSameAppIdBetweenOriginalAndRevisedError,
                    ErrorMessages = new[]
                    {
                        "Original application ID:",
                        $"{okResult.Original.Id.ToHex()}",
                        "Revised application ID:",
                        $"{okResult.Current.Id.ToHex()}"
                    },
                    Original = okResult.Original,
                    Current = okResult.Current
                };
            }

            {
                var r = await TestPatchPreviousNspError(option).ConfigureAwait(false);
                if (r.Result != NspHandleResult.Ok)
                {
                    r.Original = okResult.Original;
                    r.Current = okResult.Current;
                    return r;
                }
                okResult.Previous = r.Previous;

                var patchError = NspHandleResult.Ok;

                if (okResult.Previous != null)
                {
                    if (okResult.Original.Id != okResult.Previous.ApplicationId)
                        patchError = NspHandleResult.NspIsNotSameAppIdBetweenOrigianlAndPreviousError;
                    else if (okResult.Original.Digest != okResult.Previous.OriginalApplication?.Digest)
                        patchError = NspHandleResult.NspIsNotSameAppDigestBetweenOriginalAndPreviousError;
                }

                if (patchError != NspHandleResult.Ok)
                {
                    var historicallyOriginal = okResult.Previous.OriginalApplication;
                    return new PatchResultType
                    {
                        Result = patchError,
                        ErrorMessages = new[]
                        {
                            "[Original Application]",
                            "Application ID:",
                            $"{okResult.Original.Id.ToHex()}",
                            "ROM identification hash:",
                            $"{okResult.Original.Digest}",
                            "----",
                            "[Previous Patch]",
                            "Original application ID:",
                            $"{historicallyOriginal?.Id.ToHex()}",
                            "Original application ROM identification hash:",
                            $"{historicallyOriginal?.Digest}"
                        },
                        //
                        Original = okResult.Original,
                        Current = okResult.Current,
                        Previous = r.Previous
                    };
                }
            }

            return okResult;
        }

        public static async Task<PatchResultType> TestPatchCurrentNspError(MakePatchOption option)
        {
            var currentNspFilePath = Environment.ExpandEnvironmentVariables(option.CurrentNspFilePath ?? string.Empty);

            var currentNspFile = Project.GetNspFile(currentNspFilePath);
            if (currentNspFile == null)
                return new PatchResultType { Result = NspHandleResult.NotFoundNspFile };

            var currentNspType = await NspImporter.DetectContentMetaTypeAsync(currentNspFile).ConfigureAwait(false);
            if (currentNspType != ContentMetaType.Application)
                return new PatchResultType {Result = NspHandleResult.NspIsNotApplicationError};

            var appContentMetas =
                await NspImporter.ReadAllContentMeta<ApplicationContentMeta>(currentNspFile).WhenAll().ConfigureAwait(false);
            var appContentMeta = appContentMetas.FirstOrDefault();
            if (appContentMeta == null)
                return new PatchResultType {Result = NspHandleResult.ExtractError, ErrorMessages = new [] {"File not found: *.cnmt.xml"}};

            return new PatchResultType
            {
                Result =
                    appContentMeta.ReleaseVersion == 0
                        ? NspHandleResult.NspIsNotRevisedVersionError
                        : NspHandleResult.Ok,
                Current = appContentMeta
            };
        }

        public static async Task<PatchResultType> TestPatchPreviousNspError(MakePatchOption option)
        {
            var previousNspFilePath = Environment.ExpandEnvironmentVariables(option.PreviousNspFilePath ?? string.Empty);
            if (string.IsNullOrEmpty(previousNspFilePath))
                return new PatchResultType {Result = NspHandleResult.Ok};

            var previousNspFile = Project.GetNspFile(previousNspFilePath);
            if (previousNspFile == null)
                return new PatchResultType { Result = NspHandleResult.NotFoundNspFile };

            var previousNspFileType = await NspImporter.DetectContentMetaTypeAsync(previousNspFile).ConfigureAwait(false);
            if (previousNspFileType != ContentMetaType.Patch)
                return new PatchResultType {Result = NspHandleResult.NspIsNotPatchError };

            var patchContentMetas = await NspImporter.ReadAllContentMeta<PatchContentMeta>(previousNspFile).WhenAll().ConfigureAwait(false);
            var patchContentMeta = patchContentMetas.FirstOrDefault();
            if (patchContentMeta == null)
                return new PatchResultType {Result = NspHandleResult.ExtractError, ErrorMessages = new[] { "File not found: *.cnmt.xml" } };

            if (patchContentMeta.OriginalApplication == null)
            {
                return new PatchResultType()
                {
                    Result = NspHandleResult.PatchIsNotContainsOriginalApplicationError,
                    Previous = patchContentMeta
                };
            }

            return new PatchResultType
            {
                Result = NspHandleResult.Ok,
                Previous = patchContentMeta
            };
        }

        public static async Task<PatchResultType> TestPatchOriginalNspError(MakePatchOption option)
        {
            var originalNspFilePath = Environment.ExpandEnvironmentVariables(option.OriginalNspFilePath ?? string.Empty);

            var originalNspFile = Project.GetNspFile(originalNspFilePath);
            if (originalNspFile == null)
                return new PatchResultType { Result = NspHandleResult.NotFoundNspFile };

            var originalNspType = await NspImporter.DetectContentMetaTypeAsync(originalNspFile).ConfigureAwait(false);
            if (originalNspType != ContentMetaType.Application)
                return new PatchResultType { Result = NspHandleResult.NspIsNotApplicationError };

            var appContentMetas =
                await NspImporter.ReadAllContentMeta<ApplicationContentMeta>(originalNspFile).WhenAll().ConfigureAwait(false);
            var appContentMeta = appContentMetas.FirstOrDefault();
            if (appContentMeta == null)
                return new PatchResultType { Result = NspHandleResult.ExtractError, ErrorMessages = new[] { "File not found: *.cnmt.xml" } };

            return new PatchResultType
            {
                Result =
                    appContentMeta.ReleaseVersion != 0
                        ? NspHandleResult.NspIsNotOriginalVersionError
                        : NspHandleResult.Ok,
                Original = appContentMeta
            };
        }

        public static async Task<NspHandleResult> IsValidPatchAndOriginalApplication(string patchNspPath, string originalAppNspPath)
        {
            ApplicationContentMeta appContentMeta;
            {
                var r = await TestPatchOriginalNspError(new MakePatchOption
                {
                    OriginalNspFilePath = originalAppNspPath
                }).ConfigureAwait(false);
                if (r.Result != NspHandleResult.Ok)
                    return r.Result;
                appContentMeta = r.Original;
            }

            {
                var r = await TestPatchPreviousNspError(new MakePatchOption
                {
                    PreviousNspFilePath = patchNspPath
                }).ConfigureAwait(false);
                if (r.Result != NspHandleResult.Ok)
                    return r.Result;
                if (r.Previous.ApplicationId != appContentMeta.Id)
                    return NspHandleResult.NspIsNotSameAppIdBetweenOrigianlAndPreviousError;
                if (r.Previous.OriginalApplication?.Digest != appContentMeta.Digest)
                    return NspHandleResult.NspIsNotSameAppDigestBetweenOriginalAndPreviousError;
            }
            return NspHandleResult.Ok;
        }

        private static string GetMakeNspPatchArguments(MakePatchOption option)
        {
            var outputPath = Environment.ExpandEnvironmentVariables(option.OutputPath).ToPathString();
            var descFilePath = Environment.ExpandEnvironmentVariables(option.InputDescFilePath).ToPathString();
            var originalNspFilePath = Environment.ExpandEnvironmentVariables(option.OriginalNspFilePath).ToPathString();
            var currentNspFilePath = Environment.ExpandEnvironmentVariables(option.CurrentNspFilePath).ToPathString();

            var args = $"makepatch -v -o {outputPath.SurroundDoubleQuotes}" +
                $" --desc {descFilePath.SurroundDoubleQuotes}" +
                $" --original {originalNspFilePath.SurroundDoubleQuotes}" +
                $" --current {currentNspFilePath.SurroundDoubleQuotes}";

            var previousNspFilePath = Environment.ExpandEnvironmentVariables(option.PreviousNspFilePath ?? string.Empty);
            if (string.IsNullOrWhiteSpace(previousNspFilePath) == false)
                args += $" --previous {previousNspFilePath.ToPathString().SurroundDoubleQuotes}";

            return args;
        }

        #endregion

        #region MakeAoc

        public class MakeAocOption
        {
            public string OutputPath { get; set; }
            public string InputDescFilePath { get; set; } = NintendoSdkHelper.ApplicationDescFilePath;
            public Project Project { get; set; }
            public Action<int> ProgressChanged { get; set; }
            public CancellationToken CancellationToken { get; set; }
        }

        public async Task<AocResultType> MakeAocNspAsync(MakeAocOption option)
        {
            using (var tempDir = new DisposableDirectory())
            using (var clonedProject = option.Project.DeepClone())
            {
                // 1.AoCのデータパスをファイルパスをテンポラリディレクトリベースに書き換え
                clonedProject.ProjectDirectory = option.Project.ProjectDirectory;
                foreach (var content in clonedProject.AocMeta.Contents)
                {
                    content.DataPath.Path = clonedProject.ToAbsolutePath(content.DataPath.Path);
                }

                // 2.テンポラリディレクトリにAocMetaを出力
                var aocMetaFilePath = Path.Combine(tempDir.RootPath, "aoc.nmeta").ToPathString();
                clonedProject.OutputAocMetaXmlFileForAuthoringTool(aocMetaFilePath);

                // 3.そのAocMetaをビルド
                var args = $"creatensp -v -o {option.OutputPath.ToPathString().SurroundDoubleQuotes} " +
                           $"--type AddOnContent --meta {aocMetaFilePath.SurroundDoubleQuotes}";
                using (var job = AuthoringToolWrapper.Run(args, option.ProgressChanged))
                {
                    NspHandleResultType r;
                    try
                    {
                        r = await job.WaitForExitAsync(x => x.MakeResultType(), option.CancellationToken).ConfigureAwait(false);
                    }
                    catch (TaskCanceledException)
                    {
                        r = job.MakeResultType(NspHandleResult.Canceled);
                    }
                    return new AocResultType
                    {
                        Result = r.Result,
                        ErrorMessages = r.ErrorMessages,
                    };
                }
            }
       }

        public class AocResultType : NspHandleResultType
        {
        }

        #endregion

        #region Export

        public class ExportOption
        {
            public string OutputPath { get; set; }
            public string InputMetaFilePath { get; set; }
            public string InputProgramCodePath { get; set; }
            public string InputProgramDataPath { get; set; }
            public string InputDescFilePath { get; set; } = NintendoSdkHelper.ApplicationDescFilePath;

            public NspTypes NspType { get; set; } = NspTypes.Application;
            public OutputTypes OutputType { get; set; } = OutputTypes.Nsp;

            public Action<int> ProgressChanged { get; set; }

            public CancellationToken CancellationToken { get; set; }
        }

        public async Task<NspHandleResultType> ExportNspAsync(ExportOption option)
        {
            var result = TestNspExportError(option);
            if (result != null)
                return result;

            var args = GetExportNspArguments(option);

            using (var job = AuthoringToolWrapper.Run(args, option.ProgressChanged))
            {
                try
                {
                    return await job.WaitForExitAsync(x => x.MakeResultType(), option.CancellationToken).ConfigureAwait(false);
                }
                catch (TaskCanceledException)
                {
                    return job.MakeResultType(NspHandleResult.Canceled);
                }
            }
        }

        private static NspHandleResultType TestNspExportError(ExportOption option)
        {
            if (File.Exists(AuthoringToolHelper.AuthoringToolExe) == false)
                return new NspHandleResultType {Result = NspHandleResult.NotFoundAuthoringToolExe};

            if (File.Exists(Environment.ExpandEnvironmentVariables(option.InputMetaFilePath ?? string.Empty)) == false)
                return new NspHandleResultType {Result = NspHandleResult.NotFoundMetaFile};

            if (File.Exists(Environment.ExpandEnvironmentVariables(option.InputDescFilePath ?? string.Empty)) == false)
                return new NspHandleResultType {Result = NspHandleResult.NotFoundDescFile};

            if (Directory.Exists(Environment.ExpandEnvironmentVariables(option.InputProgramCodePath ?? string.Empty)) ==
                false)
                return new NspHandleResultType {Result = NspHandleResult.NotFoundProgramCodeFile};

            if (string.IsNullOrEmpty(option.InputProgramDataPath) == false)
            {
                var path = Environment.ExpandEnvironmentVariables(option.InputProgramDataPath ?? string.Empty);

                if (Directory.Exists(path) == false)
                    return new NspHandleResultType {Result = NspHandleResult.NotFoundProgramDataFile};
            }

            return null;
        }

        private static string GetExportNspArguments(ExportOption option)
        {
            var outputPath = Environment.ExpandEnvironmentVariables(option.OutputPath).ToPathString();
            var metaFilePath = Environment.ExpandEnvironmentVariables(option.InputMetaFilePath).ToPathString();
            var descFilePath = Environment.ExpandEnvironmentVariables(option.InputDescFilePath).ToPathString();
            var programPath =
                Environment.ExpandEnvironmentVariables(option.InputProgramCodePath)
                    .ToPathString()
                    .DirectoryWithoutLastSeparator.ToPathString();
            var dataPath = option.InputProgramDataPath == null
                ? null
                : Environment.ExpandEnvironmentVariables(option.InputProgramDataPath)
                    .ToPathString()
                    .DirectoryWithoutLastSeparator
                    .ToPathString();

            var args =
                $"create{option.OutputType.ToString().ToLower()}" +
                " -v" +
                $" -o {outputPath.SurroundDoubleQuotes}" +
                $" --meta {metaFilePath.SurroundDoubleQuotes}" +
                $" --type {option.NspType}" +
                $" --desc {descFilePath.SurroundDoubleQuotes}" +
                $" --program {programPath.SurroundDoubleQuotes}";

            if (string.IsNullOrEmpty(dataPath) == false)
                args +=  $" {dataPath.SurroundDoubleQuotes}";

            return args;
        }

        #endregion

        #region ConvertIcon

        public class ConvertIconResultType : NspHandleResultType
        {
            public PathString RawIconPath { get; set; }
            public PathString NxIconPath { get; set; }
        }

        public async Task<ConvertIconResultType> ConvertIconAsync(PathString outputDir, PathString bitmap)
        {
            var outputPath = outputDir.DirectoryWithoutLastSeparator.ToPathString();
            var args = $"converticon {bitmap.SurroundDoubleQuotes} -o {outputPath.SurroundDoubleQuotes}";
            using (var job = AuthoringToolWrapper.Create(args))
            {
                job.IsRedirectStandardError = true;
                job.Start();
                await job.WaitForExitAsync().ConfigureAwait(false);
                if (job.ExitCode != 0)
                {
                    return new ConvertIconResultType
                    {
                        Result = NspHandleResult.Error,
                        ErrorMessages = job.StandardError.ToArray()
                    };
                }
            }
            var basename = bitmap.FileNameWithoutExtension;
            return new ConvertIconResultType()
            {
                Result = NspHandleResult.Ok,
                RawIconPath = Path.Combine(outputDir, $"{basename}.raw.jpg"),
                NxIconPath = Path.Combine(outputDir, $"{basename}.nx.jpg")
            };
        }

        #endregion
    }
}
