﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.Windows.Shapes;

namespace SpatialAudioEvaluationTool
{
    public partial class WaveformViewer : UserControl
    {
        public static readonly DependencyProperty LabelProperty = DependencyProperty.Register(
            nameof(Label),
            typeof(string),
            typeof(WaveformViewer),
            new PropertyMetadata(
                string.Empty,
                (sender, e) => (sender as WaveformViewer).OnPropertyChanged(sender, e)));

        public string Label
        {
            get { return (string)GetValue(LabelProperty); }
            set { SetValue(LabelProperty, value); }
        }

        public static readonly DependencyProperty ValueProperty = DependencyProperty.Register(
            nameof(Value),
            typeof(double[]),
            typeof(WaveformViewer),
            new PropertyMetadata(
                null,
                (sender, e) => (sender as WaveformViewer).OnPropertyChanged(sender, e)));

        public double[] Value
        {
            get { return (double[])GetValue(ValueProperty); }
            set { SetValue(ValueProperty, value); }
        }
        public static readonly DependencyProperty MaxValueProperty = DependencyProperty.Register(
            nameof(MaxValue),
            typeof(double),
            typeof(WaveformViewer),
            new PropertyMetadata(
                1.0,
                (sender, e) => (sender as WaveformViewer).OnPropertyChanged(sender, e)));

        public double MaxValue
        {
            get { return (double)GetValue(MaxValueProperty); }
            set { SetValue(MaxValueProperty, value); }
        }

        public static readonly DependencyProperty MinValueProperty = DependencyProperty.Register(
            nameof(MinValue),
            typeof(double),
            typeof(WaveformViewer),
            new PropertyMetadata(
                -1.0,
                (sender, e) => (sender as WaveformViewer).OnPropertyChanged(sender, e)));

        public double MinValue
        {
            get { return (double)GetValue(MinValueProperty); }
            set { SetValue(MinValueProperty, value); }
        }

        public static readonly DependencyProperty NormalBrushProperty = DependencyProperty.Register(
            nameof(NormalBrush),
            typeof(Brush),
            typeof(WaveformViewer),
            new PropertyMetadata(
                Brushes.LightGreen,
                (sender, e) => (sender as WaveformViewer).OnPropertyChanged(sender, e)));

        public Brush NormalBrush
        {
            get { return (Brush)GetValue(NormalBrushProperty); }
            set { SetValue(NormalBrushProperty, value); }
        }

        public static readonly DependencyProperty SilenceBrushProperty = DependencyProperty.Register(
            nameof(SilenceBrush),
            typeof(Brush),
            typeof(WaveformViewer),
            new PropertyMetadata(
                Brushes.LightGreen,
                (sender, e) => (sender as WaveformViewer).OnPropertyChanged(sender, e)));

        public Brush SilenceBrush
        {
            get { return (Brush)GetValue(SilenceBrushProperty); }
            set { SetValue(SilenceBrushProperty, value); }
        }

        private void OnPropertyChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e)
        {
            Update();
        }

        protected override void OnRender(DrawingContext dc)
        {
            Update();
        }

        private void Update()
        {
            var width = canvas.ActualWidth;
            var height = canvas.ActualHeight;

            var background = Brushes.Black;
            var foreground = Brushes.Gray;

            var canvasElementList = new List<UIElement>();

            {
                var rect = new Rectangle();
                rect.Width = width;
                rect.Height = height;
                rect.Fill = Background;
                canvasElementList.Add(rect);
            }

            {
                var zeroLine = new Polyline();
                zeroLine.Points.Add(new Point(0, height / 2));
                zeroLine.Points.Add(new Point(width, height / 2));
                zeroLine.Stroke = Foreground;
                zeroLine.StrokeThickness = 2;
                canvasElementList.Add(zeroLine);
            }

            if (Value != null && Value.Length > 0)
            {
                var valueLine = new Polyline();
                var silent = true;

                var drawingData = ReduceData(Value, (int)(Value.Length / width));

                var x = 0.0;
                var dx = drawingData.Length > 0 ? width / (drawingData.Length - 1) : width;
                for ( int i = 0; i < drawingData.Length; ++i )
                {
                    if(drawingData[i] != 0)
                    {
                        silent = false;
                    }

                    var y = (-drawingData[i] + 1.0f) * height / 2;
                    y = Math.Max(Math.Min(y, height), 0);
                    valueLine.Points.Add(new Point(x, y));
                    x += dx;
                }

                valueLine.Stroke = silent ? SilenceBrush : NormalBrush;
                valueLine.StrokeThickness = 1;
                canvasElementList.Add(valueLine);
            }

            canvas.Children.Clear();
            foreach (var element in canvasElementList)
            {
                canvas.Children.Add(element);
            }
        }

        private double[] ReduceData(double[] data, int ratio)
        {
            Debug.Assert(data.Length > 0);
            if(ratio < 1)
            {
                ratio = 1;
            }
            var result = new double[data.Length / ratio];
            for(int i = 0; i < result.Length; ++i)
            {
                for (int j = 0; j < ratio; ++j)
                {
                    var v = data[i * ratio + j];
                    if (Math.Abs(v) > Math.Abs(result[i]))
                    {
                        result[i] = v;
                    }
                }
            }

            return result;
        }

        public WaveformViewer()
        {
            InitializeComponent();
        }
    }
}
