﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;

namespace Nintendo.AudioTool.Detail
{
    internal class TargetConnector : IDisposable
    {
        // Htcs
        private TcpClient _tmClient;
        private TcpClient _targetClient;
        private const string TargetServerPortName = "iywys@$audio";
        private static readonly int OldTargetManagerHtcsPort = 8003;

        private static readonly int RetryCountMax = 10;
        private static readonly TimeSpan RetryTime = TimeSpan.FromMilliseconds(200);

        public NetworkStream Stream
        {
            get
            {
                return _targetClient?.GetStream();
            }
        }

        public bool IsConnected
        {
            get
            {
                return _tmClient != null && _targetClient != null && _tmClient.Connected && _targetClient.Connected;
            }
        }

        private IPEndPoint GetSystemAudioMonitorEndpoint(string serialNumber)
        {
            using (var sw = new StreamWriter(_tmClient.GetStream(), new UTF8Encoding(false), 1024, true))
            {
                var xelement = new XElement("RequestSystemPortMapping");
                var value = xelement.ToString(SaveOptions.DisableFormatting);
                using (var task = sw.WriteLineAsync(value))
                {
                    task.Wait();
                }
            }
            using (var sr = new StreamReader(_tmClient.GetStream(), new UTF8Encoding(false), false, 1024, true))
            {
                string line;
                try
                {
                    using (var task = sr.ReadLineAsync())
                    {
                        task.Wait();
                        if ((line = task.Result) == null)
                        {
                            return null;
                        }
                    }
                    var element = XElement.Parse(line);
                    if (element.Name == "HtcsInfo")
                    {
                        foreach (var portMapItem in element.Element("PortMap").Elements("PortMapItem"))
                        {
                            if (portMapItem.Element("HtcsPortName").Value == TargetServerPortName && portMapItem.Element("HtcsPeerName").Value == serialNumber)
                            {
                                var ipAddress = IPAddress.Parse(portMapItem.Element("IPAddress").Value);
                                var portNumber = int.Parse(portMapItem.Element("TcpPortNumber").Value, CultureInfo.InvariantCulture);
                                return new IPEndPoint(ipAddress, portNumber);
                            }
                        }
                    }
                }
                catch (Exception)
                {
                    while (sr.Peek() >= 0)
                    {
                        sr.ReadLine();
                    }
                }
            }

            return null;
        }

        private Socket CreateConnectedSocket(IPEndPoint endPoint)
        {
            var socket = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);
            socket.Connect(endPoint);
            return socket;
        }

        private bool IsHtcsControlPort(int port)
        {
            try
            {
                using (var socket = CreateConnectedSocket(new IPEndPoint(IPAddress.Loopback, port)))
                {
                    using (var socketStream = new NetworkStream(socket))
                    {
                        using (var socketReader = new StreamReader(socketStream))
                        {
                            Task<string> str = socketReader.ReadLineAsync();
                            if (str.Wait(3000))
                            {
                                return (str.Result.IndexOf("<HtcsInfo><TargetList>") == 0);
                            }
                        }
                    }
                }
            }
            catch
            {
                return false;
            }
            return false;
        }

        private int FindTargetManagerHtcsPort()
        {

            try
            {
                return Nintendo.Tm.TargetManager.GetHtcsControlPort();
            }
            catch (Nintendo.Tm.TmException e)
            {
                if (e.GetErrorCode() == Nintendo.Tm.Error.IncompatibleTargetManager && IsHtcsControlPort(OldTargetManagerHtcsPort))
                {
                    return OldTargetManagerHtcsPort;
                }

                throw;
            }
        }

        private void ConnectToTargetManager()
        {
            _tmClient = new TcpClient();

            var htcsPort = FindTargetManagerHtcsPort();

            _tmClient.Connect("localhost", htcsPort);
        }

        private void ConnectToTarget(string targetName)
        {
            Nintendo.Tm.Target connectTarget;
            if (string.IsNullOrEmpty(targetName))
            {
                if (!Nintendo.Tm.TargetManager.TryGetDefaultTarget(out connectTarget))
                {
                    throw new SystemAudioCapture.TargetNotFoundException();
                }
            }
            else
            {
                var targetList = Nintendo.Tm.TargetManager.GetTargets();
                connectTarget = targetList.FirstOrDefault(value => value.GetName() == targetName);
                if (connectTarget == null)
                {
                    throw new SystemAudioCapture.TargetNotFoundException();
                }
            }

            var serialNumber = connectTarget.GetSerialNumber();

            IPEndPoint endPoint = null;
            for (int i = 0; i < RetryCountMax; ++i)
            {
                endPoint = GetSystemAudioMonitorEndpoint(serialNumber);
                if (endPoint != null)
                {
                    break;
                }
                System.Threading.Thread.Sleep(RetryTime);
            }
            if(endPoint == null)
            {
                throw new SystemAudioCapture.ServicePortNotFoundException();
            }

            _targetClient = new TcpClient();
            _targetClient.Connect(endPoint);
        }

        private void ConnectImpl(string targetName)
        {
            ConnectToTargetManager();
            try
            {
                ConnectToTarget(targetName);
            }
            catch
            {
                Disconnect();
                throw;
            }
        }

        public async Task ConnectAsync(string targetName)
        {
            await Task.Run(() => ConnectImpl(targetName));
        }

        public void Disconnect()
        {
            _targetClient?.Close();
            _targetClient = null;
            _tmClient?.Close();
            _tmClient = null;
        }


        public void Dispose()
        {
            Disconnect();
        }
    }
}
