﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Nintendo.Audio.Windows.CoreAudio;
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Threading;

namespace Nintendo.Audio
{
    public class SpatialAudioOut : IDisposable
    {
        private enum State
        {
            Started,
            Stopped,
        }

        private MMDevice _device;
        private SpatialAudioClient _client;
        private SpatialAudioObjectRenderStream _stream;
        private State _state = State.Stopped;
        private Thread _thread;
        private EventWaitHandle _eventWaitHandle;

        private bool _initialized = false;
        private AudioObjectType _mask;
        private int _staticChannelCount;
        private List<AudioObjectType> _staticAudioObjectTypes;
        private SpatialAudioObject[] _staticAudioObjects;
        private List<float>[] _staticAudioObjectPcmData;

        private object _lock = new object();

        public SpatialAudioOut(AudioObjectType mask)
        {
            _mask = mask;

            Initialize();
        }

        private void Initialize()
        {
            if (_initialized)
            {
                return;
            }

            _device = (new MMDeviceEnumerator()).GetDefaultAudioEndpoint(DataFlow.Render, Role.Console);
            _client = new SpatialAudioClient(_device.Activate(CoreAudioGuid.ISpatialAudioClientId));

            Windows.CoreAudio.AudioObjectType objectMask = (Windows.CoreAudio.AudioObjectType)_mask;
            _stream = new SpatialAudioObjectRenderStream(_client.ActivateSpatialAudioStream(objectMask, AudioStreamCategory.GameEffects, CoreAudioGuid.ISpatialAudioObjectRenderStreamId, out _eventWaitHandle));

            _staticChannelCount = 0;
            _staticAudioObjectTypes = new List<AudioObjectType>();
            for (var flag = (int)AudioObjectType.FrontLeft; flag != (int)AudioObjectType.BackCenter; flag <<= 1)
            {
                if (((int)_mask & flag) != 0)
                {
                    _staticAudioObjectTypes.Add((AudioObjectType)flag);
                    ++_staticChannelCount;
                }
            }

            _staticAudioObjects = new SpatialAudioObject[_staticChannelCount];
            for (var i = 0; i < _staticChannelCount; ++i)
            {
                _staticAudioObjects[i] = new SpatialAudioObject(_stream.ActivateSpatialAudioObject((Windows.CoreAudio.AudioObjectType)_staticAudioObjectTypes[i]));
            }

            _staticAudioObjectPcmData = new List<float>[_staticChannelCount];
            for (var i = 0; i < _staticChannelCount; ++i)
            {
                _staticAudioObjectPcmData[i] = new List<float>();
            }

            _initialized = true;
        }

        public void Reset()
        {
            Dispose();
            Initialize();
            _stream.Start();
        }

        private void ThreadFunction()
        {
            _stream.Start();
            while (_state == State.Started)
            {
                if (!_eventWaitHandle.WaitOne(TimeSpan.FromSeconds(1)))
                {
                    Reset();
                    continue;
                }
                try
                {
                    _stream.BeginUpdatingAudioObjects(out var availableDynamicObjectCount, out var frameCountPerBuffer);
                    var availableSampleCount = int.MaxValue;
                    foreach (var pcm in _staticAudioObjectPcmData)
                    {
                        availableSampleCount = Math.Min(pcm.Count, availableSampleCount);
                    }
                    for (var ch = 0; ch < _staticChannelCount; ++ch)
                    {
                        _staticAudioObjects[ch].GetBuffer(out var buffer, out var bufferLength);
                        if (_staticAudioObjectPcmData[ch].Count >= frameCountPerBuffer)
                        {
                            Marshal.Copy(_staticAudioObjectPcmData[ch].GetRange(0, (int)frameCountPerBuffer).ToArray(), 0, buffer, (int)frameCountPerBuffer);
                            _staticAudioObjectPcmData[ch].RemoveRange(0, (int)frameCountPerBuffer);
                        }
                    }
                    _stream.EndUpdatingAudioObjects();
                }
                catch
                {
                    Reset();
                }
            }
            try
            {
                _stream.Stop();
            }
            catch
            {
            }
        }

        public void Start()
        {
            if (_state != State.Started)
            {
                _state = State.Started;
                _thread = new Thread(ThreadFunction);
                _thread.Start();
            }
            else
            {
                _state = State.Started;
            }
        }

        public void Stop()
        {
            if (_state != State.Stopped)
            {
                _state = State.Stopped;
                _thread.Join();
                _thread = null;
            }
        }

        public void Append(AudioObjectType type, float[] pcm)
        {
            for (var i = 0; i < _staticChannelCount; ++i)
            {
                if (type == _staticAudioObjectTypes[i])
                {
                    _staticAudioObjectPcmData[i].AddRange(pcm);
                }
            }
        }

        public void Dispose()
        {
            foreach (var o in _staticAudioObjects)
            {
                o?.Dispose();
            }
            _stream?.Dispose();
            _stream = null;
            _eventWaitHandle?.Dispose();
            _eventWaitHandle = null;
            _client?.Dispose();
            _client = null;
            _device?.Dispose();
            _device = null;

            _initialized = false;
        }
    }
}
