﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "coredump_FormatTypes.h"

//==============================================================================
namespace coredump {
//==============================================================================

class coredump_reader
{

public:
                        coredump_reader             ( char*             pData,
                                                      s64               SizeOfData );

                        ~coredump_reader            ();

        //==============================================================================
        // Parsing API
        bool            ReadHeader                  ( coredump_file_header* pHeader );

        bool            ReadRegisterDefinitions     ( u64*                  pSizeOfDefinitions,
                                                      char**                ppDefinitions );

        //==============================================================================
        // Module info
        s32             ReadNumberOfModules         ( void );
        bool            ReadModule                  ( s32                   AtIndex,
                                                      coredump_module_info* pInfo );

        //==============================================================================
        // Thread info
        s32             ReadNumberOfThreads         ( void );
        bool            ReadThread                  ( s32                   AtIndex,
                                                      coredump_thread_info* pInfo );

        bool            ReadThreadLocalStorage      ( s32                   AtIndex,
                                                      u64*                  pThreadLS );

        //==============================================================================
        // Stack info
        s32             ReadNumberOfStacks          ( void );
        bool            ReadStackFrames             ( s32                   AtIndex,
                                                      u64*                  pThreadId,
                                                      s32*                  pNumberOfStackFrames,
                                                      u64**                 ppStackFrames );

        //==============================================================================
        // Memory info
        s32             ReadNumberOfMemorySegments  ( void );
        bool            ReadMemorySegment           ( s32                       AtIndex,
                                                      coredump_memory_header*   pInfo,
                                                      char**                    ppMemoryData );

        //==============================================================================
        // TTY info
        u64             GetTTYSize                  (  );
        u64             GetTTY                      ( char*                     pBuffer,
                                                      u64                       SizeOfBuffer );

        //==============================================================================
        // Application Id.  Returns TRUE if there is one in this file.
        bool            GetApplicationId            ( u64*                      pId );

        //==============================================================================
        // Image info
        s32             ReadNumberOfImages          ( void );
        bool            ReadImageDetail             ( s32                   AtIndex,
                                                      ScreenshotType*       pType,
                                                      u64*                  pDataSize );
        bool            ReadImage                   ( s32                   AtIndex,
                                                      void*                 pToBuffer,
                                                      u64                   DataSize );

        //==============================================================================
        // Video info
        s32             ReadNumberOfVideos          ( void );
        bool            ReadVideoDetail             ( s32                   AtIndex,
                                                      VideoType*            pType,
                                                      u64*                  pDataSize );
        bool            ReadVideo                   ( s32                   AtIndex,
                                                      void*                 pToBuffer,
                                                      u64                   DataSize );
        //==============================================================================
        // Static functionality

static  result          CanDebugCoreDump            ( char*                 pData,
                                                      s64                   SizeOfData );

static  result          IsValidCoreDump             ( char*                 pData,
                                                      s64                   SizeOfData );

private:
    //==============================================================================
    // Memory types
    enum CompressedMemoryType
    {
        CompressedMemoryType_None   = 0,
        CompressedMemoryType_ZLib,
        CompressedMemoryType_LZ4,
    };

    //========================================================================================================
    // Compressed memory objects.
    struct decompressed_memory_data
    {
                        explicit decompressed_memory_data        ( char* pSource );
                        ~decompressed_memory_data       ();

        //==============================================================================
        // Instance functionality
                bool                        Is          ( char* pSource );
                bool                        Decompress  ( s64 CompressedSize, s64 DecompressedSize, CompressedMemoryType Type );

        //==============================================================================
        // List functionality
        static  decompressed_memory_data*   Add     ( char* pSource, decompressed_memory_data** pToList );
        static  void                        Delete  ( decompressed_memory_data** pDeleteList );
        static  decompressed_memory_data*   Find    ( char* pSource, decompressed_memory_data* pList );

        //==============================================================================

        char*                       m_pSource;
        char*                       m_pDecompressed;
        decompressed_memory_data*   m_pNext;
    };

    //==============================================================================

    char *                      m_pData;
    u64                         m_SizeOfData;
    decompressed_memory_data*   m_pDecompressedMemory;

    void*   FindTag             ( const char* pTag, void* pStartAt = NULL );
    void*   FindTagAtIndex      ( const char* pTag, s32 AtIndex );
    void*   FindImageTagAtIndex ( s32 AtIndex );
    void*   FindVideoTagAtIndex ( s32 AtIndex );
    void    GetDataFromSource   ( void* pSourceData,
                                  void* pToBuffer,
                                  u64 ToBufferSize );
    s32     GetTagCount         ( const char* pTag );
};

//==============================================================================
} // namespace coredump
//==============================================================================

