﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "coredump_FormatReader.h"
#include <zlib.h>
#include <lz4.h>
#include <cstdlib>
#include <algorithm>
#include <limits>

//==============================================================================
namespace coredump {
//==============================================================================

extern bool IsRecognizedTag( coredump_file_data_header* pHdr );

//==============================================================================

coredump_reader::~coredump_reader()
{
    decompressed_memory_data::Delete( &m_pDecompressedMemory );
}

//==============================================================================

result coredump_reader::CanDebugCoreDump( char* pData, s64 SizeOfData )
{
    result Ret = IsValidCoreDump( pData, SizeOfData );
    if( Ret == RESULT_COREDUMP_OK )
    {
        //============================================================================
        // Walk through our chunks and make sure we have at least one thread.
        //============================================================================
        u64 Address = (u64)pData;
        u64 End = Address + SizeOfData;
        bool HaveRegDefinitions = false;
        bool HaveModules = false;
        bool HaveThreads = false;
        while( Address < End )
        {
            coredump_file_data_header* pHdr = (coredump_file_data_header*)Address;
            if( pHdr->IsTag( pRegisterDefinitionsTypeTag ) )
            {
                HaveRegDefinitions = true;
            }
            else if( pHdr->IsTag( pModuleTypeTag ) )
            {
                HaveModules = true;
            }
            else if( pHdr->IsTag( pThreadTypeTag ) )
            {
                HaveThreads = true;
            }
            if( HaveRegDefinitions && HaveModules && HaveThreads )
            {
                return RESULT_COREDUMP_OK;
            }

            Address += (pHdr->m_Size + sizeof( coredump_file_data_header ));
        }

        // Made it here, we didn't find everything we needed.
        Ret = RESULT_COREDUMP_UNABLE_TO_DEBUG;

    }
    return Ret;
}

//==============================================================================

result coredump_reader::IsValidCoreDump( char* pData, s64 SizeOfData )
{
    if( pData == NULL )
    {
        return RESULT_COREDUMP_NOT_FOUND;
    }

    //Make sure it has SOME size.
    if( SizeOfData <= sizeof( coredump::coredump_file_header ) )
    {
        return RESULT_COREDUMP_NOT_VALID;
    }

    //Read in the initial header.
    coredump_file_data_header* pReadHdr = (coredump_file_data_header*)pData;
    //Make sure it's our tag
    if( pReadHdr->IsTag( pHeaderTypeTag ) == false )
    {
        return RESULT_COREDUMP_NOT_VALID;
    }

    //============================================================================
    // As per Siglo-65108, check the version number.
    //============================================================================
    u64 VersionAddress = (u64)pData + (pReadHdr->m_Size + sizeof( coredump_file_data_header ));
    coredump_file_data_header* pVersionHdr = (coredump_file_data_header*)VersionAddress;

    //======================================================================
    // It's OK if this isn't a version chunk - Earlier versions didn't
    // have a version chunk, but we can still read them.
    //======================================================================
    if( pVersionHdr->IsTag( pVersionTag ) )
    {
        coredump_file_version* pVersionInfo = (coredump_file_version*)(VersionAddress + sizeof( coredump_file_data_header ));
        if( (pVersionInfo->m_Build != Version_Build) || (pVersionInfo->m_Minor != Version_Minor) || (pVersionInfo->m_Major != Version_Major) )
        {
            // Is this one a deal breaker?
            if( (pVersionInfo->m_Major > Version_Major) ) //|| ( pVersionInfo->m_Minor > Version_Minor ) )
            {
                return RESULT_COREDUMP_UNRECOGNIZED_VERSION;
            }
        }
    }

    //================================================================================
    // Walk through our chunks and make sure have at least one more that we can read.
    //===============================================================================
    u64 Address = (u64)pData;
    u64 End = Address + SizeOfData;
    int NumberOfChunksFound = 0;

    while( Address < End )
    {
        coredump_file_data_header* pHdr = (coredump_file_data_header*)Address;
        Address += (pHdr->m_Size + sizeof( coredump_file_data_header ));
        if( Address > End )
        {
            return RESULT_COREDUMP_NOT_VALID;
        }
        NumberOfChunksFound += 1;

        //We want to find 3 total.  We've already read the Header chunk and the version chunk
        if( NumberOfChunksFound >= 3 )
        {
            return RESULT_COREDUMP_OK;
        }
    }

    return RESULT_COREDUMP_NOT_VALID;
}

//==============================================================================

bool coredump_reader::ReadHeader( coredump_file_header* pHeader )
{
    bool Ret = false;
    coredump_file_data_header* pReadHdr = (coredump_file_data_header*)m_pData;

    //Make sure it's our tag
    if(pReadHdr->IsTag( pHeaderTypeTag ) )
    {
        void* pReadFrom = (void*)( (u64)pReadHdr + sizeof(coredump_file_data_header) );
        memcpy( pHeader, pReadFrom, sizeof(coredump_file_header) );
        Ret = true;
    }

    return Ret;
}

//==============================================================================

u64 coredump_reader::GetTTYSize( )
{
    u64 Ret = 0;
    void* pAddress = FindTag( pTTYTag );
    if( pAddress != NULL )
    {
        coredump_file_data_header* pHdr = (coredump_file_data_header*)pAddress;
        Ret = pHdr->m_Size;
    }
    return Ret;
}

//==============================================================================

u64 coredump_reader::GetTTY( char* pBuffer, u64 SizeOfBuffer )
{
    u64 Ret = 0;
    void* pAddress = FindTag( pTTYTag );
    if( pAddress != NULL )
    {
        coredump_file_data_header* pHdr = (coredump_file_data_header*)pAddress;
        Ret = pHdr->m_Size > SizeOfBuffer ? SizeOfBuffer : pHdr->m_Size;
        void* pReadFrom = (void*)( (u64)pHdr + sizeof(coredump_file_data_header) );
        memcpy( pBuffer, pReadFrom, static_cast<size_t>(Ret) );
    }
    return Ret;
}

//==============================================================================

void* coredump_reader::FindTag( const char* pTag, void* pStartAt )
{
    u64 Address = (u64)m_pData;
    u64 End = Address + m_SizeOfData;

    if( pStartAt != NULL )
    {
        Address = (u64)pStartAt;
    }

    while( Address < End )
    {
        coredump_file_data_header* pReadHdr = (coredump_file_data_header*)Address;
        if( pReadHdr->IsTag( pTag ) )
        {
            return (void*)Address;
        }
        Address += ( pReadHdr->m_Size + sizeof(coredump_file_data_header) ) ;
    }

    return NULL;
}

//==============================================================================

s32 coredump_reader::GetTagCount( const char* pTag )
{
    s32 Ret = 0;
    void* pStartSearchAddress = NULL;
    void* pFoundAddress = NULL;
    do
    {
        pFoundAddress = FindTag( pTag, pStartSearchAddress );
        if( pFoundAddress != NULL )
        {
            Ret += 1;
            coredump_file_data_header* pReadHdr = (coredump_file_data_header*)pFoundAddress;
            pStartSearchAddress = (void*)( (u64)pFoundAddress + sizeof(coredump_file_data_header) + pReadHdr->m_Size );
        }
    }
    while( pFoundAddress != NULL );

    return Ret;
}

//==============================================================================

void* coredump_reader::FindTagAtIndex( const char* pTag, s32 AtIndex )
{
    s32 Index = 0;
    void* pStartSearchAddress = NULL;
    void* pFoundAddress = NULL;
    do
    {
        pFoundAddress = FindTag( pTag, pStartSearchAddress );
        if( pFoundAddress != NULL )
        {
            if( Index == AtIndex )
            {
                return pFoundAddress;
            }
            Index += 1;
            coredump_file_data_header* pReadHdr = (coredump_file_data_header*)pFoundAddress;
            pStartSearchAddress = (void*)( (u64)pFoundAddress + sizeof(coredump_file_data_header) + pReadHdr->m_Size );
        }
    }
    while( pFoundAddress != NULL );

    return NULL;
}

//==============================================================================

static bool isImage( coredump_file_data_header* pReadHdr )
{
    if( pReadHdr->IsTag( pBMPImageTag )  )
    {
        return true;
    }
    if( pReadHdr->IsTag( pRawImageTag ) )
    {
        return true;
    }
    if( pReadHdr->IsTag( pJPGImageTag ) )
    {
        return true;
    }
    if( pReadHdr->IsTag( pPNGImageTag ) )
    {
        return true;
    }

    return false;
}

//==============================================================================

void* coredump_reader::FindImageTagAtIndex( s32 AtIndex )
{
    s32 Index = 0;
    u64 Address = (u64)m_pData;
    u64 End = Address + m_SizeOfData;

    while( Address < End )
    {
        coredump_file_data_header* pReadHdr = (coredump_file_data_header*)Address;
        if( isImage( pReadHdr ) )
        {
            if( Index == AtIndex )
            {
                return (void*)Address;
            }
            Index += 1;
        }
        Address += (pReadHdr->m_Size + sizeof( coredump_file_data_header ));
    }
    return NULL;
}

//==============================================================================

static bool isVideo( coredump_file_data_header* pReadHdr )
{
    if( pReadHdr->IsTag( pMP4VideoTag ) )
    {
        return true;
    }
    if( pReadHdr->IsTag( pRawVideoTag ) )
    {
        return true;
    }

    return false;
}

//==============================================================================

void* coredump_reader::FindVideoTagAtIndex( s32 AtIndex )
{
    s32 Index = 0;
    u64 Address = (u64)m_pData;
    u64 End = Address + m_SizeOfData;

    while( Address < End )
    {
        coredump_file_data_header* pReadHdr = (coredump_file_data_header*)Address;
        if( isVideo( pReadHdr ) )
        {
            if( Index == AtIndex )
            {
                return (void*)Address;
            }
            Index += 1;
        }
        Address += (pReadHdr->m_Size + sizeof( coredump_file_data_header ));
    }
    return NULL;
}


//==============================================================================

bool coredump_reader::ReadRegisterDefinitions( u64* pSizeOfDefinitions, char** ppDefinitions )
{
    bool Ret = false;
    void* pDefsAddress = FindTag( pRegisterDefinitionsTypeTag );
    if( pDefsAddress != NULL )
    {
        coredump_file_data_header* pReadHdr = (coredump_file_data_header*)pDefsAddress;
        *pSizeOfDefinitions = pReadHdr->m_Size;
        *ppDefinitions      = (char*)( (u64)pDefsAddress + sizeof(coredump_file_data_header) );
        Ret = true;
    }
    return Ret;
}

//==============================================================================
// Module info
s32 coredump_reader::ReadNumberOfModules( )
{
    return GetTagCount( pModuleTypeTag );
}

//==============================================================================

bool coredump_reader::ReadModule( s32 AtIndex, coredump_module_info* pInfo )
{
    bool Ret = false;
    void* pData = FindTagAtIndex( pModuleTypeTag, AtIndex );
    if( pData != NULL )
    {
        //Fill in the thread info
        coredump_file_data_header* pReadHdr = (coredump_file_data_header*)pData;
        coredump_module_info* pSource = (coredump_module_info*)( (u64)pData + sizeof(coredump_file_data_header) );
        memcpy( pInfo, pSource, sizeof(coredump_module_info) );
        Ret = true;
    }
    return Ret;
}

//==============================================================================
// Thread info
s32 coredump_reader::ReadNumberOfThreads()
{
    return GetTagCount( pThreadTypeTag );
}

//==============================================================================

bool coredump_reader::ReadThread( s32 AtIndex, coredump_thread_info* pThreadInfo )
{
    bool Ret = false;
    void* pData = FindTagAtIndex( pThreadTypeTag, AtIndex );
    if( pData != NULL )
    {
        //Fill in the thread info
        coredump_file_data_header* pReadHdr = (coredump_file_data_header*)pData;
        coredump_thread_info* pThreadInfoSource = (coredump_thread_info*)( (u64)pData + sizeof(coredump_file_data_header) );

        //====================================================================
        // Because the register list sizes are defined at write time,
        // we can't just memcpy our data.  We have to hand-resolve these.
        //====================================================================
        pThreadInfo->m_ThreadId                     = pThreadInfoSource->m_ThreadId;
        pThreadInfo->m_CurrentThread                = pThreadInfoSource->m_CurrentThread;
        pThreadInfo->m_Status                       = pThreadInfoSource->m_Status;
        pThreadInfo->m_Priority                     = pThreadInfoSource->m_Priority;
        pThreadInfo->m_Core                         = pThreadInfoSource->m_Core;
        pThreadInfo->m_IP                           = pThreadInfoSource->m_IP;
        pThreadInfo->m_SP                           = pThreadInfoSource->m_SP;
        memcpy( pThreadInfo->m_Name, pThreadInfoSource->m_Name, sizeof(pThreadInfo->m_Name) );

        pThreadInfo->m_NumberOfGPRegisters          = pThreadInfoSource->m_NumberOfGPRegisters;
        pThreadInfo->m_NumberOfGPControlRegisters   = pThreadInfoSource->m_NumberOfGPControlRegisters;
        pThreadInfo->m_NumberOfFPRegisters          = pThreadInfoSource->m_NumberOfFPRegisters;
        pThreadInfo->m_NumberOfFPControlRegisters   = pThreadInfoSource->m_NumberOfFPControlRegisters;

        u64 Address = (u64)(&pThreadInfoSource->m_pGPRegisters);
        pThreadInfo->m_pGPRegisters                 = (u64*)(Address);  //This pointer is OK.  The ones that follow, however...

        //Fix up the address.
        Address += pThreadInfo->m_NumberOfGPRegisters * sizeof(u64);
        pThreadInfo->m_pGPControlRegisters          = (u64*)(Address);

        Address += pThreadInfo->m_NumberOfGPControlRegisters * sizeof(u64);
        pThreadInfo->m_pFPRegisters                 = (u64*)(Address);

        Address += pThreadInfo->m_NumberOfFPRegisters * sizeof(u64);
        pThreadInfo->m_pFPControlRegisters          = (u64*)(Address);

        Ret = true;
    }
    return Ret;
}

//==============================================================================

bool coredump_reader::ReadThreadLocalStorage( s32 AtIndex, u64* pThreadLS )
{
    bool Ret = false;
    void* pData = FindTagAtIndex( pThreadLocalStorageTag, AtIndex );
    if( pData != NULL )
    {
        u64 Address = ((u64)pData + sizeof( coredump_file_data_header ));
        coredump_thread_local_storage* pStorage = (coredump_thread_local_storage*)Address;
        *pThreadLS = pStorage->m_TPIDR;
        Ret = true;
    }
    return Ret;
}

//==============================================================================
// Stack info

s32 coredump_reader::ReadNumberOfStacks( )
{
    return GetTagCount( pStackFramesTag );
}

//==============================================================================

bool coredump_reader::ReadStackFrames( s32 AtIndex, u64* pThreadId, s32* pNumberOfStackFrames, u64** ppStackFrames )
{
    bool Ret = false;
    void* pData = FindTagAtIndex( pStackFramesTag, AtIndex );
    if( pData != NULL )
    {
        u64 Address = ((u64)pData + sizeof( coredump_file_data_header ));
        //Fill in the thread info
        *pThreadId              = *((u64*)(Address + 0));
        *pNumberOfStackFrames   = *((s32*)(Address + sizeof(u64)));
        *ppStackFrames          =   (u64*)(Address + sizeof( u64 ) + sizeof( s32 ) );
        Ret = true;
    }
    return Ret;

}

//==============================================================================
// Memory info
s32 coredump_reader::ReadNumberOfMemorySegments( )
{
    return GetTagCount( pMemoryTypeTag ) + GetTagCount( pCompressedMemoryTypeTag ) + GetTagCount( pZLibCompressedMemoryTypeTag );
}

//==============================================================================

bool coredump_reader::ReadMemorySegment( s32 AtIndex, coredump_memory_header* pInfo, char** ppMemoryData )
{
    bool Ret = false;
    bool IsCompressed = false;
    CompressedMemoryType compressionType = CompressedMemoryType_None;

    void* pData = FindTagAtIndex( pMemoryTypeTag, AtIndex );
    if( pData == NULL )
    {
        //Try compressed memory type.
        pData = FindTagAtIndex( pCompressedMemoryTypeTag, AtIndex );
        if( pData == NULL )
        {
            pData = FindTagAtIndex( pZLibCompressedMemoryTypeTag, AtIndex );
            compressionType = CompressedMemoryType_ZLib;
        }
        else
        {
            compressionType = CompressedMemoryType_LZ4;
        }
    }
    if( pData != NULL )
    {
        //Fill in the memory info
        coredump_file_data_header* pReadHdr = (coredump_file_data_header*)pData;
        coredump_memory_header* pSource = (coredump_memory_header*)( (u64)pData + sizeof(coredump_file_data_header) );
        memcpy( pInfo, pSource, sizeof(coredump_memory_header) );

        if( compressionType != CompressedMemoryType_None )
        {
            char* pSourceMemory = (char*)( (u64)pData + sizeof(coredump_file_data_header) + sizeof(coredump_memory_header) );
            decompressed_memory_data* pDecompressed = decompressed_memory_data::Find( pSourceMemory, m_pDecompressedMemory );
            if( pDecompressed == NULL )
            {
                //Create it.
                pDecompressed = decompressed_memory_data::Add( pSourceMemory, &m_pDecompressedMemory );
                if( pDecompressed == NULL )
                {
                    return false;
                }
                pDecompressed->Decompress( pReadHdr->m_Size - sizeof(coredump_memory_header), pSource->m_Size, compressionType );
            }
            *ppMemoryData = pDecompressed->m_pDecompressed;
        }
        else
        {
            //Here's the memory itself:
            *ppMemoryData = (char*)( (u64)pData + sizeof(coredump_file_data_header) + sizeof(coredump_memory_header) );
        }

        Ret = true;
    }
    return Ret;
}

//==============================================================================

bool coredump_reader::GetApplicationId( u64* pId )
{
    bool Ret = false;
    void* pAddress = FindTag( pApplicationIdTag );
    if( pAddress != NULL )
    {
        coredump_file_data_header* pHdr = (coredump_file_data_header*)pAddress;
        void* pReadFrom = (void*)((u64)pHdr + sizeof( coredump_file_data_header ));
        memcpy( pId, pReadFrom, sizeof(u64) );
        Ret = true;
    }
    return Ret;
}

//==============================================================================

void coredump_reader::GetDataFromSource( void* pSourceData, void* pToBuffer, u64 ToBufferSize )
{
    coredump_file_data_header* pReadHdr = (coredump_file_data_header*)pSourceData;
    u64 Address = ((u64)pSourceData + sizeof( coredump_file_data_header ));
    size_t AmountToCopy = (ToBufferSize < pReadHdr->m_Size) ? ToBufferSize : pReadHdr->m_Size;
    memcpy( pToBuffer, (void*)Address, AmountToCopy );
}

//==============================================================================

s32 coredump_reader::ReadNumberOfImages( )
{
    s32 Ret = GetTagCount( pBMPImageTag );
    Ret += GetTagCount( pRawImageTag );
    Ret += GetTagCount( pJPGImageTag );
    Ret += GetTagCount( pPNGImageTag );
    return Ret;
}

//==============================================================================

static ScreenshotType getImageType( coredump_file_data_header* pReadHdr )
{
    if( pReadHdr->IsTag( pBMPImageTag ) )
    {
        return ScreenshotType_BMP;
    }
    if( pReadHdr->IsTag( pRawImageTag ) )
    {
        return ScreenshotType_Raw;
    }
    if( pReadHdr->IsTag( pJPGImageTag ) )
    {
        return ScreenshotType_JPG;
    }
    if( pReadHdr->IsTag( pPNGImageTag ) )
    {
        return ScreenshotType_PNG;
    }

    return ScreenshotType_Raw;
}

//==============================================================================

bool coredump_reader::ReadImageDetail( s32 AtIndex, ScreenshotType* pType, u64* pDataSize )
{
    void* pData = FindImageTagAtIndex( AtIndex );
    if( pData != NULL )
    {
        coredump_file_data_header* pReadHdr = (coredump_file_data_header*)pData;
        *pType = getImageType( pReadHdr );
        *pDataSize = pReadHdr->m_Size;
        return true;
    }

    return false;
}

//==============================================================================

bool coredump_reader::ReadImage( s32 AtIndex, void* pToBuffer, u64 DataSize )
{
    void* pData = FindImageTagAtIndex( AtIndex );
    if( pData != NULL )
    {
        GetDataFromSource( pData, pToBuffer, DataSize );
        return true;
    }

    return false;
}

//==============================================================================
// Video finery
//==============================================================================

s32 coredump_reader::ReadNumberOfVideos( )
{
    s32 Ret = GetTagCount( pMP4VideoTag );
    Ret += GetTagCount( pRawVideoTag );
    return Ret;
}

//==============================================================================

static VideoType getVideoType( coredump_file_data_header* pReadHdr )
{
    if( pReadHdr->IsTag( pMP4VideoTag ) )
    {
        return VideoType_MP4;
    }
    return VideoType_Raw;
}

//==============================================================================

bool coredump_reader::ReadVideoDetail( s32 AtIndex, VideoType* pType, u64* pDataSize )
{
    void* pData = FindVideoTagAtIndex( AtIndex );
    if( pData != NULL )
    {
        coredump_file_data_header* pReadHdr = (coredump_file_data_header*)pData;
        *pType = getVideoType( pReadHdr );
        *pDataSize = pReadHdr->m_Size;
        return true;
    }

    return false;
}

//==============================================================================

bool coredump_reader::ReadVideo( s32 AtIndex, void* pToBuffer, u64 DataSize )
{
    void* pData = FindVideoTagAtIndex( AtIndex );
    if( pData != NULL )
    {
        GetDataFromSource( pData, pToBuffer, DataSize );
        return true;
    }

    return false;
}

//==============================================================================
// Compressed memory objects.
//==============================================================================

coredump_reader::decompressed_memory_data::decompressed_memory_data( char* pSource )
{
    m_pSource       = pSource;
    m_pDecompressed = NULL;
    m_pNext         = NULL;
}

//==============================================================================

coredump_reader::decompressed_memory_data::~decompressed_memory_data()
{
    if( m_pDecompressed != NULL )
    {
        free( m_pDecompressed );
    }
}

//==============================================================================

bool coredump_reader::decompressed_memory_data::Is( char* pSource )
{
    return (m_pSource == pSource);
}

//==============================================================================

bool coredump_reader::decompressed_memory_data::Decompress( s64 CompressedSize, s64 DecompressedSize, CompressedMemoryType Type )
{
    bool Ret = false;

    m_pDecompressed = (char*)malloc( (size_t)DecompressedSize );
    if( m_pDecompressed != NULL )
    {
        if( Type == CompressedMemoryType_ZLib )
        {
            z_stream c_stream;
            c_stream.zalloc = 0;
            c_stream.zfree = 0;
            c_stream.opaque = (voidpf)0;
            c_stream.avail_in = 0;
            c_stream.next_in = 0;

            if( inflateInit( &c_stream ) == Z_OK)
            {
                c_stream.next_in  = (z_const unsigned char *)m_pSource;
                c_stream.avail_in = (unsigned int)CompressedSize;
                c_stream.avail_out = (unsigned int)DecompressedSize;
                c_stream.next_out = (z_const unsigned char *)m_pDecompressed;

                int ZRet = inflate( &c_stream, Z_FULL_FLUSH );
                ZRet = inflateEnd( &c_stream );
                if( ZRet != Z_STREAM_END )
                {
                    Ret = true;
                }
                else
                {
                    Ret = false;
                }
            }
        }
        else if( Type == CompressedMemoryType_LZ4 )
        {
            LZ4_streamDecode_t decodeStream;
            LZ4_setStreamDecode(&decodeStream, NULL, 0);
            char* pSource = m_pSource;
            char* pDest = (char*)m_pDecompressed;
            s64 AmountToDecompress = (s64)CompressedSize;
            s64 AmountDecompressed = 0;
            while( AmountToDecompress > 0 )
            {
                u32* pCompressedSize = (u32* )pSource;
                pSource += sizeof(u32);

                s32 CompressedThisPassSize = (s32)(*pCompressedSize);

                // Cap max read size to avoid overflows.
                s64 SizeLeftInBuffer = DecompressedSize - AmountDecompressed;
                s64 MaxS32 = (std::numeric_limits<s32>::max)();
                s32 MaxDecompressSize = (s32)(std::min)(MaxS32, SizeLeftInBuffer);

                int SizeDecompressed = LZ4_decompress_safe_continue ( &decodeStream, pSource, pDest, CompressedThisPassSize, MaxDecompressSize );
                if( SizeDecompressed <= 0)
                {
                    //LZ4 library ERROR.  These are non-descriptive errors, just 0 or -(something).
                    break;
                }
                AmountToDecompress -= CompressedThisPassSize;
                AmountToDecompress -= sizeof(u32);
                pSource += CompressedThisPassSize;
                pDest += SizeDecompressed;
                AmountDecompressed += SizeDecompressed;
            }

            Ret = ( AmountToDecompress == 0 );
        }
    }

    return Ret;
}

//==============================================================================

coredump_reader::decompressed_memory_data* coredump_reader::decompressed_memory_data::Add( char* pSource, decompressed_memory_data** pToList )
{
    decompressed_memory_data* pNew = new decompressed_memory_data( pSource );
    if( pNew != NULL )
    {
        decompressed_memory_data* pList = *pToList;
        if( pList == NULL )
        {
            *pToList = pNew;
        }
        else
        {
            while ( pList->m_pNext != NULL )
            {
                pList = pList->m_pNext;
            }
            pList->m_pNext = pNew;
        }
    }
    return pNew;
}

//==============================================================================

void coredump_reader::decompressed_memory_data::Delete( decompressed_memory_data** pDeleteList )
{
    decompressed_memory_data* pList = *pDeleteList;
    while( pList != NULL )
    {
        decompressed_memory_data* pNext = pList->m_pNext;
        delete pList;
        pList = pNext;
    }
    *pDeleteList = NULL;
}

//==============================================================================

coredump_reader::decompressed_memory_data* coredump_reader::decompressed_memory_data::Find( char* pSource, decompressed_memory_data* pList )
{
    while ( pList != NULL )
    {
        if( pList->Is( pSource ) )
        {
            return pList;
        }
        pList = pList->m_pNext;
    }
    return NULL;
}

//==============================================================================

coredump_reader::coredump_reader( char* pData, s64 SizeOfData )
:   m_pData( pData )
,   m_SizeOfData( SizeOfData )
,   m_pDecompressedMemory( NULL )
{
}

//==============================================================================
} //namespace coredump
//==============================================================================
