﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <utility>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/settings/system/settings_News.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_UuidTypes.h>

#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_News.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< お知らせ設定のキー
const char* const SettingNameNewsSettings = "news_settings";

//!< 通知設定のキー
const char* const SettingNameNotificationSettings = "notification_settings";

//!< アカウント毎の通知設定のキー
const char* const SettingNameAccountNotificationSettings =
    "account_notification_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< 選択肢のキー
const char* const KeyChoices = "choices";

//!< アプリケーション自動配信が有効か否かを表す値のキー
const char* const KeyAutomaticApplicationDownloadFlag =
    "automatic_application_download_flag";

//!< 通知と同時に音を出すか否かを表す値のキー
const char* const KeyRingtone = "ringtone_flag";

//!< ダウンロード完了を通知するか否かを表す値のキー
const char* const KeyDownloadCompletion = "download_completion_flag";

//!< スリープ中にサプライズニュースを表示するか否かを表す値のキー
const char* const KeyEnablesNews = "enables_news";

//!< スリープ中にサプライズニュースを受信した際に通知ランプを点灯するか否かを表す値のキー
const char* const KeyIncomingLamp = "incoming_lamp_flag";

//!< サプライズニュース通知の音量のキー
const char* const KeyVolume = "volume";

//!< サプライズニュース表示を開始する時刻のキー
const char* const KeyHeadTime = "head_time";

//!< サプライズニュース表示を終了する時刻のキー
const char* const KeyTailTime = "tail_time";

//!< アカウントの識別子のキー
const char* const KeyAccountUid = "account_uid";

//!< フレンドのオンライン通知を行うか否かを表す値のキー
const char* const KeyFriendOnlineFlag = "friend_online_flag";

//!< フレンドリクエストの通知を行うか否かを表す値のキー
const char* const KeyFriendRequestFlag = "friend_request_flag";

//!< Coral のルーム招待の通知を行うか否かを表す値のキー
const char* const KeyCoralInvitationFlag = "coral_invitation_flag";

//!< フレンドプレゼンスのオーバーレイ通知の表示範囲を表す値のキー
const char* const KeyFriendPresenceOverlayPermission =
    "friend_presence_overlay_permission";

//!< なしを表す通知設定の音量
const char* const NotificationVolumeMute = "Mute";

//!< 小さいを表す通知設定の音量
const char* const NotificationVolumeLow = "Low";

//!< 大きいを表す通知設定の音量
const char* const NotificationVolumeHigh = "High";

//!< 未設定を表すフレンドプレゼンスのオーバーレイ通知の表示範囲
const char* const FriendPresenceOverlayPermissionNotConfirmed = "NotConfirmed";

//!< 表示しないを表すフレンドプレゼンスのオーバーレイ通知の表示範囲
const char* const FriendPresenceOverlayPermissionNoDisplay = "NoDisplay";

//!< お気に入りのフレンドだけを表すフレンドプレゼンスのオーバーレイ通知の表示範囲
const char* const FriendPresenceOverlayPermissionFavoriteFriends =
    "FavoriteFriends";

//!< すべてのフレンドを表すフレンドプレゼンスのオーバーレイ通知の表示範囲
const char* const FriendPresenceOverlayPermissionFriends = "Friends";

//!< 通知設定の時刻をエンコードします。
bool EncodeNotificationTime(
    ::std::string* pOutValue,
    const ::nn::settings::system::NotificationTime& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    char buffer[6] = {};

    ::nn::util::SNPrintf(
        buffer, sizeof(buffer), "%02d:%02d", value.hour, value.minute);

    *pOutValue = buffer;

    return true;
}

//!< 通知設定の時刻をデコードします。
bool DecodeNotificationTime(::nn::settings::system::NotificationTime* pOutValue,
                            const ::std::string& string) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    // フォーマット 00:00 の数値開始位置
    const size_t indices[] = { 0, 3 };

    int32_t* const buffers[] = { &pOutValue->hour, &pOutValue->minute };

    NN_STATIC_ASSERT(NN_ARRAY_SIZE(indices) == NN_ARRAY_SIZE(buffers));

    for (size_t i = 0; i < NN_ARRAY_SIZE(indices); ++i)
    {
        if (indices[i] + 2 <= string.length())
        {
            auto value = int64_t();

            if (DecodeInteger(&value, string.substr(indices[i], 2), 10))
            {
                *buffers[i] = static_cast<int32_t>(value);

                continue;
            }
        }

        return false;
    }

    return true;
}

//!< アプリケーション自動配信が有効か否かを表す値をエクスポートします。
bool ExportAutomaticApplicationDownloadFlag(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyAutomaticApplicationDownloadFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                ::nn::settings::system::GetAutomaticApplicationDownloadFlag()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(false));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyAutomaticApplicationDownloadFlag, ::std::move(node)));

    return true;
}

//!< アプリケーション自動配信が有効か否かを表す値をインポートします。
bool ImportAutomaticApplicationDownloadFlag(const Node& dictNode) NN_NOEXCEPT
{
    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyAutomaticApplicationDownloadFlag));

    NameScope nameScope(KeyAutomaticApplicationDownloadFlag);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto value = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    ::nn::settings::system::SetAutomaticApplicationDownloadFlag(value);

    return true;
}

//!< 通知と同時に音を出すか否かを表す値をエクスポートします。
bool ExportRingtone(
    Node* pDictNode,
    const ::nn::settings::system::NotificationSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyRingtone);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::NotificationFlag::Ringtone>()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(true));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyRingtone, ::std::move(node)));

    return true;
}

//!< 通知と同時に音を出すか否かを表す値をインポートします。
bool ImportRingtone(
    ::nn::settings::system::NotificationSettings* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyRingtone));

    NameScope nameScope(KeyRingtone);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto value = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    pOutValue->flags.Set<
        ::nn::settings::system::NotificationFlag::Ringtone>(value);

    return true;
}

//!< ダウンロード完了を通知するか否かを表す値をエクスポートします。
bool ExportDownloadCompletion(
    Node* pDictNode,
    const ::nn::settings::system::NotificationSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyDownloadCompletion);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::NotificationFlag::
                        DownloadCompletion>()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(true));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyDownloadCompletion, ::std::move(node)));

    return true;
}

//!< ダウンロード完了を通知するか否かを表す値をインポートします。
bool ImportDownloadCompletion(
    ::nn::settings::system::NotificationSettings* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyDownloadCompletion));

    NameScope nameScope(KeyDownloadCompletion);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto value = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    pOutValue->flags.Set<
        ::nn::settings::system::NotificationFlag::DownloadCompletion>(value);

    return true;
}

//!< スリープ中にサプライズニュースを表示するか否かを表す値をエクスポートします。
bool ExportEnablesNews(
    Node* pDictNode,
    const ::nn::settings::system::NotificationSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyEnablesNews);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::NotificationFlag::EnablesNews>()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(true));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyEnablesNews, ::std::move(node)));

    return true;
}

//!< スリープ中にサプライズニュースを表示するか否かを表す値をインポートします。
bool ImportEnablesNews(
    ::nn::settings::system::NotificationSettings* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyEnablesNews));

    NameScope nameScope(KeyEnablesNews);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto value = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    pOutValue->flags.Set<
        ::nn::settings::system::NotificationFlag::EnablesNews>(value);

    return true;
}

//!< スリープ中にサプライズニュースを受信した際に通知ランプを点灯するか否かを表す値をエクスポートします。
bool ExportIncomingLamp(
    Node* pDictNode,
    const ::nn::settings::system::NotificationSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyIncomingLamp);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::NotificationFlag::IncomingLamp>()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(true));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyIncomingLamp, ::std::move(node)));

    return true;
}

//!< スリープ中にサプライズニュースを受信した際に通知ランプを点灯するか否かを表す値をインポートします。
bool ImportIncomingLamp(
    ::nn::settings::system::NotificationSettings* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyIncomingLamp));

    NameScope nameScope(KeyIncomingLamp);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto value = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    pOutValue->flags.Set<
        ::nn::settings::system::NotificationFlag::IncomingLamp>(value);

    return true;
}

//!< サプライズニュース通知の音量をエクスポートします。
bool ExportVolume(
    Node* pDictNode,
    const ::nn::settings::system::NotificationSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyVolume);

        auto currentValueNode = Node::CreateStringNode();

        switch (settings.volume)
        {
        case ::nn::settings::system::NotificationVolume_Mute:
            COMMAND_DO(currentValueNode.SetValue(NotificationVolumeMute));
            break;

        case ::nn::settings::system::NotificationVolume_Low:
            COMMAND_DO(currentValueNode.SetValue(NotificationVolumeLow));
        break;

        case ::nn::settings::system::NotificationVolume_High:
            COMMAND_DO(currentValueNode.SetValue(NotificationVolumeHigh));
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        COMMAND_DO(defaultValueNode.SetValue(NotificationVolumeHigh));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));

        auto choicesValueNode = Node::CreateArrayNode();

        const char* const choices[] =
        {
            NotificationVolumeMute,
            NotificationVolumeLow,
            NotificationVolumeHigh,
        };

        for (const char* const choice : choices)
        {
            auto choiceValueNode = Node::CreateStringNode();

            COMMAND_DO(choiceValueNode.SetValue(choice));

            COMMAND_DO(
                choicesValueNode.AppendElement(::std::move(choiceValueNode)));
        }

        COMMAND_DO(
            node.AppendMember(KeyChoices, ::std::move(choicesValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyVolume, ::std::move(node)));

    return true;
}

//!< サプライズニュース通知の音量をインポートします。
bool ImportVolume(
    ::nn::settings::system::NotificationSettings* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyVolume));

    NameScope nameScope(KeyVolume);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string volume;
    COMMAND_DO(pCurrentValueNode->GetValue(&volume));

    if (volume == NotificationVolumeMute)
    {
        pOutValue->volume =
            static_cast<int32_t>(
                ::nn::settings::system::NotificationVolume_Mute);

        return true;
    }

    if (volume == NotificationVolumeLow)
    {
        pOutValue->volume =
            static_cast<int32_t>(
                ::nn::settings::system::NotificationVolume_Low);

        return true;
    }

    if (volume == NotificationVolumeHigh)
    {
        pOutValue->volume =
            static_cast<int32_t>(
                ::nn::settings::system::NotificationVolume_High);

        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), volume);

    return false;
}

//!< サプライズニュース表示を開始する時刻をエクスポートします。
bool ExportHeadTime(
    Node* pDictNode,
    const ::nn::settings::system::NotificationSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyHeadTime);

        auto currentValueNode = Node::CreateStringNode();

        ::std::string value;
        COMMAND_DO(EncodeNotificationTime(&value, settings.headTime));

        COMMAND_DO(currentValueNode.SetValue(value));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        COMMAND_DO(defaultValueNode.SetValue("09:00"));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyHeadTime, ::std::move(node)));

    return true;
}

//!< サプライズニュース表示を開始する時刻をインポートします。
bool ImportHeadTime(
    ::nn::settings::system::NotificationSettings* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyHeadTime));

    NameScope nameScope(KeyHeadTime);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string value;
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    ::nn::settings::system::NotificationTime time = {};
    if (DecodeNotificationTime(&time, value))
    {
        pOutValue->headTime = time;

        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), value);

    return false;
}

//!< サプライズニュース表示を終了する時刻をエクスポートします。
bool ExportTailTime(
    Node* pDictNode,
    const ::nn::settings::system::NotificationSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyTailTime);

        auto currentValueNode = Node::CreateStringNode();

        ::std::string value;
        COMMAND_DO(EncodeNotificationTime(&value, settings.tailTime));

        COMMAND_DO(currentValueNode.SetValue(value));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        COMMAND_DO(defaultValueNode.SetValue("21:00"));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyTailTime, ::std::move(node)));

    return true;
}

//!< サプライズニュース表示を終了する時刻をインポートします。
bool ImportTailTime(
    ::nn::settings::system::NotificationSettings* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyTailTime));

    NameScope nameScope(KeyTailTime);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string value;
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    ::nn::settings::system::NotificationTime time = {};
    if (DecodeNotificationTime(&time, value))
    {
        pOutValue->tailTime = time;

        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), value);

    return false;
}

//!< アカウントの識別子をエクスポートします。
bool ExportAccountUid(
    Node* pDictNode,
    const ::nn::settings::system::AccountNotificationSettings& settings
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyAccountUid);

        auto currentValueNode = Node::CreateStringNode();

        ::std::string value;
        COMMAND_DO(EncodeUuid(&value, ConvertToUuid(settings.uid)));

        COMMAND_DO(currentValueNode.SetValue(value));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        ::nn::util::Uuid uuid = {};
        COMMAND_DO(EncodeUuid(&value, uuid));

        COMMAND_DO(defaultValueNode.SetValue(value));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyAccountUid, ::std::move(node)));

    return true;
}

//!< アカウントの識別子をインポートします。
bool ImportAccountUid(
    ::nn::settings::system::AccountNotificationSettings* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyAccountUid));

    NameScope nameScope(KeyAccountUid);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string value;
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    ::nn::util::Uuid uuid = {};

    if (!DecodeUuid(&uuid, value))
    {
        PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), value);

        return false;
    }

    pOutValue->uid = ConvertToUid(uuid);

    return true;
}

//!< フレンドのオンライン通知を行うか否かを表す値をエクスポートします。
bool ExportFriendOnlineFlag(
    Node* pDictNode,
    const ::nn::settings::system::AccountNotificationSettings& settings
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyFriendOnlineFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::AccountNotificationFlag::
                        FriendOnline>()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(true));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyFriendOnlineFlag, ::std::move(node)));

    return true;
}

//!< フレンドのオンライン通知を行うか否かを表す値をインポートします。
bool ImportFriendOnlineFlag(
    ::nn::settings::system::AccountNotificationSettings* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyFriendOnlineFlag));

    NameScope nameScope(KeyFriendOnlineFlag);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto value = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    pOutValue->flags.Set<
        ::nn::settings::system::AccountNotificationFlag::FriendOnline>(value);

    return true;
}

//!< フレンドリクエスト通知を行うか否かを表す値をエクスポートします。
bool ExportFriendRequestFlag(
    Node* pDictNode,
    const ::nn::settings::system::AccountNotificationSettings& settings
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyFriendRequestFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::AccountNotificationFlag::
                        FriendRequest>()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(true));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyFriendRequestFlag, ::std::move(node)));

    return true;
}

//!< フレンドリクエスト通知を行うか否かを表す値をインポートします。
bool ImportFriendRequestFlag(
    ::nn::settings::system::AccountNotificationSettings* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyFriendRequestFlag));

    NameScope nameScope(KeyFriendRequestFlag);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto value = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    pOutValue->flags.Set<
        ::nn::settings::system::AccountNotificationFlag::FriendRequest>(value);

    return true;
}

//!< Coral のルーム招待の通知を行うか否かを表す値をエクスポートします。
bool ExportCoralInvitationFlag(
    Node* pDictNode,
    const ::nn::settings::system::AccountNotificationSettings& settings
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyCoralInvitationFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::AccountNotificationFlag::
                        CoralInvitation>()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(true));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyCoralInvitationFlag, ::std::move(node)));

    return true;

}

//!< Coral のルーム招待の通知を行うか否かを表す値をインポートします。
bool ImportCoralInvitationFlag(
    ::nn::settings::system::AccountNotificationSettings* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyCoralInvitationFlag));

    NameScope nameScope(KeyCoralInvitationFlag);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto value = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&value));

    pOutValue->flags.Set<
        ::nn::settings::system::AccountNotificationFlag::CoralInvitation>(
            value);

    return true;
}

//!< フレンドプレゼンスのオーバーレイ通知の表示範囲をエクスポートします。
bool ExportFriendPresenceOverlayPermission(
    Node* pDictNode,
    const ::nn::settings::system::AccountNotificationSettings& settings
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyFriendPresenceOverlayPermission);

        auto currentValueNode = Node::CreateStringNode();

        switch (settings.friendPresenceOverlayPermission)
        {
        case ::nn::settings::system::
                 FriendPresenceOverlayPermission_NotConfirmed:
            COMMAND_DO(
                currentValueNode.SetValue(
                    FriendPresenceOverlayPermissionNotConfirmed));
            break;

        case ::nn::settings::system::FriendPresenceOverlayPermission_NoDisplay:
            COMMAND_DO(
                currentValueNode.SetValue(
                    FriendPresenceOverlayPermissionNoDisplay));
            break;

        case ::nn::settings::system::
                 FriendPresenceOverlayPermission_FavoriteFriends:
            COMMAND_DO(
                currentValueNode.SetValue(
                    FriendPresenceOverlayPermissionFavoriteFriends));
            break;

        case ::nn::settings::system::FriendPresenceOverlayPermission_Friends:
            COMMAND_DO(
                currentValueNode.SetValue(
                    FriendPresenceOverlayPermissionFriends));
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        COMMAND_DO(
            defaultValueNode.SetValue(
                FriendPresenceOverlayPermissionNotConfirmed));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));

        auto choicesValueNode = Node::CreateArrayNode();

        const char* const choices[] =
        {
            FriendPresenceOverlayPermissionNotConfirmed,
            FriendPresenceOverlayPermissionNoDisplay,
            FriendPresenceOverlayPermissionFavoriteFriends,
            FriendPresenceOverlayPermissionFriends,
        };

        for (const char* const choice : choices)
        {
            auto choiceValueNode = Node::CreateStringNode();

            COMMAND_DO(choiceValueNode.SetValue(choice));

            COMMAND_DO(
                choicesValueNode.AppendElement(::std::move(choiceValueNode)));
        }

        COMMAND_DO(
            node.AppendMember(KeyChoices, ::std::move(choicesValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyFriendPresenceOverlayPermission, ::std::move(node)));

    return true;
}

//!< フレンドプレゼンスのオーバーレイ通知の表示範囲をインポートします。
bool ImportFriendPresenceOverlayPermission(
    ::nn::settings::system::AccountNotificationSettings* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyFriendPresenceOverlayPermission));

    NameScope nameScope(KeyFriendPresenceOverlayPermission);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string friendPresenceOverlayPermission;
    COMMAND_DO(pCurrentValueNode->GetValue(&friendPresenceOverlayPermission));

    if (friendPresenceOverlayPermission ==
        FriendPresenceOverlayPermissionNotConfirmed)
    {
        pOutValue->friendPresenceOverlayPermission =
            static_cast<int8_t>(
                ::nn::settings::system::
                    FriendPresenceOverlayPermission_NotConfirmed);

        return true;
    }

    if (friendPresenceOverlayPermission ==
        FriendPresenceOverlayPermissionNoDisplay)
    {
        pOutValue->friendPresenceOverlayPermission =
            static_cast<int8_t>(
                ::nn::settings::system::
                    FriendPresenceOverlayPermission_NoDisplay);

        return true;
    }

    if (friendPresenceOverlayPermission ==
        FriendPresenceOverlayPermissionFavoriteFriends)
    {
        pOutValue->friendPresenceOverlayPermission =
            static_cast<int8_t>(
                ::nn::settings::system::
                    FriendPresenceOverlayPermission_FavoriteFriends);

        return true;
    }

    if (friendPresenceOverlayPermission ==
        FriendPresenceOverlayPermissionFriends)
    {
        pOutValue->friendPresenceOverlayPermission =
            static_cast<int8_t>(
                ::nn::settings::system::
                    FriendPresenceOverlayPermission_Friends);

        return true;
    }

    PrintErrorCode(
        ErrorCode::NodeValueInvalid,
        nameScope.Get(), friendPresenceOverlayPermission);

    return false;
}

} // namespace

bool IsSettingNameNewsSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameNewsSettings);
}

bool ExportNewsSettings(Node* pNode) NN_NOEXCEPT
{
    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameNewsSettings);

        COMMAND_DO(ExportAutomaticApplicationDownloadFlag(&node));
    }

    COMMAND_DO(
        pNode->AppendMember(SettingNameNewsSettings, ::std::move(node)));

    return true;
}

bool ImportNewsSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameNewsSettings);

    COMMAND_DO(ImportAutomaticApplicationDownloadFlag(node));

    return true;
}

bool IsSettingNameNotificationSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameNotificationSettings);
}

bool ExportNotificationSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameNotificationSettings);

        ::nn::settings::system::NotificationSettings settings = {};

        ::nn::settings::system::GetNotificationSettings(&settings);

        COMMAND_DO(ExportRingtone(&node, settings));

        COMMAND_DO(ExportDownloadCompletion(&node, settings));

        COMMAND_DO(ExportEnablesNews(&node, settings));

        COMMAND_DO(ExportIncomingLamp(&node, settings));

        COMMAND_DO(ExportVolume(&node, settings));

        COMMAND_DO(ExportHeadTime(&node, settings));

        COMMAND_DO(ExportTailTime(&node, settings));
    }

    COMMAND_DO(
        pNode->AppendMember(
            SettingNameNotificationSettings, ::std::move(node)));

    return true;
}

bool ImportNotificationSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameNotificationSettings);

    ::nn::settings::system::NotificationSettings settings = {};

    COMMAND_DO(ImportRingtone(&settings, node));

    COMMAND_DO(ImportDownloadCompletion(&settings, node));

    COMMAND_DO(ImportEnablesNews(&settings, node));

    COMMAND_DO(ImportIncomingLamp(&settings, node));

    COMMAND_DO(ImportVolume(&settings, node));

    COMMAND_DO(ImportHeadTime(&settings, node));

    COMMAND_DO(ImportTailTime(&settings, node));

    ::nn::settings::system::SetNotificationSettings(settings);

    return true;
}

bool IsSettingNameAccountNotificationSettings(
    const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameAccountNotificationSettings);
}

bool ExportAccountNotificationSettings(Node* pNode) NN_NOEXCEPT
{
    const int EntryCountMax =
        ::nn::settings::system::AccountNotificationSettingsCountMax;

    ::std::unique_ptr<::nn::settings::system::AccountNotificationSettings[]
          > entries(
              new ::nn::settings::system::AccountNotificationSettings[
                  EntryCountMax]());

    const int count =
        ::nn::settings::system::GetAccountNotificationSettings(
            entries.get(), EntryCountMax);

    auto arrayNode = Node::CreateArrayNode();

    for (int i = 0; i < count; ++i)
    {
        NameScope nameScope(SettingNameAccountNotificationSettings);

        const ::nn::settings::system::AccountNotificationSettings&
            settings = entries[i];

        auto dictNode = Node::CreateObjectNode();

        COMMAND_DO(ExportAccountUid(&dictNode, settings));

        COMMAND_DO(ExportFriendOnlineFlag(&dictNode, settings));

        COMMAND_DO(ExportFriendRequestFlag(&dictNode, settings));

        COMMAND_DO(ExportCoralInvitationFlag(&dictNode, settings));

        COMMAND_DO(ExportFriendPresenceOverlayPermission(&dictNode, settings));

        COMMAND_DO(arrayNode.AppendElement(::std::move(dictNode)));
    }

    COMMAND_DO(
        pNode->AppendMember(
            SettingNameAccountNotificationSettings, ::std::move(arrayNode)));

    return true;
}

bool ImportAccountNotificationSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameAccountNotificationSettings);

    const int EntryCountMax =
        ::nn::settings::system::AccountNotificationSettingsCountMax;

    ::std::unique_ptr<::nn::settings::system::AccountNotificationSettings[]
          > entries(
              new ::nn::settings::system::AccountNotificationSettings[
                  EntryCountMax]());

    int count = 0;

    auto elementCount = size_t();

    COMMAND_DO(node.GetElementCount(&elementCount));

    for (size_t i = 0; i < elementCount; ++i)
    {
        if (count >= EntryCountMax)
        {
            break;
        }

        ::nn::settings::system::AccountNotificationSettings&
              settings = entries[count];

        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(node.GetElement(&pNode, i));

        COMMAND_DO(ImportAccountUid(&settings, *pNode));

        COMMAND_DO(ImportFriendOnlineFlag(&settings, *pNode));

        COMMAND_DO(ImportFriendRequestFlag(&settings, *pNode));

        COMMAND_DO(ImportCoralInvitationFlag(&settings, *pNode));

        COMMAND_DO(ImportFriendPresenceOverlayPermission(&settings, *pNode));

        ++count;
    }

    ::nn::settings::system::SetAccountNotificationSettings(
        entries.get(), count);

    return true;
}
