﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <memory>
#include <string>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/settings/system/settings_Clock.h>

#include "SettingsManager_Clock.h"
#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< 設定情報ヘッダのキー
const char* const SettingNameClockSettings = "clock_settings";

//!< 時計設定の StandardSteadyClock 設定のキー
const char* const KeyStandardSteadyClockSettings = "standard_steady_clock";

//!< 時計設定の StandardUserSystemClock 設定のキー
const char* const KeyStandardUserSystemClockSettings =
    "standard_user_system_clock";

//!< 時計設定の StandardNetworkSystemClock 設定のキー
const char* const KeyStandardNetworkSystemClockSettings =
    "standard_network_system_clock";

//!< SteadyClockTimePoint 設定のキー
const char* const KeySteadyClockTimePointSettings = "steady_clock_time_point";

//!< SteadyClockTimePoint の値のキー
const char* const KeySteadyClockTimePointSettingsValue = "value";

//!< 時計の自動補正の有効、無効を表すフラグのキー
const char* const KeyAutomaticCorrectionEnabled = "automatic_correction";

//!< 時計のコンテキスト設定のキー
const char* const KeySystemClockContextSettings = "system_clock_context";

//!< 時計のコンテキスト設定のオフセット値のキー
const char* const KeySystemClockContextSettingsOffset = "offset";

//!< 時計のソース ID のキー
const char* const KeyClockSourceIdSettings = "source_id";

//!< シャットダウン時の RTC 設定のキー
const char* const KeyShutdownRtcSettings = "shutdown_rtc";

//!< シャットダウン時の RTC 値のキー
const char* const KeyShutdownRtcSettingsValue = "value";

//!< 時計設置の StandardSteadyClock の内部オフセットのキー
const char* const KeyStandardSteadyClockInternalOffsetSettingsValue =
    "internal_offset";

//!< 文字列を int64_t へ変換します
bool StringToInt64(int64_t *pOut, const ::std::string& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    if(!DecodeInteger(pOut, value))
    {
        PrintErrorCode(ErrorCode::NodeValueInvalid, NameScope().Get(), value);

        return false;
    }

    return true;
}

//!< 時計のソース ID をエクスポートします。
bool ExportClockSourceId(
    Node* pDictNode,
    const ::nn::settings::system::ClockSourceId& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeyClockSourceIdSettings);

        ::std::string id;
        COMMAND_DO(EncodeUuid(&id, settings));

        COMMAND_DO(node.SetValue(id));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyClockSourceIdSettings, ::std::move(node)));

    return true;
}

//!< 時計のソース ID をインポートします。
bool ImportClockSourceId(
    ::nn::settings::system::ClockSourceId* pSetting,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSetting);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyClockSourceIdSettings));

    NameScope nameScope(KeyClockSourceIdSettings);

    ::std::string id;
    COMMAND_DO(pNode->GetValue(&id));

    if (!DecodeUuid(pSetting, id))
    {
        PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), id);

        return false;
    }

    return true;
}

bool ExportStandardSteadyClockInternalOffset(
    Node* pDictNode, int64_t value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeyStandardSteadyClockInternalOffsetSettingsValue);

        ::std::string str;
        COMMAND_DO(EncodeInteger(&str, value));

        COMMAND_DO(node.SetValue(str));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyStandardSteadyClockInternalOffsetSettingsValue,
            ::std::move(node)));

    return true;
}

bool ImportStandardSteadyClockInternalOffset(
    int64_t* pValue, const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(
        dictNode.GetMember(
            &pNode, KeyStandardSteadyClockInternalOffsetSettingsValue));

    NameScope nameScope(KeyStandardSteadyClockInternalOffsetSettingsValue);

    ::std::string str;
    COMMAND_DO(pNode->GetValue(&str));

    auto value = int64_t();
    COMMAND_DO(StringToInt64(&value, str));

    *pValue = value;

    return true;
}

//!< SteadyClockTimePoint の値をエクスポートします。
bool ExportSteadyClockTimePointValue(Node* pDictNode, int64_t value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeySteadyClockTimePointSettingsValue);

        ::std::string str;
        COMMAND_DO(EncodeInteger(&str, value));

        COMMAND_DO(node.SetValue(str));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeySteadyClockTimePointSettingsValue, ::std::move(node)));

    return true;
}

//!< SteadyClockTimePoint の値をインポートします。
bool ImportSteadyClockTimePointValue(
    int64_t* pSetting, const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSetting);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(
        dictNode.GetMember(&pNode, KeySteadyClockTimePointSettingsValue));

    NameScope nameScope(KeySteadyClockTimePointSettingsValue);

    ::std::string str;
    COMMAND_DO(pNode->GetValue(&str));

    auto value = int64_t();
    COMMAND_DO(StringToInt64(&value, str));

    *pSetting = value;

    return true;
}

//!< SteadyClockTimePoint をエクスポートします。
bool ExportSteadyClockTimePoint(
    Node* pDictNode,
    const ::nn::settings::system::SteadyClockTimePoint& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeySteadyClockTimePointSettings);

        COMMAND_DO(ExportSteadyClockTimePointValue(&node, settings.value));

        COMMAND_DO(ExportClockSourceId(&node, settings.sourceId));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeySteadyClockTimePointSettings, ::std::move(node)));

    return true;
}

//!< SteadyClockTimePoint をインポートします。
bool ImportSteadyClockTimePoint(
    ::nn::settings::system::SteadyClockTimePoint* pSetting,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSetting);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeySteadyClockTimePointSettings));

    NameScope nameScope(KeySteadyClockTimePointSettings);

    COMMAND_DO(
        ImportSteadyClockTimePointValue(&pSetting->value, *pNode));

    COMMAND_DO(
        ImportClockSourceId(&pSetting->sourceId, *pNode));

    return true;
}

//!< 時計のコンテキストのオフセット値をエクスポートします。
bool ExportSystemClockContextOffset(
    Node* pDictNode,
    int64_t value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeySystemClockContextSettingsOffset);

        ::std::string str;
        COMMAND_DO(EncodeInteger(&str, value));

        COMMAND_DO(node.SetValue(str));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeySystemClockContextSettingsOffset, ::std::move(node)));

    return true;
}

//!< 時計のコンテキストのオフセット値をインポートします。
bool ImportSystemClockContextOffset(
    int64_t* pValue, const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeySystemClockContextSettingsOffset));

    NameScope nameScope(KeySystemClockContextSettingsOffset);

    ::std::string str;
    COMMAND_DO(pNode->GetValue(&str));

    auto value = int64_t();
    COMMAND_DO(StringToInt64(&value, str));

    *pValue = value;

    return true;
}

//!< 時計のコンテキストをエクスポートします。
bool ExportSystemClockContext(
    Node* pDictNode,
    const ::nn::settings::system::SystemClockContext& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeySystemClockContextSettings);


        COMMAND_DO(ExportSystemClockContextOffset(&node, settings.offset));

        COMMAND_DO(ExportSteadyClockTimePoint(&node, settings.timeStamp));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeySystemClockContextSettings, ::std::move(node)));

    return true;
}

//!< 時計のコンテキストをインポートします。
bool ImportSystemClockContext(
    ::nn::settings::system::SystemClockContext* pSetting,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSetting);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeySystemClockContextSettings));

    NameScope nameScope(KeySystemClockContextSettings);

    COMMAND_DO(ImportSystemClockContextOffset(&pSetting->offset, *pNode));

    COMMAND_DO(ImportSteadyClockTimePoint(&pSetting->timeStamp, *pNode));

    return true;
}

//!< 時計の自動補正フラグをエクスポートします。
bool ExportAutomaticCorrectionEnabled(
    Node* pDictNode, bool value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateBooleanNode();

    {
        NameScope nameScope(KeyAutomaticCorrectionEnabled);

        COMMAND_DO(node.SetValue(value));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyAutomaticCorrectionEnabled, ::std::move(node)));

    return true;
}

//!< 時計の自動補正フラグをインポートします。
bool ImportAutomaticCorrectionEnabled(
    bool* pValue, const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyAutomaticCorrectionEnabled));

    NameScope nameScope(KeyAutomaticCorrectionEnabled);

    auto value = bool();
    COMMAND_DO(pNode->GetValue(&value));

    *pValue = value;

    return true;
}

//!< StandardUserSystemClock の設定をエクスポートします。
bool ExportStandardUserSystemClock(Node* pDictNode) NN_NOEXCEPT
{
    ::nn::settings::system::SystemClockContext context;
    ::nn::settings::system::GetUserSystemClockContext(&context);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyStandardUserSystemClockSettings);

        COMMAND_DO(
            ExportAutomaticCorrectionEnabled(
                &node,
                ::nn::settings::system::
                    IsUserSystemClockAutomaticCorrectionEnabled()));

        COMMAND_DO(ExportSystemClockContext(&node, context));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyStandardUserSystemClockSettings, ::std::move(node)));

    return true;
}

//!< StandardUserSystemClock の設定をインポートします。
bool ImportStandardUserSystemClockContext(const Node& dictNode) NN_NOEXCEPT
{
    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyStandardUserSystemClockSettings));

    NameScope nameScope(KeyStandardUserSystemClockSettings);

    auto isAutomaticCorrectionEnabled = bool();
    COMMAND_DO(
        ImportAutomaticCorrectionEnabled(
            &isAutomaticCorrectionEnabled, *pNode));

    auto context = ::nn::settings::system::SystemClockContext();
    COMMAND_DO(ImportSystemClockContext(&context, *pNode));

    ::nn::settings::system::SetUserSystemClockAutomaticCorrectionEnabled(
        isAutomaticCorrectionEnabled);

    ::nn::settings::system::SetUserSystemClockContext(context);

    return true;
}

//!< StandardNetworkSystemClock の設定をエクスポートします。
bool ExportStandardNetworkSystemClock(Node* pDictNode) NN_NOEXCEPT
{
    ::nn::settings::system::SystemClockContext context = {};
    ::nn::settings::system::GetNetworkSystemClockContext(&context);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyStandardNetworkSystemClockSettings);

        COMMAND_DO(ExportSystemClockContext(&node, context));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyStandardNetworkSystemClockSettings, ::std::move(node)));

    return true;
}

//!< StandardNetworkSystemClock の設定をインポートします。
bool ImportStandardNetworkSystemClockContext(const Node& dictNode) NN_NOEXCEPT
{
    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(
        dictNode.GetMember(&pNode, KeyStandardNetworkSystemClockSettings));

    NameScope nameScope(KeyStandardNetworkSystemClockSettings);

    auto context = ::nn::settings::system::SystemClockContext();
    COMMAND_DO(ImportSystemClockContext(&context, *pNode));

    ::nn::settings::system::SetNetworkSystemClockContext(context);

    return true;
}

//!< StandardSteadyClock の設定をエクスポートします。
bool ExportStandardSteadyClock(Node* pDictNode) NN_NOEXCEPT
{
    ::nn::settings::system::ClockSourceId id;
    ::nn::settings::system::GetExternalSteadyClockSourceId(&id);

    auto internalOffset =
        ::nn::settings::system::GetExternalSteadyClockInternalOffset();

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyStandardSteadyClockSettings);

        COMMAND_DO(ExportClockSourceId(&node, id));

        COMMAND_DO(
            ExportStandardSteadyClockInternalOffset(&node, internalOffset));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyStandardSteadyClockSettings, ::std::move(node)));

    return true;
}

//!< StandardSteadyClock の設定をインポートします。
bool ImportStandardSteadyClock(const Node& dictNode) NN_NOEXCEPT
{
    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyStandardSteadyClockSettings));

    NameScope nameScope(KeyStandardSteadyClockSettings);

    auto sourceId = ::nn::settings::system::ClockSourceId();
    COMMAND_DO(ImportClockSourceId(&sourceId, *pNode));

    auto internalOffset = int64_t();
    COMMAND_DO(
        ImportStandardSteadyClockInternalOffset(&internalOffset, *pNode));

    ::nn::settings::system::SetExternalSteadyClockSourceId(sourceId);
    ::nn::settings::system::SetExternalSteadyClockInternalOffset(
        internalOffset);

    return true;
}

//!< シャットダウン時の RTC の値をエクスポートします。
bool ExportShutdownRtcValue(Node* pDictNode, int64_t value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeyShutdownRtcSettingsValue);

        ::std::string str;
        COMMAND_DO(EncodeInteger(&str, value));

        COMMAND_DO(node.SetValue(str));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyShutdownRtcSettingsValue, ::std::move(node)));

    return true;
}

//!< シャットダウン時の RTC の値をインポートします。
bool ImportShutdownRtcValue(int64_t* pSetting, const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSetting);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyShutdownRtcSettingsValue));

    NameScope nameScope(KeyShutdownRtcSettingsValue);

    ::std::string str;
    COMMAND_DO(pNode->GetValue(&str));

    auto value = int64_t();
    COMMAND_DO(StringToInt64(&value, str));

    *pSetting = value;

    return true;
}

//!< シャットダウン時の RTC 設定をエクスポートします。
bool ExportShutdownRtcSettings(Node* pDictNode) NN_NOEXCEPT
{
    auto rtcValue = ::nn::settings::system::GetShutdownRtcValue();

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyShutdownRtcSettings);

        COMMAND_DO(ExportShutdownRtcValue(&node, rtcValue));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyShutdownRtcSettings, ::std::move(node)));

    return true;
}

//!< シャットダウン時の RTC 設定をインポートします。
bool ImportShutdownRtcSettings(const Node& dictNode) NN_NOEXCEPT
{
    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyShutdownRtcSettings));

    NameScope nameScope(KeyShutdownRtcSettings);

    auto rtcValue = int64_t();
    COMMAND_DO(ImportShutdownRtcValue(&rtcValue , *pNode));

    ::nn::settings::system::SetShutdownRtcValue(rtcValue);

    return true;
}

} // namespace

bool IsSettingNameClockSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameClockSettings);
}

bool ExportClockSettings(Node* pNode) NN_NOEXCEPT
{
    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameClockSettings);

        COMMAND_DO(ExportStandardSteadyClock(&node));

        COMMAND_DO(ExportShutdownRtcSettings(&node));

        COMMAND_DO(ExportStandardUserSystemClock(&node));

        COMMAND_DO(ExportStandardNetworkSystemClock(&node));
    }

    COMMAND_DO(
        pNode->AppendMember(SettingNameClockSettings, ::std::move(node)));

    return true;
}

bool ImportClockSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameClockSettings);

    COMMAND_DO(ImportStandardSteadyClock(node));

    COMMAND_DO(ImportShutdownRtcSettings(node));

    COMMAND_DO(ImportStandardUserSystemClockContext(node));

    COMMAND_DO(ImportStandardNetworkSystemClockContext(node));

    return true;
}
