﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "DevMenuCommand_ValueCommandBuilder.h"
#include "DevMenuCommand_MakeValueCommandBuilder.h"
#include "DevMenuCommand_CommandImplUtility.h"
#include <utility>
#include <string>
#include <tuple>
#include <type_traits>
#include <algorithm>
#include <initializer_list>
#include <iterator>

namespace nn { namespace devmenucommand {

namespace detail {

bool IsHex16String(const std::string& s) NN_NOEXCEPT;

inline auto MakeHex16StringValidator() NN_NOEXCEPT
{
    return MakeValidator(IsHex16String, "must have format '0x0123456789ABCDEF' or '0'");
}

template <typename List>
inline auto MakeSelectionValidator(List&& list) NN_NOEXCEPT
{
    return MakeValidator([list = std::forward<List>(list)](auto&& x)
    {
        return std::find(list.begin(), list.end(), x) != list.end();
    }, " must be one of [" + JoinString(std::begin(list), std::end(list), "|") + "]");
}

} // detail

namespace abbreviation {

/**
    @brief 指定したアラインが必要であることを表す。
    @param alignment 要求するアライン
    @return 追加属性を返す。
*/
inline auto Align(size_t alignment) NN_NOEXCEPT
{
    return MakeValidator([alignment](auto&& n)
    {
        return n % alignment == 0;
    }, "needs " + std::to_string(alignment) + " align");
}

/**
    @brief 値の最小値を指定する。
    @param 最小値
    @return 追加属性を返す。
*/
template <typename Int>
inline auto Min(Int c) NN_NOEXCEPT
{
    return MakeValidator([c](auto&& n)
    {
        return n >= c;
    }, "needs >= " + std::to_string(c));
}

/**
    @brief 値の最大値を指定する。
    @param 最大値
    @return 追加属性を返す。
*/
template <typename Int>
inline auto Max(Int c) NN_NOEXCEPT
{
    return MakeValidator([c](auto&& n)
    {
        return n <= c;
    }, "needs <= " + std::to_string(c));
}

/**
    @brief 文字列が "0" または "0x0123456789abcdef" 形式であることを強制する。
*/
const auto Hex16 = std::make_tuple(detail::MakeHex16StringValidator(), ArgumentString("<hex16string>"));

/**
    @brief いずれかの選択であることを強制する。
    @tparam T 値の型
    @param list 選択可能な値のリスト
*/
template <typename T>
inline auto OneOf(std::initializer_list<T> list)
{
    return std::make_tuple(
        detail::MakeSelectionValidator(list),
        ArgumentString("<" + JoinString(std::begin(list), std::end(list), "|") + ">")
    );
}

}

}}
