﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_Optional.h>
#include <nn/fs/fs_ImageDirectory.h>
#include "../DevMenuCommand_Option.h"

namespace album {

enum Action
{
    Action_PrintHelp,
    Action_Upload,
    Action_Download,
    Action_List,
    Action_Info,
    Action_PrintEncryptApplicationId,
    Action_PrintDecryptApplicationId,
    Action_Clean,
};


class ProgramOption
{
public:
    static const size_t PathSizeMax = 512;

public:
    explicit ProgramOption(Action action) NN_NOEXCEPT : m_Action(action)
    {
    }

    // @brief コマンドラインオプションを解析します。
    bool TryParseArgument(const Option& option) NN_NOEXCEPT;

    Action GetAction() const NN_NOEXCEPT
    {
        return m_Action;
    }

    const nn::util::optional<nn::fs::ImageDirectoryId>& GetStorage() const NN_NOEXCEPT
    {
        return m_Storage;
    }

    const char* GetStorageName() const NN_NOEXCEPT
    {
        if (m_Storage == nullptr)
        {
            return "NotSpecified";
        }
        switch (*m_Storage)
        {
            case nn::fs::ImageDirectoryId::Nand:
            {
                return "NAND";
            }
            case nn::fs::ImageDirectoryId::SdCard:
            {
                return "SD";
            }
            default: NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* GetHostDirectory() const NN_NOEXCEPT
    {
        return m_IsHostDirectorySpecified ? m_HostDirectory : nullptr;
    }

    const char* GetLastArgument() const NN_NOEXCEPT
    {
        return m_LastArgument;
    }

    int GetArgumentCount() const NN_NOEXCEPT
    {
        return m_ArgumentCount;
    }

    bool IsSkipEnabled() const NN_NOEXCEPT
    {
        return m_IsSkipEnabled;
    }

    bool IsForceEnabled() const NN_NOEXCEPT
    {
        return m_IsForceEnabled;
    }

    bool IsEmptyCheckRequired() const NN_NOEXCEPT
    {
        return m_IsEmptyCheckRequired;
    }

    bool IsRawModeEnabled() const NN_NOEXCEPT
    {
        return m_IsRawModeEnabled;
    }

    bool IsExtraAlbumFileOnly() const NN_NOEXCEPT
    {
        return m_IsExtraAlbumFileOnly;
    }

    bool IsRegularAlbumFileOnly() const NN_NOEXCEPT
    {
        return m_IsRegularAlbumFileOnly;
    }

private:
    Action m_Action;

    nn::util::optional<nn::fs::ImageDirectoryId> m_Storage;

    bool m_IsHostDirectorySpecified = false;
    char m_HostDirectory[PathSizeMax] = {};
    char m_LastArgument[256] = {};
    int  m_ArgumentCount = 0;

    bool m_IsSkipEnabled = true;
    bool m_IsForceEnabled = false;
    bool m_IsEmptyCheckRequired = false;
    bool m_IsRawModeEnabled = false;
    bool m_IsExtraAlbumFileOnly = false;
    bool m_IsRegularAlbumFileOnly = false;
};

}   // namespace album
