﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenuCommandAlbum_ExecuteDownloadAction.h"

#include <string>
#include <vector>
#include <algorithm>

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_BitUtil.h>
#include <nn/fs.h>
#include <nn/os.h>

#include "DevMenuCommandAlbum_StorageUtility.h"
#include "DevMenuCommandAlbum_MemoryManager.h"

namespace album {

namespace {

    // ファイルオープンに失敗する場合は一定時間リトライする
    nn::Result OpenFileWithRetry(nn::fs::FileHandle* outValue, const char* path, int mode) NN_NOEXCEPT
    {
        auto result = nn::fs::OpenFile(outValue, path, mode);
        if (result.IsSuccess())
        {
            NN_RESULT_SUCCESS;
        }

        NN_LOG("     waiting ");

        auto endTick = nn::os::GetSystemTick() + nn::os::ConvertToTick( nn::TimeSpan::FromSeconds(60) );
        while (nn::os::GetSystemTick() < endTick)
        {
            NN_LOG(".");
            nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
            result = nn::fs::OpenFile(outValue, path, mode);
            if (result.IsSuccess())
            {
                NN_LOG(" ok\n");
                NN_RESULT_SUCCESS;
            }
        }
        NN_LOG(" timedout\n");
        NN_RESULT_THROW( result );
    }

    nn::Result RecursiveCopyDiscardingSubfolders(int64_t& totalCount, const std::string& dstPath, const std::string& srcPath, const ProgramOption& opts)
    {
        nn::fs::DirectoryHandle hSrcDir = {};
        NN_RESULT_DO(nn::fs::OpenDirectory(&hSrcDir, srcPath.c_str(), nn::fs::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory(hSrcDir);
        };

        int64_t count = 0;
        NN_RESULT_DO(nn::fs::GetDirectoryEntryCount(&count, hSrcDir));
        if(count == 0)
        {
            NN_RESULT_SUCCESS;
        }

        std::vector<nn::fs::DirectoryEntry> srcEntryList;
        srcEntryList.resize(static_cast<size_t>(count));

        int64_t n = 0;
        NN_RESULT_DO(nn::fs::ReadDirectory(&n, srcEntryList.data(), hSrcDir, static_cast<int64_t>(srcEntryList.size())));

        int64_t fileCount = 0;
        for (auto& e : srcEntryList)
        {
            std::string entrySrcPath = srcPath + "/" + e.name;
            if (e.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
            {
                NN_RESULT_DO(RecursiveCopyDiscardingSubfolders(totalCount, dstPath, entrySrcPath, opts));
                continue;
            }

            std::string entryDstPath = dstPath + "/" + e.name;

            nn::fs::FileHandle hSrcFile = {};
            nn::fs::FileHandle hDstFile = {};
            NN_RESULT_DO(OpenFileWithRetry(&hSrcFile, entrySrcPath.c_str(), nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(hSrcFile);
            };

            int64_t size = 0;
            NN_RESULT_DO(nn::fs::GetFileSize(&size, hSrcFile));
            NN_RESULT_TRY(nn::fs::CreateFile(entryDstPath.c_str(), size))
                NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
                {
                    if (opts.IsSkipEnabled())
                    {
                        NN_LOG("Skipped file ... %s\n", e.name);
                        continue;
                    }
                    if (!opts.IsForceEnabled())
                    {
                        NN_RESULT_RETHROW;
                    }
                }
            NN_RESULT_END_TRY;

            NN_LOG("Copying file ... %s\n", e.name);
            if (size == 0)
            {
                continue;
            }

            int writeOpenMode = nn::fs::OpenMode_Write;
            if (opts.IsForceEnabled())
            {
                writeOpenMode |= nn::fs::OpenMode_AllowAppend;
            }
            NN_RESULT_DO(nn::fs::OpenFile(&hDstFile, entryDstPath.c_str(), writeOpenMode));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(hDstFile);
            };

            // コピーバッファを使ってコピーする
            void*  copyBufferAddress = album::GetCopyBufferAddress();
            size_t copyBufferSize    = album::GetCopyBufferSize();
            NN_SDK_ASSERT(copyBufferAddress != nullptr && copyBufferSize > 0ull);

            size_t leftSize = static_cast<size_t>(size);
            size_t offset   = 0;
            while (leftSize > 0)
            {
                size_t readSize = std::min(leftSize, copyBufferSize);
                size_t realReadSize = 0;
                NN_RESULT_DO(nn::fs::ReadFile(&realReadSize, hSrcFile, offset, copyBufferAddress, readSize));
                NN_RESULT_DO(nn::fs::WriteFile(hDstFile, offset, copyBufferAddress, realReadSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
                offset   += realReadSize;
                leftSize -= realReadSize;
            }
            fileCount ++;
        }

        totalCount += fileCount;

        NN_RESULT_SUCCESS;
    }

    nn::Result CopyAllFiles(const char* dstMountName, const char* srcMountName, const ProgramOption& opts) NN_NOEXCEPT
    {
        // コピーバッファ用のメモリヒープを確保する
        NN_RESULT_DO( album::AllocateCopyBuffer() );
        NN_UTIL_SCOPE_EXIT
        {
            album::DeallocateCopyBuffer();
        };

        std::string srcRootPath = std::string(srcMountName) + ":/";
        std::string dstRootPath = std::string(dstMountName) + ":/";

        int64_t totalCount = 0;
        NN_RESULT_DO( RecursiveCopyDiscardingSubfolders(totalCount, dstRootPath, srcRootPath, opts) );

        NN_LOG("Copied %d files\n", static_cast<int>(totalCount));
        NN_RESULT_SUCCESS;
    }

}   // namespace

bool ExecuteDownloadAction(const ProgramOption& opts) NN_NOEXCEPT
{
    NN_LOG("Downloading Album: storage=\x22%s\x22\n", opts.GetStorageName());

    if (!MountTargetStorage(*opts.GetStorage()))
    {
        NN_LOG("Downloading Album ... cannot mount target storage.\n");
        return false;
    }
    if (!MountHostDirectory(opts.GetHostDirectory()))
    {
        NN_LOG("Downloading Album ... cannot mount host directory.\n");
        return false;
    }
    if (opts.IsEmptyCheckRequired())
    {
        bool isEmpty = false;
        IsEmptyStorage(&isEmpty, "HOST");
        if(!isEmpty)
        {
            NN_LOG("Failed. The destination is not empty.\n");
            NN_LOG("  dest: %s\n", opts.GetHostDirectory());
            return false;
        }
    }
    if (!CopyAllFiles("HOST", "TARG", opts).IsSuccess())
    {
        NN_LOG("Downloading Album ... failed to complete.\n");
        return false;
    }
    UnmountHostDirectory();
    UnmountTargetStorage();
    NN_LOG("Downloading Album ... completed.\n");
    return true;
}

}   // namespace album
