﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <memory>

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>

#define ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(exp) \
    do \
    { \
        auto _r = (exp); \
        if (!_r.IsSuccess()) \
        { \
            NN_LOG("[accounts] \"%s\" failed with %03d-%04d (0x%08lx)\n", #exp, _r.GetModule(), _r.GetDescription(), _r.GetInnerValueForDebug()); \
            return false; \
        } \
    } while (NN_STATIC_CONDITION(false))

#define ACCOUNTS_FAILURE_UNLESS(exp) \
    do \
    { \
        auto _r = (exp); \
        if (!_r) \
        { \
            NN_LOG("[accounts] \"%s\" failed\n", #exp); \
            return false; \
        } \
    } while (NN_STATIC_CONDITION(false))

namespace accounts {

class Buffer
{
    NN_DISALLOW_COPY(Buffer);
    NN_DISALLOW_MOVE(Buffer);
private:
    std::unique_ptr<char[]> pointer;
    size_t size;
public:
    explicit Buffer(size_t s) NN_NOEXCEPT
        : pointer(reinterpret_cast<char*>(std::malloc(s)))
        , size(s)
    {
        NN_ABORT_UNLESS(pointer);
        NN_ABORT_UNLESS(s > 0);
    }
    void* GetPointer() const NN_NOEXCEPT
    {
        return pointer.get();
    }
    template <typename T>
    T* Get() const NN_NOEXCEPT
    {
        return reinterpret_cast<T*>(GetPointer());
    }
    size_t GetSize() const NN_NOEXCEPT
    {
        return size;
    }
};

struct NetworkConnection
{
    NetworkConnection() NN_NOEXCEPT;
    ~NetworkConnection() NN_NOEXCEPT;

    bool IsAvailable() const NN_NOEXCEPT;
};

bool CreateUser(nn::account::Uid* pOut) NN_NOEXCEPT;
bool DeleteUser(const nn::account::Uid& uid, bool toKeepSavedata) NN_NOEXCEPT;
bool RegisterNsa(const nn::account::Uid& uid) NN_NOEXCEPT;
bool UnregisterNsa(const nn::account::Uid& uid, bool keepSaveData) NN_NOEXCEPT;
bool LinkNa(const nn::account::Uid& uid, const char* id, const char* password)  NN_NOEXCEPT;

bool SynchronizeProfileIfNsaAvailable(const nn::account::Uid& uid) NN_NOEXCEPT;

int UidToString(char* dst, size_t dstSize, const nn::account::Uid& uid) NN_NOEXCEPT;

} // ~namespace account
