﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenuCommand_AccountUtil.h"

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/fs.h>
#include <nn/nifm.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/ns/ns_UserResourceManagementApi.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/settings/system/settings_Account.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>

#include "DevMenuCommand_Nas.h"

namespace accounts {
namespace {
nn::Result SetDefaultProfile(nn::account::ProfileEditor& editor) NN_NOEXCEPT
{
    static const char DefaultUserIconFilePath[] = "Contents:/accounts/defaultIcon.jpg";
    static const uint8_t DefaultUseData[nn::account::UserDataBytesMax] = {
#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
        0x01, 0x00, 0x00, 0x00, // version, mainImageType, padding(2),
        0x01, 0x00, 0x00, 0x00, // charId[LSB], charId>>8, charId>>16, charId[MSB]
        0x01, 0x00, 0x00, 0x01, // bgId[LSB], bgId>>8, bgId>>16, bgId[MSB]
#elif defined(NN_BUILD_CONFIG_ENDIAN_BIG)
        0x01, 0x00, 0x00, 0x00, // version, mainImageType, padding(2),
        0x00, 0x00, 0x00, 0x01, // charId[MSB], charId>>16, charId>>8, charId[LSB]
        0x01, 0x00, 0x00, 0x01, // bgId[MSB], bgId>>16, bgId>>8, bgId[LSB]
#else
#error "Specified endian type is not supported."
#endif
    };

    nn::account::Nickname name = {"User"};
    editor.SetNickname(name);
    editor.SetUserData(reinterpret_cast<const char*>(DefaultUseData), sizeof(DefaultUseData));

    // ROMマウント
    size_t mountRomCacheBufferSize = 0;
    NN_RESULT_DO(nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
    Buffer mountRomBuffer(mountRomCacheBufferSize);
    NN_RESULT_DO(nn::fs::MountRom(
        "Contents", mountRomBuffer.GetPointer(), mountRomBuffer.GetSize())
    );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("Contents");
    };

    Buffer b(128 * 1024);
    nn::fs::FileHandle file;
    NN_RESULT_DO(nn::fs::OpenFile(&file, DefaultUserIconFilePath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };
    int64_t size;
    NN_RESULT_DO(nn::fs::GetFileSize(&size, file));
    NN_SDK_ASSERT(static_cast<size_t>(size) <= b.GetSize());
    NN_RESULT_DO(nn::fs::ReadFile(file, 0, b.GetPointer(), static_cast<size_t>(size)));
    NN_RESULT_DO(editor.FlushWithImage(b.GetPointer(), static_cast<size_t>(size)));
    NN_RESULT_SUCCESS;
}

} // ~namespace accounts::<anonymous>

NetworkConnection::NetworkConnection() NN_NOEXCEPT
{
    nn::nifm::SubmitNetworkRequestAndWait();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::InitializeForSystem());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::Suspend());
    NN_LOG("[accounts] Network connection created\n");
}
NetworkConnection::~NetworkConnection() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::Resume());
    nn::npns::FinalizeForSystem();
    nn::nifm::CancelNetworkRequest();
    NN_LOG("[accounts] Network connection destroyed\n");
}
bool NetworkConnection::IsAvailable() const NN_NOEXCEPT
{
    return nn::nifm::IsNetworkAvailable();
}

bool CreateUser(nn::account::Uid* pOut) NN_NOEXCEPT
{
    nn::account::Uid u;
    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&u));
    nn::account::ProfileEditor editor;
    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(nn::account::GetProfileEditor(&editor, u));
    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(SetDefaultProfile(editor));
    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(nn::account::CompleteUserRegistration(u));

    *pOut = u;
    return true;
}
bool DeleteUser(const nn::account::Uid& uid, bool toKeepSavedata) NN_NOEXCEPT
{
    if (!toKeepSavedata)
    {
        nn::ns::UserSaveDataStatistics total;
        ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(nn::ns::CalculateUserSaveDataStatistics(&total, uid));

        nn::ns::ProgressMonitorForDeleteUserSaveDataAll progress;
        ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(nn::ns::DeleteUserSaveDataAll(&progress, uid));

        bool done = false;
        nn::os::SystemEvent e;
        progress.GetSystemEvent(&e);
        while (!done)
        {
            done = e.TimedWait(nn::TimeSpan::FromMilliSeconds(500));
            progress.Update();
            auto current = progress.GetStatistics();
            NN_LOG(
                "%8d sec : %u of %u, %llu bytes of %llu bytes\n",
                progress.GetElapsedTime().GetSeconds(),
                current.count, total.count, current.sizeInBytes, total.sizeInBytes);
        }
        NN_SDK_ASSERT(progress.IsFinished());
    }
    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(nn::account::DeleteUser(uid));
    return true;
}
bool RegisterNsa(const nn::account::Uid& uid) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountAdministrator admin;
    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, uid));

    // NSA 取得
    nn::account::AsyncContext task;
    NN_ABORT_UNLESS_RESULT_SUCCESS(admin.RegisterAsync(&task));
    nn::os::SystemEvent e;
    NN_ABORT_UNLESS_RESULT_SUCCESS(task.GetSystemEvent(&e));
    e.Wait();
    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(task.GetResult());

    // NT 登録
    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(nn::npns::UploadTokenToBaaS(uid));
    return true;
}
bool UnregisterNsa(const nn::account::Uid& uid, bool keepSaveData) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountAdministrator admin;
    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, uid));

    bool has;
    NN_RESULT_DO(admin.IsNetworkServiceAccountRegistered(&has));
    if (!has)
    {
        return true;
    }

    if (keepSaveData)
    {
        ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(nn::ns::UnregisterNetworkServiceAccount(uid));
    }
    else
    {
        ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(nn::ns::UnregisterNetworkServiceAccountWithUserSaveDataDeletion(uid));
    }
    return true;
}

bool LinkNa(const nn::account::Uid& uid, const char* id, const char* password)  NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountAdministrator admin;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, uid));

    bool registered;
    NN_ABORT_UNLESS_RESULT_SUCCESS(admin.IsNetworkServiceAccountRegistered(&registered));
    if (registered)
    {
        bool linkage;
        NN_ABORT_UNLESS_RESULT_SUCCESS(admin.IsLinkedWithNintendoAccount(&linkage));
        if (linkage)
        {
            NN_LOG("[accounts] Already linked\n");
            return false;
        }
    }
    else
    {
        // NSA 未作成の場合は作成する
        ACCOUNTS_FAILURE_UNLESS(RegisterNsa(uid));
    }

    nn::account::NintendoAccountLinkageProcedure proc;
    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(admin.CreateProcedureToLinkWithNintendoAccount(&proc));

    nn::account::RequestUrl requestUrl;
    nn::account::CallbackUri callbackUri;
    NN_ABORT_UNLESS_RESULT_SUCCESS(proc.GetRequest(&requestUrl, &callbackUri));

    Buffer response(4096);
    // [!注意!] 非サポートの手段による連携の認可処理
    bool verbose = false;
#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
    verbose = true;
#endif
    ACCOUNTS_FAILURE_UNLESS(GetAuthorizationViaNasHelper(response, id, password, requestUrl.url, verbose));

    nn::account::AsyncContext ctx;
    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(proc.ApplyResponseAsync(&ctx, response.Get<char>(), response.GetSize()));
    nn::os::SystemEvent e;
    NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
    e.Wait();
    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(ctx.GetResult());

    bool isReplaced;
    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(proc.IsNetworkServiceAccountReplaced(&isReplaced));
    NN_LOG("[accounts] Linkage done with \"%s\".\n", id);
    if (verbose)
    {
        NN_LOG("[accounts]   - NetworkServiceAccount is %s\n", isReplaced ? "<replaced>" : "<not replaced>");
    }
    return true;
}

bool SynchronizeProfileIfNsaAvailable(const nn::account::Uid& uid) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountAdministrator admin;
    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, uid));

    auto r = admin.CheckNetworkServiceAccountAvailability();
    if (!(r.IsSuccess() || nn::account::ResultNetworkServiceAccountUnavailable::Includes(r)))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    }

    if (r.IsSuccess())
    {
        NN_LOG("[accounts] Synchronizing user profile.");
        nn::account::AsyncContext task;
        NN_ABORT_UNLESS_RESULT_SUCCESS(admin.SynchronizeProfileAsync(&task));
        nn::os::SystemEvent e;
        NN_ABORT_UNLESS_RESULT_SUCCESS(task.GetSystemEvent(&e));
        e.Wait();
        ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(task.GetResult());
    }
    return true;
}

int UidToString(char* dst, size_t dstSize, const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_ASSERT(dstSize >= 20);
    return nn::util::SNPrintf(
        dst, dstSize, "%08x_%08x_%08x_%08x",
        static_cast<uint32_t>(uid._data[0] >> 32),
        static_cast<uint32_t>(uid._data[0] & 0xFFFFFFFFull),
        static_cast<uint32_t>(uid._data[1] >> 32),
        static_cast<uint32_t>(uid._data[1] & 0xFFFFFFFFull));
}

} // ~namespace accounts
