﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenuCommand_UtilityCommand.h"

#include <cstdlib>

#include <nn/nn_Log.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWaitUtility.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

namespace
{
// Subcommands
bool Help(const Option&) NN_NOEXCEPT;

bool Echo(const Option& option) NN_NOEXCEPT
{
    NN_LOG("%s\n", option.HasTarget() ? option.GetTarget() : "");
    return true;
}

bool Sleep(const Option& option) NN_NOEXCEPT
{
    if (!option.HasTarget())
    {
        return false;
    }
    auto seconds = std::atoi(option.GetTarget());

    // Initialize seconds-counter
    nn::os::TimerEvent period(nn::os::EventClearMode_AutoClear);
    period.StartPeriodic(nn::TimeSpan::FromSeconds(1), nn::TimeSpan::FromSeconds(1));

    // Initialize timer
    nn::os::TimerEvent timer(nn::os::EventClearMode_AutoClear);
    timer.StartOneShot(nn::TimeSpan::FromSeconds(seconds));
    nn::os::Tick begin = nn::os::GetSystemTick();

    while (!timer.TryWait())
    {
        nn::os::WaitAny(timer.GetBase(), period.GetBase());
        if (period.TryWait())
        {
            NN_LOG("%d of %d\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetSeconds(), seconds);
        }
    }
    return true;
}

// Execution resource
const struct
{
    bool (*execute)(const Option& option) NN_NOEXCEPT;
    const char *name;
    const char *options;
    const char *desctiption;
} g_SubCommands[] = {
    // Sort alphabetically
    {Echo, "echo"},
    {Help, "help"},
    {Sleep, "sleep", "<seconds to sleep>"},
};

bool Help(const Option&) NN_NOEXCEPT
{
    NN_LOG("Usage of 'util' command:\n");

    const char HelpFormat[] = "  $ util %s %s\n";
    for (const auto& s : g_SubCommands)
    {
        NN_LOG(HelpFormat, s.name, s.options ? s.options : "");
    }
    return true;
}

bool Execute(const Option& option) NN_NOEXCEPT
{
    const auto* subCommand = option.GetSubCommand();

    NN_LOG("util > '%s'\n", subCommand);
    for (const auto& s : g_SubCommands)
    {
        if (strcmp(s.name, subCommand) == 0)
        {
            return s.execute(option);
        }
    }
    NN_LOG("util > Subcommand not found.\n\n");
    return false;
}
} // ~namespace <anonymous>

nn::Result UtilityCommand(bool* pOut, const Option& option)
{
    bool r = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!r) Help(option);
        *pOut = r;
    };

    if (option.HasSubCommand())
    {
        r = Execute(option);
    }
    NN_RESULT_SUCCESS;
}
