﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <vector>
#include <nn/es.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/es/es_ELicenseApiForSystem.h>
#include <nn/es/debug/es_ELicenseArchiveJsonUtil.h>
#include <nn/ns/ns_TicketSystemApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/util/util_Endian.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_ShopUtil.h"
#include "DevMenuCommand_TicketCommand.h"

using namespace nn;

namespace
{
    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " ticket count\n"
        "       " DEVMENUCOMMAND_NAME " ticket delete <rights_id>\n"
        "       " DEVMENUCOMMAND_NAME " ticket delete <content_meta_id>\n"
        "       " DEVMENUCOMMAND_NAME " ticket delete-all [--ticket-type <common|personalized|prepurchase>]\n"
        "       " DEVMENUCOMMAND_NAME " ticket export <rights_id> -o <save_path>\n"
        "       " DEVMENUCOMMAND_NAME " ticket export-with-certificate <rights_id> -o <save_path>\n"
        "       " DEVMENUCOMMAND_NAME " ticket import --ticket <ticket_path> --certificate <certificate_path>\n"
        "       " DEVMENUCOMMAND_NAME " ticket print <rights_id>\n"
        "       " DEVMENUCOMMAND_NAME " ticket list-all [--ticket-type <common|personalized|prepurchase>] [--simple]\n"
        "       " DEVMENUCOMMAND_NAME " ticket import-elicense <user_account_index> [--rights-id <rights_id> --rights-type <permanent|temporary>] (DEPRECATED!)\n";

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    uint64_t To64BitId(const char* string) NN_NOEXCEPT
    {
        uint64_t id = std::strtoull(string, NULL, 16);
        return id;
    }

    uint64_t GetTitleIdFromRightsId(es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT
    {
        uint64_t value = util::LoadBigEndian(reinterpret_cast<uint64_t*>(&rightsId));

        return value;
    }

    Result GetNintendoAccountId(account::NintendoAccountId* outValue, const account::Uid& uid) NN_NOEXCEPT
    {
        account::NetworkServiceAccountManager accountManager;
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetNetworkServiceAccountManager(&accountManager, uid));

        NN_RESULT_DO(accountManager.GetNintendoAccountId(outValue));

        NN_RESULT_SUCCESS;
    }

    const char* ToString(es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT
    {
        static char string[sizeof(nn::es::RightsIdIncludingKeyId) * 2 + 1];

        for (int i = 0; i < sizeof(nn::es::RightsIdIncludingKeyId); i++)
        {
            nn::util::SNPrintf(string + 2 * i, 3, "%02hhx", rightsId._data[i]);
        }

        string[sizeof(nn::es::RightsIdIncludingKeyId) * 2] = '\0';

        return string;
    }

    void PrintTicketInfo(const nn::es::TicketInfo& ticketInfo) NN_NOEXCEPT
    {
        DEVMENUCOMMAND_LOG("RightsId:   0x%s\n", ToString(ticketInfo.rightsId));
        DEVMENUCOMMAND_LOG("TicketId:                   0x%016llx\n", ticketInfo.ticketId);
        DEVMENUCOMMAND_LOG("DeviceId:                   0x%016llx\n", ticketInfo.deviceId);
        DEVMENUCOMMAND_LOG("AccountId:  %34u\n", ticketInfo.accountId);
        DEVMENUCOMMAND_LOG("TicketSize: %34d\n", ticketInfo.ticketSize);
        DEVMENUCOMMAND_LOG("IsElicenseRequired:           %16s\n", ticketInfo.IsElicenseRequired() ? "true" : "false");
        DEVMENUCOMMAND_LOG("IsActive:   %34s\n", nn::ns::IsActiveAccount(ticketInfo.accountId) ? "true" : "false");
        DEVMENUCOMMAND_LOG("\n");
    }

    void PrintPrepurchaseRecord(const nn::es::PrepurchaseRecord& record) NN_NOEXCEPT
    {
        DEVMENUCOMMAND_LOG("RightsId:   0x%s\n", ToString(record.rightsId));
        DEVMENUCOMMAND_LOG("TicketId:                   0x%016llx\n", record.ticketId);
        DEVMENUCOMMAND_LOG("AccountId:  %34u\n", record.accountId);
        DEVMENUCOMMAND_LOG("Time:       %34llu\n", record.deliveryScheduledTime);
        DEVMENUCOMMAND_LOG("\n");
    }

    Result CountCommand(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        NN_UNUSED(option);

        DEVMENUCOMMAND_LOG("Common ticket      : %5d\n", nn::es::CountCommonTicket());
        DEVMENUCOMMAND_LOG("Personalized ticket: %5d\n", nn::es::CountPersonalizedTicket());
        DEVMENUCOMMAND_LOG("Prepurchase record : %5d\n", nn::es::CountPrepurchaseRecord());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DeleteCommand(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        const size_t ContentMetaIdStringLength = 16;
        const size_t RightsIdStringLength = 32;

        std::string idString = std::string(option.GetTarget());

        // ID の先頭が 0x だった場合削除
        if (idString.substr(0, 2) == "0x")
        {
            idString = idString.substr(2);
        }

        // ContentMetaId が指定されていた場合、RightsId の TitleId の領域が ContentMetaId と一致するチケットを全て削除する
        if (idString.length() == ContentMetaIdStringLength)
        {
            uint64_t contentMetaId = To64BitId(idString.c_str());

            auto deleteFunc = [=](int (*countFunc)(), int (*listFunc)(es::RightsIdIncludingKeyId*, int)){
                int count = countFunc();
                std::unique_ptr<es::RightsIdIncludingKeyId[]> list(new es::RightsIdIncludingKeyId[count]);
                listFunc(list.get(), count);

                for (int i = 0; i < count; i++)
                {
                    if (contentMetaId == GetTitleIdFromRightsId(list[i]))
                    {
                        nn::es::DeleteTicket(&list[i], 1);
                    }
                }
            };

            deleteFunc(es::CountCommonTicket, es::ListCommonTicketRightsIds);
            deleteFunc(es::CountPersonalizedTicket, es::ListPersonalizedTicketRightsIds);
            deleteFunc(es::CountPrepurchaseRecord, es::ListPrepurchaseRecordRightsIds);
        }
        else if(idString.length() == RightsIdStringLength)
        {
            nn::es::RightsIdIncludingKeyId rightsId = devmenuUtil::GetRightsIdTarget(option);
            nn::es::DeleteTicket(&rightsId, 1);
        }
        else
        {
            DEVMENUCOMMAND_LOG("Unexpected Input length. Please put 0-padding to satisfy length.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    void PrintBinary(void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        uint8_t* buf = reinterpret_cast<uint8_t*>(buffer);

        for (size_t i = 0; i < bufferSize; i++)
        {
            DEVMENUCOMMAND_LOG("%02X ", buf[i] & 0x000000FF);

            if (i % 16 == 15)
            {
                DEVMENUCOMMAND_LOG("\n");
            }
        }

        DEVMENUCOMMAND_LOG("\n");
    }

    Result DeleteAllCommand(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        if (!option.HasKey("--ticket-type"))
        {
            nn::es::DeleteAllCommonTicket();
            nn::es::DeleteAllPersonalizedTicket();
            nn::es::DeleteAllPrepurchaseRecord();
        }
        else
        {
            if (strncmp(option.GetValue("--ticket-type"), "common", 6) == 0)
            {
                nn::es::DeleteAllCommonTicket();
            }
            else if (strncmp(option.GetValue("--ticket-type"), "personalized", 12) == 0)
            {
                nn::es::DeleteAllPersonalizedTicket();
            }
            else if (strncmp(option.GetValue("--ticket-type"), "prepurchase", 11) == 0)
            {
                nn::es::DeleteAllPrepurchaseRecord();
            }
            else
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ExportCommand(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        if (!option.HasTarget() || !option.GetValue("-o"))
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        std::string rightsIdString = std::string(option.GetTarget());
        std::string saveDirectoryPath = std::string(option.GetValue("-o"));
        std::string saveFilePath = saveDirectoryPath + "/" + rightsIdString + ".tik";

        nn::es::RightsIdIncludingKeyId rightsId = devmenuUtil::GetRightsIdTarget(option);

        size_t outSize;
        NN_RESULT_DO(es::GetCommonTicketSize(&outSize, rightsId));
        NN_RESULT_DO(fs::CreateFile(saveFilePath.c_str(), outSize));

        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, saveFilePath.c_str(), fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        std::unique_ptr<char[]> buffer(new char[outSize]);

        NN_RESULT_DO(es::GetCommonTicketData(&outSize, buffer.get(), outSize, rightsId));
        NN_RESULT_DO(fs::WriteFile(file, 0, buffer.get(), outSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    // Common チケットのみをサポートする
    Result ExportWithCertificateCommand(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        if (!option.HasTarget() || !option.GetValue("-o"))
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        std::string rightsIdString = std::string(option.GetTarget());
        std::string saveDirectoryPath = std::string(option.GetValue("-o"));
        std::string saveTicketFilePath = saveDirectoryPath + "/" + rightsIdString + ".tik";
        std::string saveCertificateFilePath = saveDirectoryPath + "/" + rightsIdString + ".cert";

        nn::es::RightsIdIncludingKeyId rightsId = devmenuUtil::GetRightsIdTarget(option);

        size_t outTicketSize;
        size_t outCertificateSize;
        NN_RESULT_TRY(es::GetCommonTicketAndCertificateSize(&outTicketSize, &outCertificateSize, rightsId))
            NN_RESULT_CATCH(es::ResultRightsIdNotFound)
            {
                DEVMENUCOMMAND_LOG("RightsId Not Found.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY


        std::unique_ptr<char[]> ticketBuffer(new char[outTicketSize]);
        std::unique_ptr<char[]> certificateBuffer(new char[outCertificateSize]);
        NN_ABORT_UNLESS_RESULT_SUCCESS(es::GetCommonTicketAndCertificateData(&outTicketSize, &outCertificateSize, ticketBuffer.get(), certificateBuffer.get(), outTicketSize, outCertificateSize, rightsId));

        {
            NN_RESULT_DO(fs::CreateFile(saveTicketFilePath.c_str(), outTicketSize));
            fs::FileHandle file;
            NN_RESULT_DO(fs::OpenFile(&file, saveTicketFilePath.c_str(), fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
            NN_RESULT_DO(fs::WriteFile(file, 0, ticketBuffer.get(), outTicketSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
        }

        {
            NN_RESULT_DO(fs::CreateFile(saveCertificateFilePath.c_str(), outCertificateSize));
            fs::FileHandle file;
            NN_RESULT_DO(fs::OpenFile(&file, saveCertificateFilePath.c_str(), fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
            NN_RESULT_DO(fs::WriteFile(file, 0, certificateBuffer.get(), outCertificateSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ImportCommand(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        if (!option.HasKey("--ticket") || !option.HasKey("--certificate"))
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        std::string ticketFilePath = std::string(option.GetValue("--ticket"));
        std::string certificateFilePath = std::string(option.GetValue("--certificate"));

        // チケットの読み込み
        fs::FileHandle ticketFile;
        NN_RESULT_DO(fs::OpenFile(&ticketFile, ticketFilePath.c_str(), fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(ticketFile); };

        int64_t ticketFileSize;
        NN_RESULT_DO(fs::GetFileSize(&ticketFileSize, ticketFile));

        std::unique_ptr<char[]> ticket(new char[static_cast<unsigned int>(ticketFileSize)]);
        NN_RESULT_DO(fs::ReadFile(ticketFile, 0, ticket.get(), static_cast<size_t>(ticketFileSize)));

        // 証明書の読み込み
        fs::FileHandle certificateFile;
        NN_RESULT_DO(fs::OpenFile(&certificateFile, certificateFilePath.c_str(), fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(certificateFile); };

        int64_t certificateFileSize;
        NN_RESULT_DO(fs::GetFileSize(&certificateFileSize, certificateFile));

        std::unique_ptr<char[]> certificate(new char[static_cast<unsigned int>(certificateFileSize)]);
        NN_RESULT_DO(fs::ReadFile(certificateFile, 0, certificate.get(), static_cast<size_t>(certificateFileSize)));

        NN_RESULT_DO(es::ImportTicket(ticket.get(), static_cast<size_t>(ticketFileSize), certificate.get(), static_cast<size_t>(certificateFileSize)));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result PrintCommand(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        if (!option.HasTarget())
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::es::RightsIdIncludingKeyId rightsId = devmenuUtil::GetRightsIdTarget(option);

        size_t outSize;
        NN_RESULT_DO(es::GetCommonTicketSize(&outSize, rightsId));

        es::TicketInfo ticketInfo;
        nn::es::GetTicketInfo(&ticketInfo, 1, &rightsId, 1);

        std::unique_ptr<char[]> buffer(new char[outSize]);

        NN_RESULT_DO(es::GetCommonTicketData(&outSize, buffer.get(), outSize, rightsId));

        DEVMENUCOMMAND_LOG("--------------------------------------------------------------------\n");
        PrintTicketInfo(ticketInfo);
        DEVMENUCOMMAND_LOG("--------------------------------------------------------------------\n");
        PrintBinary(buffer.get(), outSize);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    void PrintCommonTicketList() NN_NOEXCEPT
    {
        int commonTicketCount = nn::es::CountCommonTicket();
        std::unique_ptr<es::RightsIdIncludingKeyId[]> list(new es::RightsIdIncludingKeyId[commonTicketCount]);

        int commonRightsIdCount = nn::es::ListCommonTicketRightsIds(list.get(), commonTicketCount);

        std::unique_ptr<es::TicketInfo[]> ticketInfoList(new es::TicketInfo[commonTicketCount]);
        int ticketInfoCount = nn::es::GetTicketInfo(ticketInfoList.get(), commonTicketCount, list.get(), commonRightsIdCount);

        if (ticketInfoCount > 0)
        {
            DEVMENUCOMMAND_LOG("Common ticket list: number of ticket is %d.\n", commonTicketCount);
            DEVMENUCOMMAND_LOG("--------------------------------------------------------------------\n");
            for (int i = 0; i < ticketInfoCount; i++)
            {
                if (ticketInfoList[i].deviceId == 0)
                {
                    PrintTicketInfo(ticketInfoList[i]);
                }
            }
        }
    };

    void PrintPersonalizedTicketList() NN_NOEXCEPT
    {
        int personalizedTicketCount = nn::es::CountPersonalizedTicket();
        std::unique_ptr<es::RightsIdIncludingKeyId[]> list(new es::RightsIdIncludingKeyId[personalizedTicketCount]);

        int personalizedRightsIdCount = nn::es::ListPersonalizedTicketRightsIds(list.get(), personalizedTicketCount);

        // RightsId が personalized チケットと同じ common チケットを持っていた場合、 GetTicketInfo で Common チケットの情報も取得してしまうので、
        // common チケットの枚数 + personalized チケットの枚数分の配列を確保して GetTicketInfo に渡す
        int ticketCount = nn::es::CountCommonTicket() + nn::es::CountPersonalizedTicket();

        std::unique_ptr<es::TicketInfo[]> ticketInfoList(new es::TicketInfo[ticketCount]);
        int ticketInfoCount = nn::es::GetTicketInfo(ticketInfoList.get(), personalizedTicketCount, list.get(), personalizedRightsIdCount);

        if (ticketInfoCount > 0)
        {
            DEVMENUCOMMAND_LOG("Personalized ticket list: number of ticket is %d.\n", personalizedTicketCount);
            DEVMENUCOMMAND_LOG("--------------------------------------------------------------------\n");
            for (int i = 0; i < ticketInfoCount; i++)
            {
                if (ticketInfoList[i].deviceId != 0)
                {
                    PrintTicketInfo(ticketInfoList[i]);
                }
            }
        }
    };

    void PrintPrepurchaseRecordList() NN_NOEXCEPT
    {
        int prepurchaseRecordCount = nn::es::CountPrepurchaseRecord();
        std::unique_ptr<es::RightsIdIncludingKeyId[]> list(new es::RightsIdIncludingKeyId[prepurchaseRecordCount]);

        int prepurchaseRecordRightsIdCount = nn::es::ListPrepurchaseRecordRightsIds(list.get(), prepurchaseRecordCount);

        if (prepurchaseRecordCount > 0)
        {
            DEVMENUCOMMAND_LOG("Prepurchase record list: number of prepurchase record is %d.\n", prepurchaseRecordCount);
            DEVMENUCOMMAND_LOG("--------------------------------------------------------------------\n");
        }

        for (int i = 0; i < prepurchaseRecordRightsIdCount; i++)
        {
            // 1つの RightsId に対して最大で持ちえるチケットと予約購入情報の数(最大でリンクされる VA の数 + DA + コモンチケット)
            const int MaxSameRightsIdTicketCount = account::UserCountMax + 1 + 1;

            nn::es::PrepurchaseRecord recordList[MaxSameRightsIdTicketCount];
            int recordCount = nn::es::ListPrepurchaseRecordInfo(recordList, MaxSameRightsIdTicketCount, list[i]);

            for (int j = 0; j < recordCount; j++)
            {
                PrintPrepurchaseRecord(recordList[j]);
            }
        }
    }

    Result ListAllCommand(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        if (option.HasKey("--simple"))
        {
            auto printFunc = [](int(*countFunc)(), int(*listFunc)(es::RightsIdIncludingKeyId*, int), const char* label) {
                int count = countFunc();
                std::unique_ptr<es::RightsIdIncludingKeyId[]> list(new es::RightsIdIncludingKeyId[count]);
                listFunc(list.get(), count);

                for (int i = 0; i < count; i++)
                {
                    DEVMENUCOMMAND_LOG("0x%s%s\n", ToString(list[i]), label);
                }
            };

            if (!option.HasKey("--ticket-type"))
            {
                DEVMENUCOMMAND_LOG("RightsId                              Type\n");
                printFunc(nn::es::CountCommonTicket, nn::es::ListCommonTicketRightsIds,             "          Common");
                printFunc(nn::es::CountPersonalizedTicket, nn::es::ListPersonalizedTicketRightsIds, "    Personalized");
                printFunc(nn::es::CountPrepurchaseRecord, nn::es::ListPrepurchaseRecordRightsIds,   "     Prepurchase");
            }
            else
            {
                if (strncmp(option.GetValue("--ticket-type"), "common", 6) == 0)
                {
                    DEVMENUCOMMAND_LOG("RightsId                              Type\n");
                    printFunc(nn::es::CountCommonTicket, nn::es::ListCommonTicketRightsIds, "          Common");
                }
                else if (strncmp(option.GetValue("--ticket-type"), "personalized", 12) == 0)
                {
                    DEVMENUCOMMAND_LOG("RightsId                              Type\n");
                    printFunc(nn::es::CountPersonalizedTicket, nn::es::ListPersonalizedTicketRightsIds, "    Personalized");
                }
                else if (strncmp(option.GetValue("--ticket-type"), "prepurchase", 11) == 0)
                {
                    DEVMENUCOMMAND_LOG("RightsId                              Type\n");
                    printFunc(nn::es::CountPrepurchaseRecord, nn::es::ListPrepurchaseRecordRightsIds, "     Prepurchase");
                }
                else
                {
                    *outValue = false;
                    NN_RESULT_SUCCESS;
                }
            }
        }
        else
        {
            if (!option.HasKey("--ticket-type"))
            {
                PrintCommonTicketList();
                PrintPersonalizedTicketList();
                PrintPrepurchaseRecordList();
            }
            else
            {
                if (strncmp(option.GetValue("--ticket-type"), "common", 6) == 0)
                {
                    PrintCommonTicketList();
                }
                else if (strncmp(option.GetValue("--ticket-type"), "personalized", 12) == 0)
                {
                    PrintPersonalizedTicketList();
                }
                else if (strncmp(option.GetValue("--ticket-type"), "prepurchase", 11) == 0)
                {
                    PrintPrepurchaseRecordList();
                }
                else
                {
                    *outValue = false;
                    NN_RESULT_SUCCESS;
                }
            }
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    // DEPRECATED DevMenuCommand_ElicenseCommand.cpp create に移行します。
    Result ImportELicenseCommandDeprecated(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        account::InitializeForAdministrator();
        NN_RESULT_DO(time::InitializeForMenu());

        auto userIndex = std::strtol(option.GetTarget(), nullptr, 10);
        util::optional<account::Uid> uid;
        NN_RESULT_DO(GetUidFromIndex(&uid, userIndex));

        if (!uid)
        {
            DEVMENUCOMMAND_LOG("Not found user index %d\n", userIndex);

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        account::NintendoAccountId naId;
        NN_RESULT_TRY(GetNintendoAccountId(&naId, *uid))
            NN_RESULT_CATCH(account::ResultNetworkServiceAccountUnavailable)
            {
                DEVMENUCOMMAND_LOG("User account isn't linked with nintendo account.\n");

                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        es::ELicenseImportContext context;
        NN_RESULT_DO(es::BeginImportELicenseArchive(&context, naId));

        size_t jsonSize = 0;
        char jsonBuffer[16 * 1024];
        {
            nn::es::debug::ELicenseArchiveInfoForDebug archiveInfo = nn::es::debug::MakeELicenseArchiveInfo(context.GetChallenge().value, naId);

            nn::es::debug::ELicenseInfoForDebug eLicense = {};
            int eLicenseCount = 0;

            bool hasRightsId = option.HasKey("--rights-id");
            if (hasRightsId)
            {
                es::RightsId rightsId = To64BitId(option.GetValue("--rights-id"));

                bool isCreateSucceeded = false;
                if (strcmp(option.GetValue("--rights-type"), "permanent") == 0)
                {
                    isCreateSucceeded = es::debug::MakeDeviceLinkedPermanentELicenseFromTicketDBInfo(&eLicense, rightsId, naId);
                }
                else if (strcmp(option.GetValue("--rights-type"), "temporary") == 0)
                {
                    isCreateSucceeded = es::debug::MakeAccountRestrictiveTemporaryELicenseFromTicketDBInfo(&eLicense, rightsId, naId);
                }
                else
                {
                    *outValue = false;
                    NN_RESULT_SUCCESS;
                }

                if (!isCreateSucceeded)
                {
                    DEVMENUCOMMAND_LOG("Making eLicense failed because ticket don't exist.\n");
                    *outValue = false;
                    NN_RESULT_SUCCESS;
                }

                eLicenseCount++;
            }

            jsonSize = es::debug::BuildELicenseArchiveJsonString(jsonBuffer, NN_ARRAY_SIZE(jsonBuffer), archiveInfo, &eLicense, eLicenseCount);
            NN_ABORT_UNLESS(jsonSize > 0);

        }
        DEVMENUCOMMAND_LOG("eLicense json:\n");
        DEVMENUCOMMAND_LOG("%s\n", jsonBuffer);

        NN_RESULT_DO(es::ImportELicenseArchive(context, jsonBuffer, jsonSize));

        es::ELicenseArchiveId archiveId;
        NN_RESULT_DO(es::EndImportELicenseArchiveForDebug(&archiveId, context));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    const SubCommand g_SubCommands[] =
    {
        { "count", CountCommand },
        { "delete", DeleteCommand },
        { "delete-all", DeleteAllCommand },
        { "export", ExportCommand },
        { "export-with-certificate", ExportWithCertificateCommand },
        { "import", ImportCommand },
        { "print", PrintCommand },
        { "list-all", ListAllCommand },
        { "import-elicense", ImportELicenseCommandDeprecated },
    };
}

Result TicketCommand(bool* outValue, const Option& option) NN_NOEXCEPT
{
    if (!option.HasSubCommand())
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    DEVMENUCOMMAND_LOG("'%s' is not a DevMenu ticket command. See '" DEVMENUCOMMAND_NAME " ticket --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
