﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>
#include <string>

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>

#include <nn/os.h>
#include <nn/nifm.h>
#include <nn/socket.h>
#include <nn/ssl/ssl_Context.h>
#include <curl/curl.h>

#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>

#include <nn/nsd/nsd_Types.h>
#include <nn/nsd/nsd_TypesPrivate.h>
#include <nn/nsd/nsd_ApiForMenu.h>
#include <nn/nsd/nsd_ApiForTest.h>
#include <nn/nsd/nsd_DeleteMode.h>
#include <nn/nsd/nsd_Result.h>
#include <nn/nsd/nsd_ResultPrivate.h>

#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_MakeArgv.h"
#include "DevMenuCommand_ServiceDiscoveryCommand.h"

using namespace nn;

namespace
{
    // ヘルプ用メッセージ
    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " servicediscovery info\n"
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " servicediscovery info-detail\n"
#endif
        "       " DEVMENUCOMMAND_NAME " servicediscovery import <passcode>\n"
        "       " DEVMENUCOMMAND_NAME " servicediscovery import-all <passcode>\n"
        "       " DEVMENUCOMMAND_NAME " servicediscovery delete\n"
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " servicediscovery delete-all\n"
        "       " DEVMENUCOMMAND_NAME " servicediscovery enable-change-environment-identifier\n"
        "       " DEVMENUCOMMAND_NAME " servicediscovery disable-change-environment-identifier\n"
#endif
        ;

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    uint8_t g_WorkBuffer[1024 * 256] = {}; //nn::nsd::WorkBufferSizeForImportSettings でいいかも

    const size_t ResponseBufferSizeMin = 128 * 1024; //!< ネットワークサービスディスカバリのサーバレスポンス最低サイズ

    // ダウンロード用バッファ
    uint8_t g_DownloadBuffer[ResponseBufferSizeMin];
    size_t g_DownloadSize;

    struct CurlWriteFunctionUserData
    {
        uint8_t* pWriteBuffer;
        size_t writePosition;
        size_t buffreSize;
    };

    size_t CurlWriteFunction(char *pData, size_t blobsize, size_t blobcount, void *userdata) NN_NOEXCEPT;
    CURLcode CurlSslContextFunction(CURL* pCurl, void* pSslContext, void* pUserData) NN_NOEXCEPT;
} // namespace

CURLcode DownloadSettingData(
    size_t* pOutActualSize,
    void* pDownloadBuffer, size_t downloadBufferSize,
    CURL* curl,
    const nn::nsd::PassCode& passCode
    ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutActualSize);
    NN_ABORT_UNLESS_NOT_NULL(pDownloadBuffer);
    NN_ABORT_UNLESS_GREATER_EQUAL(downloadBufferSize, ResponseBufferSizeMin);
    NN_ABORT_UNLESS_NOT_NULL(curl);

    //
    // [注意]
    // このリクエスト URL は今後変更される可能性があります
    //
    const char *url = "https://api.sect.srv.nintendo.net/v1/settings";

    CurlWriteFunctionUserData userData = { static_cast<uint8_t*>(pDownloadBuffer), 0, downloadBufferSize };

    curl_easy_setopt(curl, CURLOPT_URL, url);
    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION,  CurlWriteFunction);
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, &userData);
    curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);

    // 検証が有効化されたため VERIFYPEER を 1L, VERIFYHOST を 2L 指定しています。
    curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 1L);
    curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 2L);
    curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, CurlSslContextFunction);

    // POSTデータの設定
    char postBuffer[256];
    int len = nn::util::SNPrintf(postBuffer, sizeof(postBuffer), "passcode=%s&device_id=0", passCode.value);
    NN_ABORT_UNLESS(len < static_cast<int>(sizeof(postBuffer)));

    curl_easy_setopt(curl, CURLOPT_POST, 1);
    curl_easy_setopt(curl, CURLOPT_POSTFIELDS, postBuffer);
    curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, len);

    // Perform the request
    CURLcode curlCode = curl_easy_perform(curl);

    if(curlCode == CURLE_OK)
    {
        *pOutActualSize = userData.writePosition;
    }

    return curlCode;
}

namespace
{
    size_t CurlWriteFunction(char *pData, size_t blobsize, size_t blobcount, void *userdata) NN_NOEXCEPT
    {
        CurlWriteFunctionUserData* userData = static_cast<CurlWriteFunctionUserData*>(userdata);

        auto count = blobsize * blobcount;

        if(userData->writePosition + count <= userData->buffreSize)
        {
            memcpy(&userData->pWriteBuffer[userData->writePosition], pData, count);
            userData->writePosition += count;
        }
        else
        {
            DEVMENUCOMMAND_LOG("At CurlWriteFunction, download buffer was overflowed. bufferSize:%zu\n", userData->buffreSize);
            return 0; // 処理を中断します
        }

        return count;
    }

    CURLcode CurlSslContextFunction(CURL* pCurl, void* pSslContext, void* pUserData) NN_NOEXCEPT
    {
        NN_UNUSED(pCurl);
        NN_UNUSED(pUserData);

        auto& context = *reinterpret_cast<nn::ssl::Context*>(pSslContext);
        if (context.Create(nn::ssl::Context::SslVersion_Auto).IsFailure())
        {
            return CURLE_ABORTED_BY_CALLBACK;
        }
        return CURLE_OK;
    }
}

// ネットワークサービスディスカバリの設定データのダウンロードを実行します
bool ExecuteDownload( nn::nsd::PassCode passCode ) NN_NOEXCEPT
{
    // curl セッションインスタンスの生成
    CURL* curl = curl_easy_init();
    if(!curl)
    {
        DEVMENUCOMMAND_LOG("curl_easy_init failed.\n");
        return false;
    }

    // ダウンロード
    CURLcode result = DownloadSettingData(
        &g_DownloadSize,
        g_DownloadBuffer, sizeof(g_DownloadBuffer),
        curl,
        passCode);

    curl_easy_cleanup(curl);

    if( result != CURLE_OK )
    {
        DEVMENUCOMMAND_LOG("Download Failed. curlcode:%d\n", result);
        return false;
    }
    return true;
}

//----------------------------------------------------------------
// インポート
nn::Result ImportSettingData( const char* inputData, size_t inputDataSize, nn::nsd::ImportMode importMode )
{
    // import
    nn::Result result = nn::nsd::ImportSettings(
                                                inputData,
                                                inputDataSize,
                                                g_WorkBuffer,
                                                sizeof(g_WorkBuffer),
                                                importMode);

    if ( result.IsFailure() )
    {
        if ( nn::nsd::ResultInvalidSettingData::Includes( result ) ) // 不正なデータだった
        {
            DEVMENUCOMMAND_LOG("Import Failed. Importing data is invalid.\n");
        }
        else if ( nn::nsd::ResultInvalidImportMode::Includes( result ) ) // 不正なインポートモードだった
        {
            DEVMENUCOMMAND_LOG("Import Failed. Invalid import mode was specified.\n");
        }
        else // その他のエラー
        {
            DEVMENUCOMMAND_LOG("Import Failed. Error occurred. (%08x, %03d-%04d)\n",
                   result.GetInnerValueForDebug(),
                   result.GetModule(), result.GetDescription());
        }
    }

    return result;
}

//----------------------------------------------------------------
// インポート処理
//
Result ServiceDiscoveryCommandImportCore( bool* outValue, const Option& option, nn::nsd::ImportMode importMode )
{
    auto passCodeString = option.GetTarget();
    size_t passCodeStringSize = std::strlen( passCodeString );

    // パスコード指定がない
    if ( passCodeStringSize == 0 )
    {
        DEVMENUCOMMAND_LOG("No passcode.\n");
        DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " servicediscovery [import|import-all] <passcode>\n");
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    // パスコード指定
    DEVMENUCOMMAND_LOG("passcode = [%s]\n", passCodeString);
    if ( passCodeStringSize >= nn::nsd::PassCode::Size ) // サイズオーバー
    {
        DEVMENUCOMMAND_LOG("Passcode is too long.\n");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    nn::nsd::PassCode passCode;
    nn::util::Strlcpy( passCode.value, passCodeString, passCode.Size );

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
    DEVMENUCOMMAND_LOG("Waiting for network availability...\n");
    nn::nifm::SubmitNetworkRequestAndWait();
    if(!nn::nifm::IsNetworkAvailable())
    {
        DEVMENUCOMMAND_LOG("Network is not available.\n");
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    // サーバから設定をダウンロード
    bool isSuccess = ExecuteDownload( passCode );
    if ( !isSuccess  )
    {
        DEVMENUCOMMAND_LOG("Error occurred in download.\n");
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    DEVMENUCOMMAND_LOG("Download done.\n");
    DEVMENUCOMMAND_LOG("Data: [%s]\n", g_DownloadBuffer);
    DEVMENUCOMMAND_LOG("DataSize: %zu\n", g_DownloadSize);

    // ダウンロードした設定データをインポート
    auto result = ImportSettingData( reinterpret_cast<const char*>(g_DownloadBuffer), g_DownloadSize, importMode );
    if ( result.IsFailure() )
    {
        DEVMENUCOMMAND_LOG("Error occurred while import. (%08x, %03d-%04d)\n",
               result.GetInnerValueForDebug(),
               result.GetModule(), result.GetDescription());

        if ( nn::nsd::ResultEnvironmentIdentifierNotUpdated::Includes(result) )
        {
            DEVMENUCOMMAND_LOG("\033[32m*** Please reinitialize the target to resolve this error.\033[m\n");
        }

        *outValue = false;
        return result;
    }
    DEVMENUCOMMAND_LOG("Import done.\n");
    DEVMENUCOMMAND_LOG("\033[32m*** Please reboot the target to allow new settings to take effect.\033[m\n");
    *outValue = true;
    NN_RESULT_SUCCESS;
}
//----------------------------------------------------------------
// サブコマンド : import
//
Result ServiceDiscoveryCommandImport( bool* outValue, const Option& option )
{
    DEVMENUCOMMAND_LOG("[Import application setting.]\n");
    return ServiceDiscoveryCommandImportCore( outValue, option, nn::nsd::ImportMode_ApplicationSettings );
}

//----------------------------------------------------------------
// サブコマンド : import
//
Result ServiceDiscoveryCommandImportAll( bool* outValue, const Option& option )
{
    DEVMENUCOMMAND_LOG("[Import all setting.]\n");
    return ServiceDiscoveryCommandImportCore( outValue, option, nn::nsd::ImportMode_All );
}

//----------------------------------------------------------------
// サブコマンド : delete
//
Result ServiceDiscoveryCommandDelete( bool* outValue, const Option& option )
{
    if ( std::strlen(option.GetTarget()) > 0 )
    {
        DEVMENUCOMMAND_LOG("delete command must be used with no argument.\n");
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    nn::nsd::DeleteSettings( nn::nsd::DeleteMode::DeleteMode_ApplicationSettings ); // void 関数
    DEVMENUCOMMAND_LOG("[Deleted application settings.]\n");
    DEVMENUCOMMAND_LOG("\033[32m*** Please reboot the target to allow changes to take effect.\033[m\n");

    *outValue = true;
    NN_RESULT_SUCCESS;
}
//----------------------------------------------------------------
// サブコマンド : delete-all
//
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
Result ServiceDiscoveryCommandDeleteAll( bool* outValue, const Option& option )
{
    if ( std::strlen(option.GetTarget()) > 0 )
    {
        DEVMENUCOMMAND_LOG("delete-all command must be used with no argument.\n");
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    nn::nsd::DeleteSettings( nn::nsd::DeleteMode::DeleteMode_All ); // void 関数
    DEVMENUCOMMAND_LOG("[Deleted all settings.]\n");
    DEVMENUCOMMAND_LOG("\033[32m*** Please reboot the target to allow changes to take effect.\033[m\n");

    *outValue = true;
    NN_RESULT_SUCCESS;
}
#endif
//----------------------------------------------------------------
// サブコマンド : info
//
Result ServiceDiscoveryCommandInfo( bool* outValue, const Option& option )
{
    if ( std::strlen(option.GetTarget()) > 0 )
    {
        DEVMENUCOMMAND_LOG("info command must be used with no argument.\n");
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    DEVMENUCOMMAND_LOG("[Information of the settings imported.]\n");
    NN_UNUSED(option);

    nn::nsd::SettingName settingName;
    nn::Result result = nn::nsd::GetSettingName( &settingName );
    if ( result.IsSuccess() )
    {
        DEVMENUCOMMAND_LOG( "Setting name : [%s]\n", settingName.value );
    }
    else if ( nn::nsd::ResultNotFound::Includes(result) ) // 設定が存在しない
    {
        DEVMENUCOMMAND_LOG( "Setting name : n/a (no setting data)\n" );
    }
    else // その他のエラー
    {
        DEVMENUCOMMAND_LOG( "Failed. Error occurred.\n" );
        *outValue = false;
        return result;
    }

    nn::nsd::EnvironmentIdentifier identifier;
    nn::nsd::GetEnvironmentIdentifier( &identifier );  // void 関数
    DEVMENUCOMMAND_LOG( "Environment identifier : [%s]\n", identifier.value );

    DEVMENUCOMMAND_LOG( "Environment identifier change : [%s]\n",
        nn::nsd::IsChangeEnvironmentIdentifierDisabled() ? "disabled" : "enabled");

    *outValue = true;
    NN_RESULT_SUCCESS;
}

namespace{
    nn::nsd::SaveData s_SaveData;
}
void DisplayString( const void* buffer, size_t size, bool newLine )
{
    const char* p = static_cast<const char*>(buffer);
    bool isContainNondisplayable = false;
    size_t length = 0;

    while( *p && length < size )
    {
        if ( isgraph( *p ) || *p == ' ' )
        {
            DEVMENUCOMMAND_LOG("%c", *p);
        }
        else
        {
            DEVMENUCOMMAND_LOG("?");
            isContainNondisplayable = true;
        }

        p ++;
        length ++;
    }

    DEVMENUCOMMAND_LOG( " (%zu byte%s)%s",
            length,
            isContainNondisplayable? ", contains non-displayable char": "",
            newLine? "\n": "" );
}
void DisplayVersion( uint32_t version )
{
    DEVMENUCOMMAND_LOG( "Version: 0x%x\n", version );
}
void DisplayDeviceId( nn::nsd::DeviceId& deviceId )
{
    DEVMENUCOMMAND_LOG( "DeviceId: " );
    DisplayString( &deviceId.value, deviceId.Size, true );
}
void DisplaySettingName( nn::nsd::SettingName& settingName )
{
    DEVMENUCOMMAND_LOG( "SettingName: " );
    DisplayString( &settingName.value, settingName.Size, true );
}
void DisplayEnvironmentIdentifier( nn::nsd::EnvironmentIdentifier& identifier )
{
    DEVMENUCOMMAND_LOG( "EnvironmentIdentifier: " );
    DisplayString( &identifier.value, identifier.Size, true );
}
void DisplayExpireTime( int64_t expireTime )
{
    DEVMENUCOMMAND_LOG( "ExpireTime: 0x%llx\n", expireTime );
}
void DisplayBackboneSettings( nn::nsd::BackboneSettings& settings )
{
    // NAS設定群
    DEVMENUCOMMAND_LOG( "NasServiseSettings: NasRequestServiceFqdn: " );
    DisplayString( &settings.nasServiceSettings.nasRequestServiceFqdn.value, settings.nasServiceSettings.nasRequestServiceFqdn.Size, true );
    DEVMENUCOMMAND_LOG( "NasServiseSettings: NasApiServiceFqdn: " );
    DisplayString( &settings.nasServiceSettings.nasApiServiceFqdn.value, settings.nasServiceSettings.nasApiServiceFqdn.Size, true );

    for( int n=0; n<settings.nasServiceSettings.NasServiceEntryCountMax; n++ )
    {
        DEVMENUCOMMAND_LOG( "NasServiseSettings: Entry_%d: Name: ", n );
        DisplayString( &settings.nasServiceSettings.entries[n].nasServiceName.value, settings.nasServiceSettings.entries[n].nasServiceName.Size, false );
        DEVMENUCOMMAND_LOG( ",  ClientId: %llx, RedirectUri: ", settings.nasServiceSettings.entries[n].nasServiceSetting.clientId );
        DisplayString( &settings.nasServiceSettings.entries[n].nasServiceSetting.redirectUri.value, settings.nasServiceSettings.entries[n].nasServiceSetting.redirectUri.Size, true );
    }

    // FQDN変換ルール
    for( int n=0; n<settings.FqdnEntryCountMax; n++ )
    {
        DEVMENUCOMMAND_LOG( "BackboneSetting_%03d: Src: ", n );
        DisplayString( &settings.fqdnEntries[n].src.value, settings.fqdnEntries[n].src.Size, false );
        DEVMENUCOMMAND_LOG( ", Dest: ");
        DisplayString( &settings.fqdnEntries[n].dst.value, settings.fqdnEntries[n].dst.Size, true );
    }
}
void DisplayApplicationSettings( nn::nsd::ApplicationSettings& settings )
{
    // FQDN変換ルール
    for( int n=0; n<settings.FqdnEntryCountMax; n++ )
    {
        DEVMENUCOMMAND_LOG( "ApplicationSetting_%03d: Src: ", n );
        DisplayString( &settings.fqdnEntries[n].src.value, settings.fqdnEntries[n].src.Size, false );
        DEVMENUCOMMAND_LOG( ", Dest: ");
        DisplayString( &settings.fqdnEntries[n].dst.value, settings.fqdnEntries[n].dst.Size, true );
    }
}
void DisplayIsAvailable( bool isAvailable )
{
    DEVMENUCOMMAND_LOG( "IsAvailable: %s\n", isAvailable? "true": "false" );
}

void DisplaySaveData( nn::nsd::SaveData& saveData )
{
    DisplayVersion( saveData.version );
    DisplayDeviceId( saveData.deviceId );
    DisplaySettingName( saveData.settingName );
    DisplayEnvironmentIdentifier( saveData.environmentIdentifier );
    DisplayExpireTime( saveData.expireUnixTime );
    DisplayIsAvailable( saveData.isAvailable );

    DisplayBackboneSettings( saveData.backboneSettings );
    DisplayApplicationSettings( saveData.applicationSettings );
}

Result ServiceDiscoveryCommandInfoDetail( bool* outValue, const Option& option )
{
    if ( std::strlen(option.GetTarget()) > 0 )
    {
        DEVMENUCOMMAND_LOG("info-detail command must be used with no argument.\n");
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    DEVMENUCOMMAND_LOG("[Detailed information of the settings imported.]\n");
    nn::nsd::GetCurrentSaveData( &s_SaveData );

    // saveData の情報表示
    DisplaySaveData( s_SaveData );

    *outValue = true;
    NN_RESULT_SUCCESS;
}

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
Result ServiceDiscoveryCommandEnableChangeEnvironmentIdentifier( bool* outValue, const Option& Option )
{
    nn::nsd::EnableChangeEnvironmentIdentifier();

    *outValue = true;
    NN_RESULT_SUCCESS;
}

Result ServiceDiscoveryCommandDisableChangeEnvironmentIdentifier( bool* outValue, const Option& Option )
{
    nn::nsd::DisableChangeEnvironmentIdentifier();

    *outValue = true;
    NN_RESULT_SUCCESS;
}
#endif

//----------------------------------------------------------------
namespace{
    // サブコマンド情報
    const SubCommand g_SubCommands[] =
    {
        { "import", ServiceDiscoveryCommandImport },
        { "import-all", ServiceDiscoveryCommandImportAll },
        { "delete", ServiceDiscoveryCommandDelete },
        { "info", ServiceDiscoveryCommandInfo },
        { "info-detail", ServiceDiscoveryCommandInfoDetail },
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        { "delete-all", ServiceDiscoveryCommandDeleteAll },
        { "enable-change-environment-identifier", ServiceDiscoveryCommandEnableChangeEnvironmentIdentifier },
        { "disable-change-environment-identifier", ServiceDiscoveryCommandDisableChangeEnvironmentIdentifier },
#endif
    };
}

Result ServiceDiscoveryCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    auto secondArg = option.GetSubCommand();
    if (std::string(secondArg) == "--help")
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    // サブコマンド処理
    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    // 不正なサブコマンドを指定
    DEVMENUCOMMAND_LOG("'%s' is not a DevMenu servicediscovery command. See '" DEVMENUCOMMAND_NAME " servicediscovery --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
